/***************************************************************************
                          engine_tools.c  -  description
                             -------------------
    begin                : Fri Mar 9 2001
    copyright            : (C) 2001 by Michael Speck
    email                : kulkanie@gmx.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <SDL.h>

#ifdef WITH_SOUND
#include <SDL_mixer.h>
#include "audio.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dirent.h>

#include "tools.h"
#include "sdl.h"
#include "dynlist.h"
#include "gui.h"
#include "theme.h"
#include "config.h"
#include "nation.h"
#include "unit.h"
#include "ai_action.h"
#include "player.h"
#include "date.h"
#include "map.h"
#include "scenario.h"
#include "engine_tools.h"

extern Config config;
extern Map map;
extern Scen scen;
extern int cur_weather; /* declared in unit.c; is the current weather */

/* set auxiliary spotting mask */
void set_aux_spot_mask( Unit *unit );
/* set spot mask recursively */
void set_aux_spot_mask_rec( Unit *unit, int x, int y, int points, int step );
/* set move mask recursively */
void set_move_mask_rec( Engine *engine, Unit *unit, int x, int y, int points, int first );

/* same like set_spot_mask but remember not to change this mask when FOREIGN_FOG is set */
void set_fog( Engine *engine, int player_id )
{
    int i;
    int x, y;
    Unit *unit;
    Player *player = scen.players[player_id];

    clear_mask( F_FOG );

    /* get spot_mask for each unit and add to fog */
    /* use map::mask::aux as buffer */
    scen.units.cur_entry = &scen.units.head;
    for ( i = 0; i < scen.units.count; i++ ) {
        unit = (Unit*)(dl_next_entry( &scen.units )->data);
        if ( unit->player_id == player_id ||
             player->dipl[unit->player_id] == ALLIES )
        {
            /* it's your unit or at least it's allied... */
            set_aux_spot_mask( unit );
            for ( x = 0; x < map.width; x++ )
                for ( y = 0; y < map.height; y++ )
                    if ( mask_tile( x, y )->aux )
                        mask_tile( x, y )->fog = 0;
        }
    }
    /* check all flags; if flag belongs to you or any of your partners you see the surrounding, too */
    for ( x = 0; x < map.width; x++ )
        for ( y = 0; y < map.height; y++ )
            if ( map_tile( x, y )->player_id != -1 )
                if ( map_tile( x, y )->player_id == player_id ||
                     player->dipl[map_tile( x, y )->player_id] == ALLIES
                ) {
                    mask_tile( x, y )->fog = 0;
                    mask_tile( x, y + 1 )->fog = 0;
                    mask_tile( x, y - 1)->fog = 0;
                    mask_tile( x + 1, y )->fog = 0;
                    mask_tile( x - 1, y )->fog = 0;
                    if ( x & 1 ) {
                        mask_tile( x + 1, y + 1 )->fog = 0;
                        mask_tile( x - 1, y + 1 )->fog = 0;
                    }
                    else {
                        mask_tile( x + 1, y - 1 )->fog = 0;
                        mask_tile( x - 1, y - 1 )->fog = 0;
                    }
                }
}

/* spot mask indicating which tiles the current player sees */
void set_spot_mask( Engine *engine )
{
    int i;
    int x, y;
    Unit *unit;

    clear_mask( F_SPOT );
    /* if OWN_FOG is set spot_mask and fog mask are indentically */
    if ( engine->fog_status == OWN_FOG )
        clear_mask( F_FOG );

    /* get spot_mask for each unit and add to fog */
    /* use map::mask::aux as buffer */
    scen.units.cur_entry = &scen.units.head;
    for ( i = 0; i < scen.units.count; i++ ) {
        unit = (Unit*)(dl_next_entry( &scen.units )->data);
        if ( unit->player_id == engine->player_id ||
             engine->player->dipl[unit->player_id] == ALLIES )
        {
            /* it's your unit or at least it's allied... */
            set_aux_spot_mask( unit );
            for ( x = 0; x < map.width; x++ )
                for ( y = 0; y < map.height; y++ )
                    if ( mask_tile( x, y )->aux ) {
                        mask_tile( x, y )->spot = 1;
                        if ( engine->fog_status == OWN_FOG )
                            mask_tile( x, y )->fog = 0;
                    }
        }
    }
    /* check all flags; if flag belongs to you or any of your partners you see the surrounding, too */
    for ( x = 0; x < map.width; x++ )
        for ( y = 0; y < map.height; y++ )
            if ( map_tile( x, y )->player_id != -1 )
                if ( map_tile( x, y )->player_id == engine->player_id ||
                     engine->player->dipl[map_tile( x, y )->player_id] == ALLIES
                ) {
                    /* clear tiles */
                    mask_tile( x, y )->spot = 1;
                    mask_tile( x, y + 1 )->spot = 1;
                    mask_tile( x, y - 1)->spot = 1;
                    mask_tile( x + 1, y )->spot = 1;
                    mask_tile( x + 1, y )->spot = 1;
                    if ( x & 1 ) {
                        mask_tile( x + 1, y + 1 )->spot = 1;
                        mask_tile( x + 1, y + 1 )->spot = 1;
                    }
                    else {
                        mask_tile( x + 1, y - 1 )->spot = 1;
                        mask_tile( x + 1, y - 1 )->spot = 1;
                    }
                    if ( engine->fog_status == OWN_FOG ) {
                        mask_tile( x, y )->fog = 0;
                        mask_tile( x, y + 1 )->fog = 0;
                        mask_tile( x, y - 1)->fog = 0;
                        mask_tile( x + 1, y )->fog = 0;
                        mask_tile( x - 1, y )->fog = 0;
                        if ( x & 1 ) {
                            mask_tile( x + 1, y + 1 )->fog = 0;
                            mask_tile( x - 1, y + 1 )->fog = 0;
                        }
                        else {
                            mask_tile( x + 1, y - 1 )->fog = 0;
                            mask_tile( x - 1, y - 1 )->fog = 0;
                        }
                    }
                }
}

/* update spotting mask with sight of one unit; return true if an enemy was spotted */
int update_spot_mask( Engine *engine, Unit *unit )
{
    int x, y;
    int enemy_spotted = 0;

    if ( unit->player_id == engine->player_id ||
         engine->player->dipl[unit->player_id] == ALLIES )
    {
        /* it's your unit or at least it's allied... */
        set_aux_spot_mask( unit );
        for ( x = 0; x < map.width; x++ )
            for ( y = 0; y < map.height; y++ )
                if ( mask_tile( x, y )->aux ) {
                    /* if there is an enemy in this auxialiary mask that wasn't spotted before */
                    /* set enemy_spotted true */
                    if ( !mask_tile( x, y )->spot ) {
                        if ( map_tile( x, y )->unit && is_enemy( unit, map_tile( x, y )->unit ) )
                            enemy_spotted = 1;
                        if ( map_tile( x, y )->air_unit && is_enemy( unit, map_tile( x, y )->air_unit ) )
                            enemy_spotted = 1;
                    }
                    mask_tile( x, y )->spot = 1;
                    if ( engine->fog_status == OWN_FOG )
                        mask_tile( x, y )->fog = 0;
                }

    }
    return enemy_spotted;
}
/*
====================================================================
Backup spot mask to mask::backup. Used to restore sight when
undo unit move.
====================================================================
*/
void backup_spot_mask( Engine *engine )
{
    int x, y;
    clear_mask( F_BACKUP );
    for ( x = 0; x < map.width; x++ )
        for ( y = 0; y < map.height; y++ )
            mask_tile( x, y )->backup = mask_tile( x, y )->spot;
}
/*
====================================================================
Restore backuped spot mask.
====================================================================
*/
void restore_spot_mask( Engine *engine )
{
    int x, y;
    for ( x = 0; x < map.width; x++ )
        for ( y = 0; y < map.height; y++ ) {
            mask_tile( x, y )->spot = mask_tile( x, y )->backup;
            /* if fog is OWN_FOG then we must update it, too */
            if ( engine->fog_status == OWN_FOG )
                mask_tile( x, y )->fog = !mask_tile( x, y )->spot;
        }
    clear_mask( F_BACKUP );
}

/* does unit spot this tile and how much does it cost */
int unit_spot_tile( Unit *unit, int x, int y, int *cost )
{
    if ( map_tile( x, y )->prop->flags & NO_SPOTTING ) {
        if ( get_dist( unit->x, unit->y, x, y ) == 1 ) {
            *cost = unit->sel_prop->spot;
            if ( config.weather )
                if ( cur_weather == SNOW || cur_weather == RAIN )
                    *cost *= 2;
            return 1;
        }
        return 0;
    }
    *cost = map_tile( x, y )->prop->spot;
    if ( config.weather )
        if ( cur_weather == SNOW || cur_weather == RAIN )
            *cost *= 2;
    return 1;
}

/* set move mask recursively while points >0 */
void set_aux_spot_mask_rec( Unit *unit, int x, int y, int points, int step )
{
    int go_on = 1;
    int cost;

    if ( step == 0 ) {
        /* always see your own tile */
        mask_tile( x, y )->aux = points;
    }
    else {
        /* on map? */
        if ( x < 0 || y < 0 || x >= map.width || y >= map.height )
            return;
        /* dont visit spotted tiles again */
        if ( points <= mask_tile( x, y )->aux ) return;
        if ( mask_tile( x, y )->aux &&
             unit_spot_tile( unit, x, y, &cost ) &&
             points - cost <= mask_tile( x, y )->aux
        )
            return;
        if ( unit_spot_tile( unit, x, y, &cost ) ) {
            points -= cost;
            if ( points < 0 ) points = 0;
            mask_tile( x, y )->aux = points;
            if ( step == 1 && points == 0 ) mask_tile( x, y )->aux = 1; /* at least see surrounding */
            if ( points == 0 ) go_on = 0;
        }
        else
            go_on = 0;
    }

    if ( go_on ) {
        step++;
        /* recursive calls */
        set_aux_spot_mask_rec( unit, x, y - 1, points, step );
        set_aux_spot_mask_rec( unit, x, y + 1, points, step );
        set_aux_spot_mask_rec( unit, x + 1, y, points, step );
        set_aux_spot_mask_rec( unit, x - 1, y, points, step );
        /* secondary neighbors differ depending on x is odd or even */
        if ( x & 1 ) {
            set_aux_spot_mask_rec( unit, x + 1, y + 1, points, step );
            set_aux_spot_mask_rec( unit, x - 1, y + 1, points, step );
        }
        else {
            set_aux_spot_mask_rec( unit, x + 1, y - 1, points, step );
            set_aux_spot_mask_rec( unit, x - 1, y - 1, points, step );
        }
    }
}

/* set auxiliary spotting mask */
void set_aux_spot_mask( Unit *unit )
{
    clear_mask( F_AUX );
    set_aux_spot_mask_rec( unit, unit->x, unit->y, unit->sel_prop->spot + 1, 0 );
}

/*
====================================================================
Check the weather influence on ground units.
====================================================================
*/
void check_weather( Unit *unit, int *cost )
{
    if ( !config.weather ) return;
    if ( cur_weather == FAIR || cur_weather == CLOUDS ) return;
    if ( unit->sel_prop->flags & SWIMMING || unit->sel_prop->flags & FLYING ) return;
    /* snow or rain: cut non-tracked movement in half */
    if ( !(unit->sel_prop->flags & TRACKED) )
        *cost *= 2;
}

/* check if unit can enter this tile and return the move cost entering requires */
int unit_can_enter_tile( Unit *unit, int x, int y, int *cost )
{
    /* on map? */
    if ( x <= 0 || y <= 0 || x >= map.width - 1 || y >= map.height - 1 )
        return 0;
    /* you can move over allied units but then mask::blocked must be set because you must
    not stop at this tile */
    /* enemies are impassible */
    if ( ( x != unit->x  || y != unit->y ) && mask_tile( x, y )->spot ) {
        if ( map_tile( x, y )->air_unit && ( unit->sel_prop->flags & FLYING ) ) {
            if ( is_enemy( unit, map_tile( x, y )->air_unit ) )
                return 0;
            else
                mask_tile( x, y )->blocked = 1;
        }
        if ( map_tile( x, y )->unit && !( unit->sel_prop->flags & FLYING ) ) {
            if ( is_enemy( unit, map_tile( x, y )->unit ) )
                return 0;
            else
                mask_tile( x, y )->blocked = 1;
        }
    }
    /* bonus or malus? check the flags! */
    if ( !(unit->sel_prop->flags & FLYING ) ) {
        if ( map_tile( x, y )->prop->flags & MOUNTAIN )
            if ( unit->sel_prop->class != INFANTRY )
                return 0;
        if ( map_tile( x, y )->prop->flags & DIFFICULT ) {
            /* if it's a river and a bridge engineer is standing there it just costs one point to pass */
            if ( map_tile( x, y )->prop->flags & RIVER )
                if ( mask_tile( x, y )->blocked )
                    if ( map_tile( x, y )->unit->sel_prop->flags & BRIDGE_ENG )
                        if ( map_tile( x, y )->unit->no_action_yet ) {
                            *cost = 1;
                            check_weather( unit, cost );
                            return 1;
                        }
            if ( get_dist( unit->x, unit->y, x, y ) == 1 ) {
                *cost = unit->cur_prop.mov; /* doesn't need to check weather as all points are lost */
                return 1;
            }
            return 0;
        }
    }
    /* compute basic moving cost for this tile */
    if ( unit->sel_prop->flags & FLYING )
        *cost = 1;
    else
        if ( unit->sel_prop->flags & SWIMMING ) {
            if ( map_tile( x, y )->prop->flags & WATER || map_tile( x, y )->prop->flags & HARBOR )
                *cost = 1;
            else
                return 0;
        }
        else
            if ( map_tile( x, y )->prop->flags & WATER )
                return 0;
            else
                *cost = map_tile( x, y )->prop->mov;
    /* entering an influenced tile doubles moving cost; if tile is influenced by
    two or more enemies it costs ALL moving points to enter it */
    if ( unit->sel_prop->flags & FLYING ) {
        /* check influence */
        if ( mask_tile( x, y )->vis_air_infl == 1 )
            *cost *= 2;
        else
            if ( mask_tile( x, y )->vis_air_infl > 1 )
                *cost = unit->cur_prop.mov;
    }
    else {
        /* check influence */
        if ( mask_tile( x, y )->vis_infl == 1 )
            *cost *= 2;
        else
            if ( mask_tile( x, y )->vis_infl > 1 )
                *cost = unit->cur_prop.mov;
    }
    check_weather( unit, cost );
    return 1;
}

/* set move mask recursively while points >0 */
/* if pedes_points drops below 0 transporter is needed for movement */
void set_move_mask_rec( Engine *engine, Unit *unit, int x, int y, int points, int step )
{
    int go_on = 1;
    int cost = 1;

    /* very first step: clear tile unit is on */
    if ( step == 0 )
        mask_tile( x, y )->in_range = points; /* sel_unit is on this position */
    else
    if ( step == 1 ) {
        /* tile can be entered if not completely impassible without transporter */
        /* unit can only debark if this tile is near the unit */
        /* check if debarking is allowed */
        if ( unit->embark == SEA_EMBARK &&
             unit->sel_prop->flags & SWIMMING &&
             !( map_tile( x, y )->prop->flags & WATER ) &&
             !( map_tile( x, y )->prop->flags & HARBOR ) )
        {
            /* check for debarking */
            if ( unit_can_debark( engine, unit, x, y, SEA_EMBARK) )
                if ( points == unit->cur_prop.mov )
                    mask_tile( x, y )->sea_embark = 1;
            go_on = 0; /* debark yes; move there: no, so don't go recursive */
        } else
        /* check if embarking is allowed */
        if ( !( unit->sel_prop->flags & FLYING ) &&
             !( unit->sel_prop->flags & SWIMMING ) &&
             ( map_tile( x, y )->prop->flags & WATER ) )
        {
            /* check for embarking */
            if ( unit_can_embark( engine, unit, x, y, SEA_EMBARK) )
                if ( points == unit->cur_prop.mov )
                    mask_tile( x, y )->sea_embark = 1;
            go_on = 0; /* embark yes; move there: no, so don't go recursive */
        }
        else
        /* check if unit can move there */
        if ( unit_can_enter_tile( unit, x, y, &cost ) ) {
            points -= cost;
            if ( points < 0 ) points = 0;
            mask_tile( x, y )->in_range = points;
            /* set aux mask true if this tile is  reached; this is used to check if unit must mount or not */
            mask_tile( x, y )->aux = 1;
            /* still at least one point left? if not: dead end. */
            if ( points == 0 )
                go_on = 0;
        }
        else
            go_on = 0; /* dead end */
    }
    else {
        /* normal step */
        /* important break condition don't check tiles that where already visited again if there not
        more points remaining */
        if ( points <= mask_tile( x, y )->in_range ) return;
        /* only check for higher points if pedes_points less then 0 */
        if ( mask_tile( x, y )->in_range != -1 )
            if ( unit_can_enter_tile( unit, x, y, &cost ) && ( points - cost ) <= mask_tile( x, y )->in_range )
                return;
        /* check if unit can move there */
        if ( unit_can_enter_tile( unit, x, y, &cost ) ) {
            points -= cost;
            /* if the auxiliary mask is set this function was previously run to check the legged range
            of a unit with transporter; so if mask_tile::aux is set this tile is reached by leg, no mounting
            needed */
            if ( unit->ground_tran ) {
                if ( !mask_tile( x, y )->aux )
                    mask_tile( x, y )->mount = 1;
                else
                    mask_tile( x, y )->mount = 0;
            }
            if ( points < 0 ) points = 0; /* last tile in range */
            mask_tile( x, y )->in_range = points;
            /* set aux mask true if this tile is  reached; this is used to check if unit must mount or not */
            /* must only be set if unit's checking its range without transporter */
            if ( !unit->ground_tran && unit->tran_prop_set )
                mask_tile( x, y )->aux = 1;
            /* still at least one point left? if not: dead end. */
            if ( points == 0 )
                go_on = 0;
        }
        else
            go_on = 0;
    }

    if ( go_on ) {
        /* next step */
        step++;
        /* recursive calls */
        set_move_mask_rec( engine, unit, x, y - 1, points, step );
        set_move_mask_rec( engine, unit, x, y + 1, points, step );
        set_move_mask_rec( engine, unit, x + 1, y, points, step );
        set_move_mask_rec( engine, unit, x - 1, y, points, step );
        /* secondary neighbors differ depending on x is even or uneven */
        if ( x & 1 ) {
            set_move_mask_rec( engine, unit, x + 1, y + 1, points, step );
            set_move_mask_rec( engine, unit, x - 1, y + 1, points, step );
        }
        else {
            set_move_mask_rec( engine, unit, x + 1, y - 1, points, step );
            set_move_mask_rec( engine, unit, x - 1, y - 1, points, step );
        }
    }
}

/* set move mask from unit and map */
void set_move_mask( Engine *engine, Unit *unit )
{
    int points = 0;

    /* clear */
    clear_mask( F_IN_RANGE | F_MOUNT | F_SEA_EMBARK | F_BLOCKED | F_AUX );

    if ( unit->cur_prop.mov <= 0 ) return;

    /* maximum range is already set as unit->cur_prop.mov including ground transport and fuel */
    /* if this unit uses a ground transporter run set_move_mask with pedes_points first
    and then with cur_prop.mov. when set_move_mask_rec finds a mask_tile with in_range != -1 it will not
    set mount */
    if ( unit->ground_tran ) {
        /* for this check unit must not have ground tran */
        unit->ground_tran = 0;
        /* unit can't leave ground tran behind so if transporter can't move as far as the unit the unit can
        neither */
        points = unit->prop.mov;
        if ( unit->cur_prop.mov < points )
            points = unit->cur_prop.mov;
        set_move_mask_rec( engine, unit, unit->x, unit->y, points, 0 );
        unit->ground_tran = 1;
    }

    /* compute */
    set_move_mask_rec( engine, unit, unit->x, unit->y, unit->cur_prop.mov, 0 );
}

/* add a units influence to the infl mask */
void add_infl( Unit *unit )
{
    if ( unit->sel_prop->flags & FLYING ) {
        mask_tile( unit->x, unit->y )->air_infl++;
        mask_tile( unit->x + 1, unit->y )->air_infl++;
        mask_tile( unit->x - 1, unit->y )->air_infl++;
        mask_tile( unit->x, unit->y + 1 )->air_infl++;
        mask_tile( unit->x, unit->y - 1 )->air_infl++;
        if ( unit->x & 1 ) {
            mask_tile( unit->x + 1, unit->y + 1)->air_infl++;
            mask_tile( unit->x - 1, unit->y + 1)->air_infl++;
        }
        else {
            mask_tile( unit->x + 1, unit->y - 1)->air_infl++;
            mask_tile( unit->x - 1, unit->y - 1)->air_infl++;
        }
    }
    else {
        mask_tile( unit->x, unit->y )->infl++;
        mask_tile( unit->x + 1, unit->y )->infl++;
        mask_tile( unit->x - 1, unit->y )->infl++;
        mask_tile( unit->x, unit->y + 1 )->infl++;
        mask_tile( unit->x, unit->y - 1 )->infl++;
        if ( unit->x & 1 ) {
            mask_tile( unit->x + 1, unit->y + 1)->infl++;
            mask_tile( unit->x - 1, unit->y + 1)->infl++;
        }
        else {
            mask_tile( unit->x + 1, unit->y - 1)->infl++;
            mask_tile( unit->x - 1, unit->y - 1)->infl++;
        }
    }
}
/* remove a units influence (used when an enemy flees) */
void remove_infl( Unit *unit )
{
    if ( unit->sel_prop->flags & FLYING ) {
        mask_tile( unit->x, unit->y )->air_infl--;
        mask_tile( unit->x + 1, unit->y )->air_infl--;
        mask_tile( unit->x - 1, unit->y )->air_infl--;
        mask_tile( unit->x, unit->y + 1 )->air_infl--;
        mask_tile( unit->x, unit->y - 1 )->air_infl--;
        if ( unit->x & 1 ) {
            mask_tile( unit->x + 1, unit->y + 1)->air_infl--;
            mask_tile( unit->x - 1, unit->y + 1)->air_infl--;
        }
        else {
            mask_tile( unit->x + 1, unit->y - 1)->air_infl--;
            mask_tile( unit->x - 1, unit->y - 1)->air_infl--;
        }
    }
    else {
        mask_tile( unit->x, unit->y )->infl--;
        mask_tile( unit->x + 1, unit->y )->infl--;
        mask_tile( unit->x - 1, unit->y )->infl--;
        mask_tile( unit->x, unit->y + 1 )->infl--;
        mask_tile( unit->x, unit->y - 1 )->infl--;
        if ( unit->x & 1 ) {
            mask_tile( unit->x + 1, unit->y + 1)->infl--;
            mask_tile( unit->x - 1, unit->y + 1)->infl--;
        }
        else {
            mask_tile( unit->x + 1, unit->y - 1)->infl--;
            mask_tile( unit->x - 1, unit->y - 1)->infl--;
        }
    }
}
/* add a units influence to the infl mask */
void add_vis_infl( Unit *unit )
{
    if ( unit->sel_prop->flags & FLYING ) {
        mask_tile( unit->x, unit->y )->vis_air_infl++;
        mask_tile( unit->x + 1, unit->y )->vis_air_infl++;
        mask_tile( unit->x - 1, unit->y )->vis_air_infl++;
        mask_tile( unit->x, unit->y + 1 )->vis_air_infl++;
        mask_tile( unit->x, unit->y - 1 )->vis_air_infl++;
        if ( unit->x & 1 ) {
            mask_tile( unit->x + 1, unit->y + 1)->vis_air_infl++;
            mask_tile( unit->x - 1, unit->y + 1)->vis_air_infl++;
        }
        else {
            mask_tile( unit->x + 1, unit->y - 1)->vis_air_infl++;
            mask_tile( unit->x - 1, unit->y - 1)->vis_air_infl++;
        }
    }
    else {
        mask_tile( unit->x, unit->y )->vis_infl++;
        mask_tile( unit->x + 1, unit->y )->vis_infl++;
        mask_tile( unit->x - 1, unit->y )->vis_infl++;
        mask_tile( unit->x, unit->y + 1 )->vis_infl++;
        mask_tile( unit->x, unit->y - 1 )->vis_infl++;
        if ( unit->x & 1 ) {
            mask_tile( unit->x + 1, unit->y + 1)->vis_infl++;
            mask_tile( unit->x - 1, unit->y + 1)->vis_infl++;
        }
        else {
            mask_tile( unit->x + 1, unit->y - 1)->vis_infl++;
            mask_tile( unit->x - 1, unit->y - 1)->vis_infl++;
        }
    }
}
/* remove a units influence (used when an enemy flees) */
void remove_vis_infl( Unit *unit )
{
    if ( unit->sel_prop->flags & FLYING ) {
        mask_tile( unit->x, unit->y )->vis_air_infl--;
        mask_tile( unit->x + 1, unit->y )->vis_air_infl--;
        mask_tile( unit->x - 1, unit->y )->vis_air_infl--;
        mask_tile( unit->x, unit->y + 1 )->vis_air_infl--;
        mask_tile( unit->x, unit->y - 1 )->vis_air_infl--;
        if ( unit->x & 1 ) {
            mask_tile( unit->x + 1, unit->y + 1)->vis_air_infl--;
            mask_tile( unit->x - 1, unit->y + 1)->vis_air_infl--;
        }
        else {
            mask_tile( unit->x + 1, unit->y - 1)->vis_air_infl--;
            mask_tile( unit->x - 1, unit->y - 1)->vis_air_infl--;
        }
    }
    else {
        mask_tile( unit->x, unit->y )->vis_infl--;
        mask_tile( unit->x + 1, unit->y )->vis_infl--;
        mask_tile( unit->x - 1, unit->y )->vis_infl--;
        mask_tile( unit->x, unit->y + 1 )->vis_infl--;
        mask_tile( unit->x, unit->y - 1 )->vis_infl--;
        if ( unit->x & 1 ) {
            mask_tile( unit->x + 1, unit->y + 1)->vis_infl--;
            mask_tile( unit->x - 1, unit->y + 1)->vis_infl--;
        }
        else {
            mask_tile( unit->x + 1, unit->y - 1)->vis_infl--;
            mask_tile( unit->x - 1, unit->y - 1)->vis_infl--;
        }
    }
}
/* set influence mask; meaning's explained in map.h */
/* includes visible influence mask */
void set_infl_mask( Engine *engine )
{
    int i;
    Unit *unit = 0;

    clear_mask( F_INFL | F_AIR_INFL );
    /* add all hostile units influence */
    for ( i = 0; i < scen.units.count; i++ ) {
        unit = (Unit*)dl_get( &scen.units, i );
        if ( engine->player->dipl[unit->player_id] == ENEMIES )
            add_infl( unit );
    }
    /* visible influence must also be updated */
    set_vis_infl_mask( engine );
}
/* set influence mask; meaning's explained in map.h */
/* includes visible influence mask */
void set_vis_infl_mask( Engine *engine )
{
    int i;
    Unit *unit = 0;

    clear_mask( F_VIS_INFL | F_VIS_AIR_INFL );
    /* add all hostile units influence */
    for ( i = 0; i < scen.units.count; i++ ) {
        unit = (Unit*)dl_get( &scen.units, i );
        if ( engine->player->dipl[unit->player_id] == ENEMIES )
            if ( mask_tile( unit->x, unit->y )->spot )
                add_vis_infl( unit );
    }
}

/* check if unit is an attackable target for attacking unit */
int is_target( Unit *att, Unit *def )
{
    if ( att == 0 ) return 0;
    if ( def == 0 ) return 0;

    /* flying units can'T attack ground units except they are on the same tile */
    if ( att->sel_prop->flags & FLYING )
        if ( !( def->sel_prop->flags & FLYING ) )
            if ( get_dist( att->x, att->y, def->x, def->y ) > 0 )
                return 0;
    /* check if enemies and attack is allowed */
    if ( is_enemy( att, def ) )
        if ( can_attack_unit( att, def, ATTACK ) )
            return 1;
    return 0;
}

/* check if an valid interceptor flies at this position nd return this unit */
Unit* is_interceptor( Damage_Pred *pred, int x, int y ) {
    Unit *unit = 0;

    unit = map_tile( x, y )->air_unit;
    if ( !unit ) return 0;
    if ( !is_enemy( pred->att, unit ) ) return 0;
    if ( !( unit->sel_prop->flags & INTERCEPTOR ) ) return 0;
    if ( unit->cur_prop.ammo <= 0 ) return 0;
    return unit;
}

/* check all units in the surroudning map region;
 if they are hostile air defences that are in range add them to the support list */
/* attacked unit itself can't be support */
void add_air_defs( Damage_Pred *pred )
{
    Unit *unit = 0;
    int x, y;

    if ( pred->support_count == SUPPORT_LIMIT ) return;

    for ( x = pred->def->x - 5; x <= pred->def->x + 5; x++ ) {
        for ( y = pred->def->y - 5; y <= pred->def->y + 5; y++ ) {
            if ( x >= 0 && y >= 0 && x < map.width && y < map.height )
                if ( ( unit = map_tile( x, y )->unit ) != 0 )
                    if ( unit->sel_prop->flags & AIR_DEFENCE )
                        if ( is_enemy( unit, pred->att ) )
                            if ( unit->cur_prop.ammo > 0 )
                                if ( unit != pred->def )
                                    if  ( get_dist( unit->x, unit->y, pred->att->x, pred->att->y ) <= unit->sel_prop->range )
                                        pred->support[pred->support_count++] = unit;
            if ( pred->support_count == SUPPORT_LIMIT ) break;
        }
        if ( pred->support_count == SUPPORT_LIMIT ) break;
    }
}

/* check all units in the surroudning map region;
 if they are hostile artilleries that are in range add them to the support list */
/* attacked unit itself can't be support */
/* if the attacker is anartillery no supporting fire! */
void add_arts( Damage_Pred *pred )
{
    Unit *unit = 0;
    int x, y;

    if ( pred->support_count == SUPPORT_LIMIT ) return;
    if ( pred->att->sel_prop->flags & ARTILLERY ) return;
    if ( pred->att->sel_prop->flags & AIR_DEFENCE ) return;

    for ( x = pred->def->x - 5; x <= pred->def->x + 5; x++ ) {
        for ( y = pred->def->y - 5; y <= pred->def->y + 5; y++ ) {
            if ( x >= 0 && y >= 0 && x < map.width && y < map.height )
                if ( ( unit = map_tile( x, y )->unit ) != 0 )
                    if ( unit->sel_prop->flags & ARTILLERY )
                        if ( is_enemy( unit, pred->att ) )
                            if ( unit->cur_prop.ammo > 0 )
                                if ( unit != pred->def )
                                    if  ( get_dist( unit->x, unit->y, pred->att->x, pred->att->y ) <= unit->sel_prop->range )
                                        pred->support[pred->support_count++] = unit;
            if ( pred->support_count == SUPPORT_LIMIT ) break;
        }
        if ( pred->support_count == SUPPORT_LIMIT ) break;
    }
}

/* check surrounding for units that may support the defending unit */
void get_support( Damage_Pred* pred )
{
    Unit *unit = 0;

    if ( ( pred->att->sel_prop->flags & FLYING ) ) {
        /* attacker may be attacked by air_defence and interceptors */
        /* interceptors protect all close units that are not intercetpors */
        if ( !( pred->def->sel_prop->flags & INTERCEPTOR ) ) {
/*        if ( pred->def->sel_prop->flags & FLYING )
            if ( pred->def->sel_prop->flags & BOMBER ||
                 pred->def->cur_prop.attack[pred->att->sel_prop->type] == 0
            ) {*/
                /* ok, check surrounding tiles for interceptors and add as support */
                if ( ( unit = is_interceptor( pred, pred->def->x, pred->def->y - 1 ) ) != 0 )
                    pred->support[pred->support_count++] = unit;
                if ( ( unit = is_interceptor( pred, pred->def->x, pred->def->y + 1 ) ) != 0 )
                    pred->support[pred->support_count++] = unit;
                if ( ( unit = is_interceptor( pred, pred->def->x + 1, pred->def->y ) ) != 0 )
                    pred->support[pred->support_count++] = unit;
                if ( ( unit = is_interceptor( pred, pred->def->x - 1, pred->def->y ) ) != 0 )
                    pred->support[pred->support_count++] = unit;
                if ( pred->def->x & 1 ) {
                    if ( ( unit = is_interceptor( pred, pred->def->x + 1, pred->def->y + 1) ) != 0 )
                        pred->support[pred->support_count++] = unit;
                    if ( ( unit = is_interceptor( pred, pred->def->x - 1, pred->def->y + 1) ) != 0 )
                        pred->support[pred->support_count++] = unit;
                }
                else {
                    if ( ( unit = is_interceptor( pred, pred->def->x + 1, pred->def->y - 1) ) != 0 )
                        pred->support[pred->support_count++] = unit;
                    if ( ( unit = is_interceptor( pred, pred->def->x - 1, pred->def->y - 1) ) != 0 )
                        pred->support[pred->support_count++] = unit;
                }
            }
        /* check air defences */
        add_air_defs( pred );
    }
    else {
        /* its a ground unit so check for artilleries */
        add_arts( pred );
    }
}

/* add prediction */
void add_pred( Dyn_List *list, Unit *att, Unit *def )
{
    Damage_Pred *pred = 0;

    /* get mem */
    pred = calloc( 1, sizeof( Damage_Pred ) );
    pred->x = pred->y = -1;
    /* assign */
    pred->att = att; pred->def = def;
    /* damage reports */
    pred->def_damage = get_damage( att, def, DAMAGE_PRED ) * 10; /* *10 because absolute damage's returned */
    /* defender able to shoot back? */
    if ( can_attack_unit( def, att, DEFEND ) )
        pred->att_damage = get_damage( def, att, DAMAGE_PRED ) * 10; /* *10 because absolute damage's returned */
    /* check out supporting fire */
    get_support( pred );
    /* add to list */
    dl_add( list, pred );
}

/* get damage predictions recursively for unit and add to list */
void get_pred_rec( Dyn_List *list, Unit *att, int x, int y, int range )
{
    /* break condition */
    if ( range <= 0 ) return;
    if ( x < 0 || y < 0 || x >= map.width || y >= map.height )
        return;
    if ( range <= mask_tile( x, y )->aux )
        return;

    /* check if there is an attackable target ( must see map tile )*/
    if ( mask_tile( x, y )->spot ) {
        if ( map_tile( x, y )->unit )
            if ( is_target( att, map_tile( x, y )->unit ) )
                add_pred( list, att, map_tile( x, y )->unit );
        if ( map_tile( x, y )->air_unit )
            if ( is_target( att, map_tile( x, y )->air_unit ) )
                add_pred( list, att, map_tile( x, y )->air_unit );
    }

    /* use aux mask to optimize */
    mask_tile( x, y )->aux = range;
    range--;

    if ( range ) {
        /* recursive calls */
        get_pred_rec( list, att, x, y - 1, range);
        get_pred_rec( list, att, x, y + 1, range );
        get_pred_rec( list, att, x + 1, y, range );
        get_pred_rec( list, att, x - 1, y, range );
        /* secondary neighbors differ depending on x is odd or even */
        if ( x & 1 ) {
            get_pred_rec( list, att, x + 1, y + 1, range );
            get_pred_rec( list, att, x - 1, y + 1, range );
        }
        else {
            get_pred_rec( list, att, x + 1, y - 1, range );
            get_pred_rec( list, att, x - 1, y - 1, range );
        }
    }
}

/* check surrounding of passed unit for targets and set up a list of damage predictions */
Damage_Pred* get_pred( Unit *att, int *count )
{
    Dyn_List list;
    Damage_Pred *pred = 0;
    int i;

    /* setup list */
    dl_init( &list, AUTO_DELETE | NO_CALLBACK, 0 );

    /* auxiliary mask is used to optimize calls */
    clear_mask( F_AUX );

    /* get targets */
    get_pred_rec( &list, att, att->x, att->y, att->cur_prop.range + 1 );

    /* create static list */
    *count = list.count;
    if ( list.count > 0 ) {
        pred = calloc( list.count, sizeof( Damage_Pred ) );
        for ( i = 0; i < *count; i++ )
            memcpy( &pred[i], (Damage_Pred*)dl_get( &list, i ), sizeof( Damage_Pred ) );
    }

    /* clear list */
    dl_clear( &list );

    return pred;
}

/* select unit, get possible targets, and save moving range */
/* return 1 if selection happened */
int select_unit( Engine *engine, Unit *unit )
{
    /* can this player select this unit ? */
    if ( unit && unit->player_id != engine->player_id ) return 0;

    engine->sel_unit = unit;

    if ( engine->sel_unit == 0 ) {

        /* if unselected you doesn't know which unit may undo its move so fuck it */
        engine->undo_ok = 0;
        clear_backup( &engine->backup );
        /* unselected */
        return 0;

    }

    /* get possible targets */
    FREE( engine->pred );
    engine->pred = get_pred( unit, &engine->pred_count );

    /* get possible mergeing partners */
    check_merge_units( engine, engine->sel_unit );

    /* store moving range */
    set_move_mask( engine, unit );

    return 1;
}

/* check if unit can embark at wanted position to wanted type */
int unit_can_embark( Engine *engine, Unit *unit, int x, int y, int type )
{
    if ( type == AIR_EMBARK ) {

        if ( unit->embark != NO_EMBARK ) return 0;
        if ( map_tile( x, y )->air_unit ) return 0;
        if ( engine->player->air_tran_used >= engine->player->air_tran_count ) return 0;
        if ( !unit->no_action_yet ) return 0;
        if ( !( map_tile( x, y )->prop->flags & AIRFIELD ) ) return 0;
        if ( unit->sel_prop->flags & SWIMMING ) return 0;
        if ( unit->sel_prop->flags & FLYING ) return 0;
        if ( unit->sel_prop->flags & WHEELED ) return 0;
        if ( unit->sel_prop->flags & TRACKED ) return 0;
        return 1;

    }
    if ( type == SEA_EMBARK ) {

        if ( unit->embark != NO_EMBARK ) return 0;
        if ( !( map_tile( x, y )->prop->flags & WATER ) ) return 0;
        if ( map_tile( x, y )->unit ) return 0;
        if ( engine->player->sea_tran_used >= engine->player->sea_tran_count ) return 0;
        if ( !unit->no_action_yet ) return 0;
        return 1;

    }

    return 0;
}

/* check if unit can debark at wanted position to wanted type */
int unit_can_debark( Engine *engine, Unit *unit, int x, int y, int type )
{
    if ( type == SEA_EMBARK ) {

        if ( unit->embark != SEA_EMBARK ) return 0;
        if ( map_tile( x, y )->unit ) return 0;
        if ( !unit->no_action_yet ) return 0;
        if ( map_tile( x, y )->prop->flags & WATER ) return 0;
        return 1;

    }
    if ( type == AIR_EMBARK ) {

        if ( unit->embark != AIR_EMBARK ) return 0;
        if ( map_tile( x, y )->unit ) return 0;
        if ( !unit->no_action_yet ) return 0;
        if ( map_tile( x, y )->prop->flags & WATER ) return 0;
        if ( !( map_tile( x, y )->prop->flags & AIRFIELD ) && !( unit->prop.flags & PARACHUTE ) )
            return 0;
        return 1;

    }

    return 0;
}

/* embark unit to sea/air (specified by type) at coords x,y */
void embark_unit( Engine *engine, Unit *unit, int x, int y, int type )
{
    if ( type == AIR_EMBARK ) {

        /* action taken */
        unit->no_action_yet = 0;
        /* abandon ground transporter */
        unit->ground_tran = 0;
        /* set and change to air_tran_prop */
        memcpy( &unit->tran_prop, engine->player->def_air_tran, sizeof( Unit_Lib_Entry ) );
        unit->tran_prop_set = 1;
        unit->sel_prop = &unit->tran_prop;
        unit->embark = AIR_EMBARK;
        /* unit now flies */
        map_tile( x, y )->air_unit = unit;
        map_tile( x, y )->unit = 0;
        /* save old fuel value and set to transporters maximum fuel */
        unit->old_fuel = unit->cur_prop.fuel;
        give_fuel( unit, 100 );
        /* set full move range */
        unit->cur_prop.mov = unit->tran_prop.mov;
        /* cancel attacks */
        unit->cur_prop.attack_count = 0;
        /* no entrenchment */
        unit->entr = 0;
        /* update_values */
        update_cur_unit_prop( unit );
        /* set action */
        engine->action = EMBARK;
        /* adjust pic offset */
        adjust_unit_pic_offset( unit );
        /* another air_tran in use */
        engine->player->air_tran_used++;
        /* in any case your spotting must be updated */
        if ( update_spot_mask( engine, unit ) )
            deny_undo( engine );
        return;

    }
    if ( type == SEA_EMBARK ) {

        /* action taken */
        unit->no_action_yet = 0;
        /* abandon ground transporter */
        unit->ground_tran = 0;
        /* set and change to sea_tran_prop */
        memcpy( &unit->tran_prop, engine->player->def_sea_tran, sizeof( Unit_Lib_Entry ) );
        unit->tran_prop_set = 1;
        unit->sel_prop = &unit->tran_prop;
        unit->embark = SEA_EMBARK;
        /* update position */
        map.tiles[unit->x][unit->y].unit = 0;
        unit->x = x; unit->y = y;
        map_tile( x, y )->unit = unit;
        /* save old fuel value and set to transporters maximum fuel */
        unit->old_fuel = unit->cur_prop.fuel;
        give_fuel( unit, 100 );
        /* set full move range */
        unit->cur_prop.mov = unit->tran_prop.mov;
        /* cancel attacks */
        unit->cur_prop.attack_count = 0;
        /* no entrenchment */
        unit->entr = 0;
        /* update_values */
        update_cur_unit_prop( unit );
        /* set action */
        engine->action = EMBARK;
        /* adjust pic offset */
        adjust_unit_pic_offset( unit );
        /* another air_tran in use */
        engine->player->sea_tran_used++;
        /* in any case your spotting must be updated */
        if ( update_spot_mask( engine, unit ) )
            deny_undo( engine );
        return;

    }
}

/* debark unit to sea/air (specified by type) at coords x,y */
void debark_unit( Engine *engine, Unit *unit, int x, int y, int type )
{
    if ( type == SEA_EMBARK ) {

        /* action taken */
        unit->no_action_yet = 0;
        /* change back to unit_prop */
        unit->tran_prop_set = 0;
        unit->sel_prop = &unit->prop;
        unit->embark = NO_EMBARK;
        /* set position */
        map.tiles[unit->x][unit->y].unit = 0;
        unit->x = x; unit->y = y;
        map_tile( x, y )->unit = unit;
        /* restore old fuel */
        unit->cur_prop.fuel = unit->old_fuel;
        /* no movement allowed */
        unit->cur_prop.mov = 0;
        /* cancel attacks */
        unit->cur_prop.attack_count = 0;
        /* no entrenchment */
        unit->entr = 0;
        /* update_values */
        update_cur_unit_prop( unit );
        /* set action */
        engine->action = DEBARK;
        /* adjust pic offset */
        adjust_unit_pic_offset( unit );
        /* free occupied sea transporter */
        engine->player->sea_tran_used--;
        /* in any case your spotting must be updated */
        if ( update_spot_mask( engine, unit ) )
            deny_undo( engine );
        return;

    }
    if ( type == AIR_EMBARK ) {

        /* action taken */
        unit->no_action_yet = 0;
        /* change back to unit_prop */
        unit->tran_prop_set = 0;
        unit->sel_prop = &unit->prop;
        unit->embark = NO_EMBARK;
        /* get down to earth */
        map.tiles[unit->x][unit->y].air_unit = 0;
        map.tiles[unit->x][unit->y].unit = unit;
        /* restore old fuel */
        unit->cur_prop.fuel = unit->old_fuel;
        /* no movement allowed */
        unit->cur_prop.mov = 0;
        /* cancel attacks */
        unit->cur_prop.attack_count = 0;
        /* no entrenchment */
        unit->entr = 0;
        /* update_values */
        update_cur_unit_prop( unit );
        /* set action */
        engine->action = DEBARK;
        /* adjust pic offset */
        adjust_unit_pic_offset( unit );
        /* free occupied sea transporter */
        engine->player->air_tran_used--;
        /* in any case your spotting must be updated */
        if ( update_spot_mask( engine, unit ) )
            deny_undo( engine );
        return;

    }
}

/* check if units are enemies */
int is_enemy( Unit *unit, Unit *target_unit )
{
    return ( scen.players[unit->player_id]->dipl[target_unit->player_id] == ENEMIES );
}

/*
====================================================================
Backup data that will be restored when unit move was undone.
====================================================================
*/
void backup_unit( Unit *unit, Unit_Backup *backup )
{
    /* do not overwrite unit if already saved one! */
    if ( !backup->unit_saved ) {
        memcpy( &backup->unit, unit, sizeof( Unit ) );
        backup->unit_saved = 1;
    }
}
void restore_unit( Unit *unit, Unit_Backup *backup )
{
    int new_embark;
    /* the status of the map flag on destination position the unit is currently on */
    /* was saved manually after backup_unit() in start_unit_move() */
    if ( backup->flag_saved ) {
        map_tile( unit->x, unit->y )->player_id = backup->dest_player_id;
        map_tile( unit->x, unit->y )->nation_id = backup->dest_nation_id;
        backup->flag_saved = 0;
    }
    if ( backup->unit_saved ) {
        backup->unit_saved = 0;
        /* get stuff before restoring pointer */
        new_embark = unit->embark;
        /* restore */
        memcpy( unit, &backup->unit, sizeof( Unit ) );
        /* check debark/embark counters */
        if ( unit->embark == NO_EMBARK ) {
            if ( new_embark == AIR_EMBARK )
                scen.players[unit->player_id]->air_tran_used--;
            if ( new_embark == SEA_EMBARK )
                scen.players[unit->player_id]->sea_tran_used--;
        }
        else
            if ( unit->embark == SEA_EMBARK && new_embark == NO_EMBARK )
                scen.players[unit->player_id]->sea_tran_used++;
            else
                if ( unit->embark == AIR_EMBARK && new_embark == NO_EMBARK )
                    scen.players[unit->player_id]->air_tran_used++;
        /* adjust picture as direction may have changed */
        adjust_unit_pic_offset( unit );
        update_cur_unit_prop( unit );
    }
    else
        fprintf( stderr, "restore_unit: error: no unit backuped\n" );
}
void clear_backup( Unit_Backup *backup )
{
    backup->flag_saved = 0;
    backup->unit_saved = 0;
}
/*
====================================================================
Clear backup and undo_ok flag.
====================================================================
*/
void deny_undo( Engine *engine )
{
    engine->undo_ok = 0;
    clear_backup( &engine->backup );
}

/*
====================================================================
If there is a unit on this tile mergeing is allowed with return
the pointer.
====================================================================
*/
Unit* get_merge_unit( Unit *unit, int x, int y )
{
    if ( map_tile( x, y )->unit && units_may_merge( unit, map_tile( x, y )->unit ) )
        return map_tile( x, y )->unit;
    if ( map_tile( x, y )->air_unit && units_may_merge( unit, map_tile( x, y )->air_unit ) )
        return map_tile( x, y )->air_unit;
    return 0;
}
/*
====================================================================
Checks surrounding of unit for other units this unit may merge with.
Sets the static array engine::merge_units as there may be
6 other units at maximum. Includes the setting of mask::merge
====================================================================
*/
void check_merge_units( Engine *engine, Unit *unit )
{
    Unit *partner;
    int x = unit->x, y = unit->y;

    engine->merge_unit_count = 0;

    /* clear merge mask */
    clear_mask( F_MELT );

    /* check surrounding tiles */
    if ( ( partner = get_merge_unit( unit, x + 1, y ) ) != 0 ) {
        engine->merge_units[engine->merge_unit_count++] = partner;
        mask_tile( x + 1, y )->merge = 1;
    }
    if ( ( partner = get_merge_unit( unit, x - 1, y ) ) != 0 ) {
        engine->merge_units[engine->merge_unit_count++] = partner;
        mask_tile( x - 1, y )->merge = 1;
    }
    if ( ( partner = get_merge_unit( unit, x, y + 1 ) ) != 0 ) {
        engine->merge_units[engine->merge_unit_count++] = partner;
        mask_tile( x, y + 1 )->merge = 1;
    }
    if ( ( partner = get_merge_unit( unit, x, y - 1 ) ) != 0 ) {
        engine->merge_units[engine->merge_unit_count++] = partner;
        mask_tile( x, y - 1 )->merge = 1;
    }
    if ( x & 1 ) {
        if ( ( partner = get_merge_unit( unit, x + 1, y + 1 ) ) != 0 ) {
            engine->merge_units[engine->merge_unit_count++] = partner;
            mask_tile( x + 1, y + 1 )->merge = 1;
        }
        if ( ( partner = get_merge_unit( unit, x - 1, y + 1 ) ) != 0 ) {
            engine->merge_units[engine->merge_unit_count++] = partner;
            mask_tile( x - 1, y + 1 )->merge = 1;
        }
    }
    else {
        if ( ( partner = get_merge_unit( unit, x + 1, y - 1 ) ) != 0 ) {
            engine->merge_units[engine->merge_unit_count++] = partner;
            mask_tile( x + 1, y - 1 )->merge = 1;
        }
        if ( ( partner = get_merge_unit( unit, x - 1, y - 1 ) ) != 0 ) {
            engine->merge_units[engine->merge_unit_count++] = partner;
            mask_tile( x - 1, y - 1 )->merge = 1;
        }
    }
}
