/*b
 * Copyright (C) 2001,2002  Rick Richardson
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Rick Richardson <rickr@mn.rr.com>
b*/

#include <locale.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>
#include <ncurses.h>
#include <panel.h>
#include <errno.h>
#include <time.h>
#include <sys/time.h>
#include "error.h"
#include "rc.h"
#include "streamer.h"
#include "curse.h"
#include "linuxtrade.h"
#include "info.h"

static WINDOW	*Win;
static WINDOW	*Subwin;
static PANEL	*Panel;

static STREAMER	Sr;

static char	Symbol[SYMLEN+1];

void
info_popup(char *sym, STREAMER sr, char *type)
{
	int	cols;

	Win = bestwin(30);
	if (!Win)
		error(1, "Can't create info window\n");

	cols = getmaxx(Win);

	wbkgd(Win, Reverse ? A_REVERSE : A_NORMAL);

	box(Win, 0, 0);
	mvwprintw(Win, 0, (cols - strlen(sym))/2, sym);

	Subwin = derwin(Win, getmaxy(Win) - 2, getmaxx(Win) - 2, 1, 1);
	if (!Subwin)
		error(1, "Can't create info subwindow\n");

	mvwhline(Subwin, 6, 0, ACS_HLINE, getmaxx(Subwin));
	if (type[0] == 'f')
		mvwhline(Subwin, 15, 0, ACS_HLINE, getmaxx(Subwin));
	else
		mvwhline(Subwin, 18, 0, ACS_HLINE, getmaxx(Subwin));

	Panel = new_panel(Win);

	Sr = sr;

	strncpy(Symbol, sym, SYMLEN);
	Symbol[SYMLEN] = 0;

	(*sr->send_quickquote)(sr, sym);

	if (sr->send_info)
	{
		if (type[0] == 'f')
		{
			(*sr->send_info)(sr, sym, 9);
			(*sr->send_info)(sr, sym, 7);
			(*sr->send_info)(sr, sym, 6);
		}
		else
		{
			(*sr->send_info)(sr, sym, 4);
			(*sr->send_info)(sr, sym, 5);
		}
	}
	else
	{
		if (type[0] == 'f')
			mvwcenter(Subwin, 15+6,
				"Extended info not available "
				"with this streamer");
		else
			mvwcenter(Subwin, 18+4,
				"Extended info not available "
				"with this streamer");
	}
}

void
info_quickquote(QUICKQUOTE *qq)
{
	int	x;
	char	*p;
	int	cols;

	if (!Win)
		return;

	if (strcmp(qq->sym, Symbol))
		return;

	cols = getmaxx(Win);

	for (p = strchr(qq->fullname, 0); *--p == ' '; *p = 0)
		{}
	for (p = strchr(qq->cusip, 0); *--p == ' '; *p = 0)
		{}
	for (p = strchr(qq->description, 0); *--p == ' '; *p = 0)
		{}
	wattrset(Win, A_BOLD);
	mvwprintw(Win, 0, 3, "%s", qq->sym);
	mvwprintw(Win, 0, (cols-strlen(qq->fullname))/2, "%s", qq->fullname);
	mvwprintw(Win, 0, cols-strlen(qq->cusip)-3, "%s", qq->cusip);
	mvwprintw(Win, getmaxy(Win)-1, (cols-strlen(qq->description))/2,
			"%s", qq->description);
	wattrset(Win, A_NORMAL);

	x = 0;
	mvwprintw(Subwin, 0, x, "Last: %7.2f", qq->last);
	mvwprintw(Subwin, 1, x, "Bid:  %7.2f", qq->bid);
	mvwprintw(Subwin, 2, x, "Ask:  %7.2f", qq->ask);
	mvwprintw(Subwin, 3, x, "High: %7.2f", qq->high);
	mvwprintw(Subwin, 4, x, "Low:  %7.2f", qq->low);
	mvwprintw(Subwin, 5, x, "Vol:  %'d", qq->volume);

	x = 1 * (getmaxx(Subwin)/4);
	mvwprintw(Subwin, 0, x, "Last Tick: %s",
			qq->last_tick>0 ? "Up" : "Down");
	mvwprintw(Subwin, 1, x, "PrevClose: %.2f", qq->prev_close);
	mvwprintw(Subwin, 2, x, "BxA Size:  %dx%d", qq->bid_size, qq->ask_size);
	mvwprintw(Subwin, 3, x, "BxA Exch:  %cx%c", qq->bid_id, qq->ask_id);
	mvwprintw(Subwin, 4, x, "52wk High: %.2f", qq->high52);
	mvwprintw(Subwin, 5, x, "52wk Low:  %.2f", qq->low52);

	x = 2 * (getmaxx(Subwin)/4);
	mvwprintw(Subwin, 0, x, "$ Change: %+.2f",
			(qq->last-qq->prev_close));
	mvwprintw(Subwin, 1, x, "%% Change: %+.3f%%",
			(qq->last-qq->prev_close)/qq->prev_close*100.0);
	mvwprintw(Subwin, 2, x, "LastSize: %d", qq->last_size);
	mvwprintw(Subwin, 3, x, "Time:     %02d:%02d:%02d",
		qq->timetrade/3600, qq->timetrade/60%60, qq->timetrade%60);
	mvwprintw(Subwin, 4, x, "Exch:     %c", qq->exchange);
	mvwprintw(Subwin, 5, x, "Yield:    %.3f%%",
			(qq->annualdiv/100.0) / qq->last );

	x = 3 * (getmaxx(Subwin)/4);
	if (qq->sharesout == 12345678)
		mvwprintw(Subwin, 0, x, "SharesOut: ?");
	else
		mvwprintw(Subwin, 0, x, "SharesOut: %'dK", qq->sharesout);
	mvwprintw(Subwin, 1, x, "AnnualDiv: $%.2f", qq->annualdiv/10000.0);
	if (qq->divMM)
		mvwprintw(Subwin, 2, x, "Div. Paid: %02d/%02d/%04d",
			qq->divMM, qq->divDD, qq->divYY);
	else
		mvwprintw(Subwin, 2, x, "Div. Paid: ?");
	if (qq->last_eps == 12345678)
		mvwprintw(Subwin, 3, x, "LastQ EPS: ?");
	else
		mvwprintw(Subwin, 3, x, "LastQ EPS: $%.2f", qq->last_eps/100.0);
	if (qq->cur_eps == 12345678)
		mvwprintw(Subwin, 4, x, "CurrQ EPS: ?");
	else
		mvwprintw(Subwin, 4, x, "CurrQ EPS: $%.2f", qq->cur_eps/100.0);
	mvwprintw(Subwin, 5, x, "PE Ratio:  %.3f", (qq->last*100.0)/qq->year_eps);

	touchwin(Win);
}

void
info_dividends(char *buf)
{
}

void
info_splits(char *buf)
{
}

void
info_calcs(char *buf)
{
	int	rc;
	int	x, y;
	// Calcs...
	// high-date,low-date,last-updated,high-price,low-price(18.96),
	// moving-avg(9days=30.0877,14day,21day,50day,100day,200days=28.1068),
	// closing-price(week=30.63,mon,qtr,year=30.0625),
	// ave-volume-22days,ave-volume-100days,
	// ave-volatility-20days(0.5151),ave-volatility-6mons,
	// ytd-rate-of-return(0.0275)
	// > _G,4,INTC,
	// < _G4,INTC,11/27/2000,9/21/2001,11/21/2001,46.750000,18.960000,
	// 30.087700,29.205000,27.828000,24.849500,26.960100,28.106800,
	// 30.630000,24.420000,20.440000,30.062500,
	// 52743913,50375889,
	// 0.515100,0.545500,0.027500,
	char	sym[SYMLEN+1];
	int	hiMM, hiDD, hiYY;
	int	loMM, loDD, loYY;
	int	upMM, upDD, upYY;
	float	hi, lo;
	float	ma9, ma14, ma21, ma50, ma100, ma200;
	float	closeW, closeM, closeQ, closeY;
	int	avevol22, avevol100;
	float	avevola20, avevola6mo;
	float	ytdror;
	rc = sscanf(buf+4,
		"%[^,],"
		"%d/%d/%d,"
		"%d/%d/%d,"
		"%d/%d/%d,"
		"%f,%f,"
		"%f,%f,%f,%f,%f,%f,"
		"%f,%f,%f,%f,"
		"%d,%d,"
		"%f,%f,"
		"%f,"
		, sym
		, &hiMM, &hiDD, &hiYY
		, &loMM, &loDD, &loYY
		, &upMM, &upDD, &upYY
		, &hi, &lo
		, &ma9, &ma14, &ma21, &ma50, &ma100, &ma200
		, &closeW, &closeM, &closeQ, &closeY
		, &avevol22, &avevol100
		, &avevola20, &avevola6mo
		, &ytdror
		);

	if (rc != 27)
	{
		if (Debug)
			error(0, "Bogus _G4 rc=%d '%s'\n", rc, buf);
		return;
	}

	if (strcmp(sym, Symbol))
	{
		// error(0, "Bogus symbol '%s', wanted '%s'\n", sym, Symbol);
		return;
	}

	//
	// Fix missing values
	//
	#define fixval(X) if (X <= -214748) X = 0; else
	fixval(ma9);
	fixval(ma14);
	fixval(ma21);
	fixval(ma50);
	fixval(ma100);
	fixval(ma200);
	fixval(closeW);
	fixval(closeM);
	fixval(closeQ);
	fixval(closeY);
	fixval(avevol22);
	fixval(avevol100);
	fixval(avevola20);
	fixval(avevola6mo); if (avevola6mo >= 21474836) avevola6mo = 0;
	fixval(ytdror);

	y = 6;
	mvwprintw(Subwin, ++y, 0, "CALCS");

	x = 0;
	mvwprintw(Subwin, ++y, x, "High Date:            %02d/%02d/%04d",
			hiMM, hiDD, hiYY);
	mvwprintw(Subwin, ++y, x, "High Price:           %7.2f", hi);
	mvwprintw(Subwin, ++y, x, "Low Date:             %02d/%02d/%04d",
			loMM, loDD, loYY);
	mvwprintw(Subwin, ++y, x, "Low Price:            %7.2f", lo);
	mvwprintw(Subwin, ++y, x, "Closing Price [Week]: %7.2f", closeW);
	mvwprintw(Subwin, ++y, x, "Closing Price [Mon.]: %7.2f", closeM);
	mvwprintw(Subwin, ++y, x, "Closing Price [Qtr.]: %7.2f", closeQ);
	mvwprintw(Subwin, ++y, x, "Closing Price [Year]: %7.2f", closeY);
	mvwprintw(Subwin, ++y, x, "YTD Rate of Return:   %7.3f", ytdror);
	mvwprintw(Subwin, ++y, x, "Last Updated:         %02d/%02d/%04d",
			upMM, upDD, upYY);

	y = 6; ++y;
	x = getmaxx(Subwin)/2;
	mvwprintw(Subwin, ++y, x, "Moving avg [  9 day]: %7.3f", ma9);
	mvwprintw(Subwin, ++y, x, "Moving avg [ 14 day]: %7.3f", ma14);
	mvwprintw(Subwin, ++y, x, "Moving avg [ 21 day]: %7.3f", ma21);
	mvwprintw(Subwin, ++y, x, "Moving avg [ 50 day]: %7.3f", ma50);
	mvwprintw(Subwin, ++y, x, "Moving avg [100 day]: %7.3f", ma100);
	mvwprintw(Subwin, ++y, x, "Moving avg [200 day]: %7.3f", ma200);
	mvwprintw(Subwin, ++y, x, "Avg Volume [ 22 day]: %'d", avevol22);
	mvwprintw(Subwin, ++y, x, "Avg Volume [100 day]: %'d", avevol100);
	mvwprintw(Subwin, ++y, x, "Volatility [ 20 day]: %7.3f", avevola20);
	mvwprintw(Subwin, ++y, x, "Volatility [180 day]: %7.3f", avevola6mo);

	touchwin(Win);
}

void
info_histcalcs(char *buf)
{
	int	rc;
	int	x, y;
	// HistCalcs...
	// volatility(1mo...12mo),
	// call-imp-volatility,put-imp-volatility,last-updated
	// > _G,5,INTC,
	// _G5,INTC,0.500800,0.586000,0.621400,0.578100,0.557100,0.545500,
	// 0.540300,0.651300,0.657400,0.656500,0.651500,0.686700,
	// 0.438400,0.441200,11/21/2001,
	char	sym[SYMLEN+1];
	float	vola[12];
	float	call_imp_vola, put_imp_vola;
	int	upMM, upDD, upYY;
	rc = sscanf(buf+4,
		"%[^,],"
		"%f,%f,%f,%f,%f,%f,"
		"%f,%f,%f,%f,%f,%f,"
		"%f,%f,%d/%d/%d,"
		, sym
		, &vola[0], &vola[1], &vola[2], &vola[3]
		, &vola[4], &vola[5], &vola[6], &vola[7]
		, &vola[8], &vola[9], &vola[10]
		, &vola[11]
		, &call_imp_vola, &put_imp_vola
		, &upMM, &upDD, &upYY
		);

	if (rc != 18)
	{
		if (Debug)
			error(0, "Bogus _G5 rc=%d '%s'\n", rc, buf);
		return;
	}

	if (strcmp(sym, Symbol))
	{
		// error(0, "Bogus symbol '%s', wanted '%s'\n", sym, Symbol);
		return;
	}

	//
	// Fix missing values
	//
	fixval(call_imp_vola);
	fixval(put_imp_vola);
	for (x = 0; x < 12; ++x)
		fixval(vola[x]);

	y = 18;
	mvwprintw(Subwin, ++y, 0, "VOLATILITY");

	x = 0;
	mvwprintw(Subwin, ++y, x, "Call Implied Vol.: %7.4f", call_imp_vola);
	mvwprintw(Subwin, ++y, x, " 1 Mo. Volatility: %7.4f", vola[0]);
	mvwprintw(Subwin, ++y, x, " 2 Mo. Volatility: %7.4f", vola[1]);
	mvwprintw(Subwin, ++y, x, " 3 Mo. Volatility: %7.4f", vola[2]);
	mvwprintw(Subwin, ++y, x, " 4 Mo. Volatility: %7.4f", vola[3]);
	mvwprintw(Subwin, ++y, x, " 5 Mo. Volatility: %7.4f", vola[4]);
	mvwprintw(Subwin, ++y, x, " 6 Mo. Volatility: %7.4f", vola[5]);
	mvwprintw(Subwin, ++y, x, "Last Updated:      %02d/%02d/%04d",
			upMM, upDD, upYY);

	y = 18; ++y;
	x = getmaxx(Subwin)/2;
	mvwprintw(Subwin, ++y, x, "Put Implied Vol.:  %7.4f", put_imp_vola);
	mvwprintw(Subwin, ++y, x, " 7 Mo. Volatility: %7.4f", vola[6]);
	mvwprintw(Subwin, ++y, x, " 8 Mo. Volatility: %7.4f", vola[7]);
	mvwprintw(Subwin, ++y, x, " 9 Mo. Volatility: %7.4f", vola[8]);
	mvwprintw(Subwin, ++y, x, "10 Mo. Volatility: %7.4f", vola[9]);
	mvwprintw(Subwin, ++y, x, "11 Mo. Volatility: %7.4f", vola[10]);
	mvwprintw(Subwin, ++y, x, "12 Mo. Volatility: %7.4f", vola[11]);

	touchwin(Win);
}

void
info_earnings(char *buf)
{
	int	rc;
	int	x;
	// Earnings...
	// sym,est-report-date,fiscal-year-ends,???,???,last12mo-eps(1.07),
	// prev12mo-eps(1.49),last-fiscal-year-eps(1.65),
	// prev-fiscal-year-eps(1.16),last-qtr-eps(0.12),
	// prev-qtr-eps(0.16),number-of-brokers(27),cur-qtr-est-eps(0.1),
	// next-qtr-est-eps(0.13),cur-yr-est-eps(0.52),next-yr-est-eps(0.69),
	// 5yr-growth-rate(17.5)
	// > _G,6,INTC,
	// < _G6,INTC,7/17/2001,2,?,?,107,149,165,116,12,16,27,10,13,52,69,1750,
	char	sym[SYMLEN+1];
	int	mm, dd, yy, last12;
	char	ends[32];
	int	prev12, lastfisc;
	int	prevfisc, lastqtr;
	int	prevqtr, brokers, curqtr;
	int	nextqtr, curyr, nextyr;
	int	growth;

	rc = sscanf(buf+4,
		"%[^,],%d/%d/%d,%[^,],%*[^,],%*[^,],%d,"
		"%d,%d,"
		"%d,%d,"
		"%d,%d,%d,"
		"%d,%d,%d,"
		"%d,"
		, sym, &mm, &dd, &yy, ends, &last12
		, &prev12, &lastfisc
		, &prevfisc, &lastqtr
		, &prevqtr, &brokers, &curqtr
		, &nextqtr, &curyr, &nextyr
		, &growth
		);

	if (rc != 17)
	{
		if (Debug)
			error(0, "Bogus _G2 rc=%d '%s'\n", rc, buf);
		return;
	}

	if (strcmp(sym, Symbol))
	{
		// error(0, "Bogus symbol '%s', wanted '%s'\n", sym, Symbol);
		return;
	}

	mvwprintw(Subwin, 7, 0, "EARNINGS");

	x = 0;
	mvwprintw(Subwin,  8, x, "Last 12 months EPS: %7.2f", last12/100.0);
	mvwprintw(Subwin,  9, x, "Prev 12 months EPS: %7.2f", prev12/100.0);
	mvwprintw(Subwin, 10, x, "Last fiscal yr EPS: %7.2f", lastfisc/100.0);
	mvwprintw(Subwin, 11, x, "Prev fiscal yr EPS: %7.2f", prevfisc/100.0);
	mvwprintw(Subwin, 12, x, "Last quarter EPS:   %7.2f", lastqtr/100.0);
	mvwprintw(Subwin, 13, x, "Prev quarter EPS:   %7.2f", prevqtr/100.0);
	mvwprintw(Subwin, 14, x, "Fiscal year ends:   %7.7s", ends);

	x = getmaxx(Subwin)/2;
	mvwprintw(Subwin,  8, x, "Curr Qtr est. EPS:  %7.2f", curqtr/100.0);
	mvwprintw(Subwin,  9, x, "Next Qtr est. EPS:  %7.2f", nextqtr/100.0);
	mvwprintw(Subwin, 10, x, "Curr year est. EPS: %7.2f", curyr/100.0);
	mvwprintw(Subwin, 11, x, "Next year est. EPS: %7.2f", nextyr/100.0);
	mvwprintw(Subwin, 12, x, "Est. report date: %02d/%02d/%4d",
			mm, dd, yy);
	mvwprintw(Subwin, 13, x, "5 year growth rate: %7.2f%%", growth/100.0);
	mvwprintw(Subwin, 14, x, "Number of brokers:     %d", brokers);

	touchwin(Win);
}

void
info_balance(char *buf)
{
	int	rc;
	int	x, y;
	// Balance Sheet...
	// report-date(income,balance,cash),revenue-sales(32,410,000),
	// gross-profit(19,524,000),depreciation-depletion(1,858,000),
	// income(interest=0,other=0,net=8,324,000),current-assets,
	// long-term-investments(2,300,000),property-plant-equipment,
	// total-assets(46,249,000),current-liabilities,long-term-debt,
	// capital-stock(8,712,000),additional-paid-in-capital(0),
	// retained-earnings(28,321,000),
	// cash-from(operating=1,181,000,investing,financing=-750,000),
	// net-change-in-cash(-115,000)
	// > _G,7,INTC,
	// < _G7,INTC,3/31/2001,3/31/2001,3/31/2001,32410000,
	// 19524000,1858000,
	// 0,0,8324000,18739000,
	// 2300000,16774000,
	// 46249000,7387000,704000,
	// 8712000,0,28321000,
	// 1181000,-546000,-750000,
	// -115000,
	char	sym[SYMLEN+1];
	int	incMM, incDD, incYY;
	int	balMM, balDD, balYY;
	int	cashMM, cashDD, cashYY;
	int	revenue_sales;
	int	gross_profit, depreciation;
	int	income_interest, income_other, income_net, current_assets;
	int	lt_investments, property_plant_equip;
	int	total_assets, current_liabilities, lt_debt;
	int	capital_stock, additional_capital, retained_earnings;
	int	cash_from_operating, cash_from_investing, cash_from_financing;
	int	net_change_in_cash;
	rc = sscanf(buf+4,
		"%[^,],%d/%d/%d,%d/%d/%d,%d/%d/%d,%d,"
		"%d,%d,"
		"%d,%d,%d,%d,"
		"%d,%d,"
		"%d,%d,%d,"
		"%d,%d,%d,"
		"%d,%d,"
		"%d,%d,"
		, sym
		, &incMM, &incDD, &incYY
		, &balMM, &balDD, &balYY
		, &cashMM, &cashDD, &cashYY, &revenue_sales
		, &gross_profit, &depreciation
		, &income_interest, &income_other, &income_net, &current_assets
		, &lt_investments, &property_plant_equip
		, &total_assets, &current_liabilities, &lt_debt
		, &capital_stock, &additional_capital, &retained_earnings
		, &cash_from_operating, &cash_from_investing
		, &cash_from_financing, &net_change_in_cash
		);

	if (rc != 29)
	{
		if (Debug)
			error(0, "Bogus _G7 rc=%d '%s'\n", rc, buf);
		return;
	}
	if (strcmp(sym, Symbol))
	{
		// error(0, "Bogus symbol '%s', wanted '%s'\n", sym, Symbol);
		return;
	}

	mvwprintw(Subwin, y = 16, 0, "BALANCE SHEET (amounts in K)");

	x = 0;
	mvwprintw(Subwin, ++y, x, "Report date [Income]:  %02d/%02d/%02d",
			incMM, incDD, incYY);
	mvwprintw(Subwin, ++y, x, "Report date [Balance]: %02d/%02d/%02d",
			balMM, balDD, balYY);
	mvwprintw(Subwin, ++y, x, "Report date [Cash]:    %02d/%02d/%02d",
			cashMM, cashDD, cashYY);
	mvwprintw(Subwin, ++y, x, "Revenue (sales):       %'10d",
			revenue_sales);
	mvwprintw(Subwin, ++y, x, "Long term investments: %'10d",
			lt_investments);
	mvwprintw(Subwin, ++y, x, "Total assets:          %'10d",
			total_assets);
	mvwprintw(Subwin, ++y, x, "Current liabilities:   %'10d",
			current_liabilities);
	mvwprintw(Subwin, ++y, x, "Capital stock:         %'10d",
			capital_stock);
	mvwprintw(Subwin, ++y, x, "Cash from operating:   %'10d",
			cash_from_operating);
	mvwprintw(Subwin, ++y, x, "Cash from investing:   %'10d",
			cash_from_investing);
	mvwprintw(Subwin, ++y, x, "Cash from financing:   %'10d",
			cash_from_financing);

	y = 16;
	x = getmaxx(Subwin)/2;
	mvwprintw(Subwin, ++y, x, "Gross profit:           %'10d",
			gross_profit);
	mvwprintw(Subwin, ++y, x, "Income [Interest]:      %'10d",
			income_interest);
	mvwprintw(Subwin, ++y, x, "Income [Other]:         %'10d",
			income_other);
	mvwprintw(Subwin, ++y, x, "Income [Net]:           %'10d",
			income_net);
	mvwprintw(Subwin, ++y, x, "Long term debt:         %'10d",
			lt_debt);
	mvwprintw(Subwin, ++y, x, "Retained earnings:      %'10d",
			retained_earnings);
	mvwprintw(Subwin, ++y, x, "Current assets:         %'10d",
			current_assets);
	mvwprintw(Subwin, ++y, x, "Net change in cash:     %'10d",
			net_change_in_cash);
	mvwprintw(Subwin, ++y, x, "Depreciation depletion: %'10d",
			depreciation);
	mvwprintw(Subwin, ++y, x, "Property, plant, equip: %'10d",
			property_plant_equip);
	mvwprintw(Subwin, ++y, x, "Additional paidin cap.: %'10d",
			additional_capital);

	touchwin(Win);
}

void
info_dataG(char *buf)
{
	switch (atoi(buf+2))
	{
	case 2:	info_dividends(buf); break;
	case 3: info_splits(buf); break;
	case 4: info_calcs(buf); break;
	case 5: info_histcalcs(buf); break;
	case 6: info_earnings(buf); break;
	case 7: info_balance(buf); break;
	}
}

static void
popdown(void)
{
	del_panel(Panel);
	delwin(Subwin);
	delwin(Win);
	Win = NULL;
	Subwin = NULL;
	Symbol[0] = 0;
}

int
info_command(int c, STREAMER sr)
{
	MEVENT	m;

	switch (c)
	{
	case KEY_F(11):
		print_rect_troff(getbegy(Win), getbegx(Win),
				getmaxy(Win), getmaxx(Win),
				NULL, "screen.tr");
		break;

	case KEY_PRINT:
	case KEY_F(12):
	case CTRL('P'):
		print_window(curscr, LINES,
				get_rc_value(RcFile, "print_cmd"));
		break;

	case KEY_MOUSE:
		if (getmouse(&m) != OK)
			break;

		// Ignore clicks in our window
		if (m.y >= getbegy(Win)
			&& m.y < getbegy(Win) + getmaxy(Win))
			break;

		// popdown and reprocess clicks in main window
		if (ungetmouse(&m) == OK)
			Ungetch = 1;
		popdown();
		return 2;

	default:
		popdown();
		return 1;
	}
	return (0);
}
