/*b
 * Copyright (C) 2001,2002  Rick Richardson
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Rick Richardson <rickr@mn.rr.com>
b*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>
#include <ncurses.h>
#include <panel.h>
#include <errno.h>
#include <time.h>
#include <ctype.h>
#include <unistd.h>
#include <sys/socket.h>
#include "util.h"
#include "error.h"
#include "rc.h"
#include "streamer.h"
#include "curse.h"
#include "linuxtrade.h"
#include "news.h"
#include "article.h"
#include "help.h"

static WINDOW	*Win;
static WINDOW	*Subwin;
static PANEL	*Panel;

static int	Cursor;		// PAD line that cursor is on
static int	Top;		// PAD line that is at top of window
static int	Height;		// Height of scrolling portion of window
static int	MaxTop;		// Maximum value for Top

#define	NUMNEWS	300
static ARTINFO	ArtInfo[NUMNEWS];
static int	NumNews;

static WINDOW	*Savewin;
static ARTINFO	SaveInfo[NUMNEWS];
static int	NumSave;

//
// TODO: make this asynchronous
//

static void
showpad(void)
{
	if (0 && Debug)
	{
		fprintf(stderr,
			"showpad: Top=%d MaxTop=%d Height=%d Cursor=%d\n",
			Top, MaxTop, Height, Cursor);
	}

	mvwaddch(Win, 1, getmaxx(Win)-1,
		Top ? ACS_UARROW : ACS_VLINE);

	mvwaddch(Win, getmaxy(Win)-2, getmaxx(Win)-1,
		(Top < MaxTop) ? ACS_DARROW : ACS_VLINE);

	copywin(Subwin, Win,
		Top, 0,
		1, 1,
		Height, getmaxx(Win)-2,
		FALSE);

	move(LINES-1, CursorX);
	update_panels();
	doupdate();
}

/*
 * Decode HTML character escapes
 */
static void
html_decode(char *in)
{
	char	*out = in;

	while (*in)
	{
		if (*in == '&')
		{
			if (strncmp(in, "&quot;", 6) == 0)
				{ *out++ = '"'; in += 6; }
			else if (strncmp(in, "&amp;", 5) == 0)
				{ *out++ = '&'; in += 5; }
			else if (strncmp(in, "&lt;", 4) == 0)
				{ *out++ = '<'; in += 4; }
			else if (strncmp(in, "&gt;", 4) == 0)
				{ *out++ = '>'; in += 4; }
			else
				*out++ = *in++;
		}
		else
			*out++ = *in++;
	}
	*out = 0;
}

static void
schwab_decode(char *in)
{
	char		*out = in;
	unsigned char	ch;

	while ((ch = *in++))
	{
		switch (ch)
		{
		case 0x1b:
			*out++ = ' ';
			break;
		default:
			if (isprint(ch))
				*out++ = ch;
			break;
		}
	}
	*out = 0;
}

static int
duplicate_headline(char	*hline, char *oldhline, int bufsiz)
{
	// Simple attempt to get rid of some, but not all, duplicates
	if (strcmp(hline, oldhline) == 0)
		return TRUE;

	strncpy(oldhline, hline, bufsiz);
	oldhline[bufsiz-1] = 0;
	return FALSE;
}

/*
 * Save a headline in the save pad, for possible later display
 */
void
save_headline(char *sym, char *artkey, char *headline, time_t utime,
		char *source, STREAMER sr, int offset)
{
	int		y;
	char    	*datefmt = "%m/%d %H:%M";
	char    	datetimebuf[64];
	struct tm	*tmp;
	static char	oldhline[256];

	if (duplicate_headline(headline, oldhline, sizeof(oldhline)))
		return;

	// We do this here in case someday we figure out how to get fancy
	if (sr)
		schwab_decode(headline);

	// format date and time
	tmp = localtime(&utime);
	strftime(datetimebuf, sizeof(datetimebuf), datefmt, tmp);

	if (ToolMode)
	{
		if (ToolMode & TOOL_NEWS)
			printf("NEWS|%s|%ld|%s|%s|\n",
				sym, utime, datetimebuf, headline);
		return;
	}

	if (!Savewin)
	{
		Savewin = newpad(NUMNEWS, cols80() - 2);
		if (!Savewin)
			error(1, "Can't create news savewin pad\n");
		wbkgd(Savewin, Reverse ? A_REVERSE : A_NORMAL);
	}

	//
	// Place article at beginning or end of list, as desired
	//
	if (offset < 0)
		offset = NumSave + offset;
	wmove(Savewin, offset, 0);
	winsertln(Savewin);

	for (y = NumSave-1; y > offset; --y)
		SaveInfo[y] = SaveInfo[y-1];
	y = offset;


	if (strcmp(sym, "hot") == 0)
	{
		wbkgdset(Savewin,
			(Reverse?A_REVERSE:A_NORMAL) | COLOR_PAIR(REDonBG));
		wattrset(Savewin, COLOR_PAIR(REDonBG));
	}
	else if (strcmp(sym, "inplay") == 0)
	{
		wbkgdset(Savewin,
			(Reverse?A_REVERSE:A_NORMAL) | COLOR_PAIR(GREENonBG));
		wattrset(Savewin, COLOR_PAIR(GREENonBG));
	}

	mvwprintw(Savewin, y, 0,
		"%-5.5s %11.11s %-*.*s",
		sym,
		datetimebuf,
		getmaxx(Savewin) - 12 - 6,
		getmaxx(Savewin) - 12 - 6,
		headline);
	wattrset(Savewin, A_NORMAL);
	wbkgdset(Savewin, Reverse ? A_REVERSE : A_NORMAL);

	strncpy(SaveInfo[y].artkey, artkey, AI_ARTKEYLEN);
	SaveInfo[y].artkey[AI_ARTKEYLEN] = 0;

	SaveInfo[y].utime = utime;

	strncpy(SaveInfo[y].source, source, AI_SOURCELEN);
	SaveInfo[y].source[AI_SOURCELEN] = 0;

	SaveInfo[y].sr = sr;

	// SaveInfo[NumNews].words = words;

	if (NumSave < NUMNEWS)
		++NumSave;
}

/*
 * Add a headline to the popup window
 */
void
add_headline(char *sym, char *headline, char *artkey, time_t utime,
		char *source, STREAMER sr, int atend)
{
	int		y;
	char    	*datefmt = "%m/%d %H:%M";
	char    	datetimebuf[64];
	struct tm	*tmp;
	static char	oldhline[256];

	if (!Subwin)
		return;
	if (NumNews == NUMNEWS)
		return;

	if (duplicate_headline(headline, oldhline, sizeof(oldhline)))
		return;

	// We do this here in case someday we figure out how to get fancy
	if (sr)
		schwab_decode(headline);

	//
	// Place article at beginning or end of list, as desired
	//
	if (atend || NumNews == 0)
		y = NumNews;
	else
	{
		wlinecursor(Subwin, Cursor, 0, 0, FALSE, RevOrBold);
		wmove(Subwin, 0, 0);
		winsertln(Subwin);
		for (y = NumNews; y > 0; --y)
			ArtInfo[y] = ArtInfo[y-1];
		y = 0;
	}

	// format date and time
	tmp = localtime(&utime);
	strftime(datetimebuf, sizeof(datetimebuf), datefmt, tmp);

	mvwprintw(Subwin, y, 0,
		"%11.11s %-*.*s",
		datetimebuf,
		getmaxx(Subwin) - 12,
		getmaxx(Subwin) - 12,
		headline);
	wlinecursor(Subwin, Top = Cursor = 0, 0, 0, TRUE, RevOrBold);

	strncpy(ArtInfo[y].artkey, artkey, AI_ARTKEYLEN);
	ArtInfo[y].artkey[AI_ARTKEYLEN] = 0;

	ArtInfo[y].utime = utime;

	strncpy(ArtInfo[y].source, source, AI_SOURCELEN);
	ArtInfo[y].source[AI_SOURCELEN] = 0;

	ArtInfo[y].sr = sr;

	// ArtInfo[NumNews].words = words;

	++NumNews;
	MaxTop = NumNews - Height; if (MaxTop < 0) MaxTop = 0;
	//touchwin(Win);
	showpad();
}

static void
get_quotecom_news(char *sym)
{
	int	y;
	FILE	*fp;
	char	buf[512];
	char	date[512];
	char	time[512];
	char	artno[512];
	char	source[512];
	int	words;
	char	headline[512];

	if (strcmp(sym, "DJI") == 0)
		sym = "INDEX:INDU";
	else if (strcmp(sym, "COMP") == 0)
		sym = "INDEX:COMPX";

	sprintf(buf, "%s \""
		"http://fast.quote.com/fq/quotetracker/headlines?symbols="
		"%s\"", SUCKURL, sym);

	fp = popen(buf, "r");
	if (!fp)
	{
		mvwprintw(Subwin, getmaxy(Subwin)/2, 20,
				"Can't access headlines");
		touchwin(Win);
		return;
	}

	//
	//INTC	12/18/2002	13:00	30603352	BusinessWire	
	//786	Intel Appoints 13 New Vice Presidents
	//
	while (fscanf(fp, "%*[^\t]\t%[^\t]\t%[^\t]\t%[^\t]\t%[^\t]\t%d\t%[^\n]",
			date, time, artno, source, &words, headline) == 6)
	{
		html_decode(headline);
		add_headline(sym, headline, artno,
				datetime2unix(date, time),
				source, NULL, TRUE);
	}
	pclose(fp);

	y = NumNews;
	while (y < getmaxy(Subwin))
	{
		mvwprintw(Subwin, y++, 0, " ");
	}
}

static void
get_streamer_news(STREAMER sr, char *sym)
{
	(*sr->send_headlines)(sr, sym, 25);
}

void
news_popup(STREAMER sr, STOCK *sp, int mode)
{
	int	cols;
	char	*sym;

	if (!sp || mode == NEWS_SAVED)
		sym = "Saved";
	else
	{
		sym = sp->sym;
		sp->alerting &= ~6;
		display_alert(sp, FALSE);
	}

	#if 0
		Win = bestwin(14);
	#else
		Win = bestrightwin(14, TRUE);
	#endif
	if (!Win)
		error(1, "Can't create news window\n");

	cols = getmaxx(Win);
	Height = getmaxy(Win) - 2 - 2;

	wbkgd(Win, Reverse ? A_REVERSE : A_NORMAL);

	box(Win, 0, 0);
	mvwprintw(Win, 0, cols/2 - 3 - strlen(sym), " %s News ", sym);

	// Subwin = derwin(Win, getmaxy(Win) - 2, cols - 2, 1, 1);
	Subwin = newpad(NUMNEWS, cols - 2);
	if (!Subwin)
		error(1, "Can't create news pad\n");
	wbkgd(Subwin, Reverse ? A_REVERSE : A_NORMAL);

	Panel = new_panel(Win);

	NumNews = 0;
	Top = MaxTop = Cursor = 0;

	switch (mode)
	{
	case NEWS_DEFAULT:
		if (sr && sr->send_headlines && sr->usenews)
			get_streamer_news(sr, sym);
		else
			get_quotecom_news(sym);
		break;
	case NEWS_MAIN:
		get_quotecom_news(sym);
		break;
	case NEWS_STREAMER:
		if (sr && sr->send_headlines)
			get_streamer_news(sr, sym);
		else
			get_quotecom_news(sym);
		break;
	case NEWS_SAVED:
		if (Debug)
		{
			save_headline("hot", "  ", "red line red line red line",
				12345678, "", NULL, 0);
			save_headline("nhot", "  ", "black line black line",
				12345678, "", NULL, 0);
			save_headline("inplay", "  ", "blue line blue line",
				12345678, "", NULL, 0);
			save_headline("hot", "  ", "red line red line red line",
				12345678, "", NULL, 0);
		}
		NumNews = NumSave;
		MaxTop = NumNews - Height; if (MaxTop < 0) MaxTop = 0;
		memcpy(ArtInfo, SaveInfo, sizeof(ArtInfo));
		overwrite(Savewin, Subwin);
		wlinecursor(Subwin, Cursor, 0, 0, TRUE, RevOrBold);
		break;
	}

	mvwprintw(Win, getmaxy(Win) - 2, 1,
		" Use the up/down arrows to select an article, "
		"then press enter to view it\n");

	//touchwin(Win);
	showpad();
}

static void
popdown(void)
{
	del_panel(Panel);
	delwin(Subwin);
	delwin(Win);
	Win = NULL;
	Subwin = NULL;
}

int
news_command(int c, STREAMER sr)
{
	static int	(*handler)(int c, STREAMER sr);
	int		rc;
	MEVENT		m;
	ARTINFO		*ai;

	if (handler)
	{
		rc = (*handler)(c, sr);
		if (rc > 0)
		{
			handler = NULL;
			rc = 0;
			wrefresh(curscr);	// Hack for panel refresh prob
		}
		else if (rc == 0)
		{
			touchwin(Win);
			move(LINES-1, CursorX);
			update_panels(); refresh(); // doupdate();
			wrefresh(curscr);	// Hack for panel refresh prob
		}
		return rc;
	}

	switch (c)
	{
	case '\f':
		move(LINES-1, CursorX);
		wrefresh(curscr);
		return 0;
	case '?':
		handler = help_command;
		help_popup("news");
		break;
	case 'a':
	case KEY_ENTER:
	case '\r':
		handler = article_command;
		ai = &ArtInfo[Cursor];
		if (ai->artkey[0])
			article_popup(ai);
		else
		{
			// Popup inplay window
			ungetch('p');
			Ungetch = 1;
			popdown();
			return 2;
		}
		break;
	case 'j':
	case KEY_DOWN:
		if (++Cursor >= NumNews)
		{
			--Cursor;
			beep();
			break;
		}

		wlinecursor(Subwin, Cursor-1, 0, 0, FALSE, RevOrBold);
		wlinecursor(Subwin, Cursor, 0, 0, TRUE, RevOrBold);
		if (Cursor >= (Top+Height))
			++Top;
		showpad();
		break;
	case 'k':
	case KEY_UP:
		if (--Cursor < 0)
		{
			++Cursor;
			beep();
			break;
		}
		wlinecursor(Subwin, Cursor+1, 0, 0, FALSE, RevOrBold);
		wlinecursor(Subwin, Cursor, 0, 0, TRUE, RevOrBold);
		if (Cursor < Top)
			--Top;
		showpad();
		break;
	case '-':
	case KEY_PPAGE:
		if (Top == 0)
		{
			beep();
			break;
		}
		wlinecursor(Subwin, Cursor, 0, 0, FALSE, RevOrBold);
		Top -= Height - 1;
		if (Top < 0)
			Top = 0;
		Cursor = Top;
		wlinecursor(Subwin, Cursor, 0, 0, TRUE, RevOrBold);
		showpad();
		break;
	case '+':
	case ' ':
	case KEY_NPAGE:
		if (Top >= MaxTop)
		{
			beep();
			break;
		}
		wlinecursor(Subwin, Cursor, 0, 0, FALSE, RevOrBold);
		Top += Height - 1;
		if (Top > MaxTop)
			Top = MaxTop;
		Cursor = Top;
		wlinecursor(Subwin, Cursor, 0, 0, TRUE, RevOrBold);
		showpad();
		break;
	case '0':
	case KEY_HOME:
		if (Top == 0)
			beep();
		wlinecursor(Subwin, Cursor, 0, 0, FALSE, RevOrBold);
		Cursor = Top = 0;
		wlinecursor(Subwin, Cursor, 0, 0, TRUE, RevOrBold);
		showpad();
		break;
	case '$':
	case KEY_END:
		if (Top >= MaxTop)
			beep();
		wlinecursor(Subwin, Cursor, 0, 0, FALSE, RevOrBold);
		Cursor = Top = MaxTop;
		wlinecursor(Subwin, Cursor, 0, 0, TRUE, RevOrBold);
		showpad();
		break;

	case KEY_F(11):
		print_rect_troff(getbegy(Win), getbegx(Win),
				getmaxy(Win), getmaxx(Win),
				NULL, "screen.tr");
		break;
	case KEY_PRINT:
	case KEY_F(12):
	case CTRL('P'):
		print_window(curscr, LINES,
				get_rc_value(RcFile, "print_cmd"));
		break;

	case KEY_MOUSE:
		if (getmouse(&m) != OK)
			break;

		// Ignore clicks in our window
		// TODO: set current line
		if (m.y >= getbegy(Win)
			&& m.y < getbegy(Win) + getmaxy(Win))
			break;

		// popdown and reprocess clicks in main window
		if (ungetmouse(&m) == OK)
			Ungetch = 1;
		popdown();
		return 2;

	case 033:
	case 'q':
	case 'x':
		popdown();
		return 2;
	default:
		beep();
		break;
	}
	return 0;
}
