import unittest
import shutil
import os

import Storage


def isfile(dir, basename):
    return os.path.isfile(os.path.join(dir, basename))


class StorageTest(unittest.TestCase):

    name1 = u"unittest-1.lodju"
    name2 = u"unittest-2.lodju"

    xml1 = os.path.join(name1, u"lodju.xml")
    orig1 = os.path.join(name1, u"orig")
    orignew1 = os.path.join(name1, u"orig.new")
    thumb1 = os.path.join(name1, u"thumb")
    thumbnew1 = os.path.join(name1, u"thumb.new")

    xml2 = os.path.join(name2, u"lodju.xml")
    orig2 = os.path.join(name2, u"orig")
    orignew2 = os.path.join(name2, u"orig.new")
    thumb2 = os.path.join(name2, u"thumb")
    thumbnew2 = os.path.join(name2, u"thumb.new")

    def remove_dirs(self):
    	for name in [self.name1, self.name2]:
	    if os.path.exists(name):
		shutil.rmtree(name)

    def setUp(self):
    	self.remove_dirs()
    
    def tearDown(self):
    	self.remove_dirs()

    def testNew(self):
    	# Create a new Storage object. Make sure nothing happens on disk.
    	self.failUnless(not os.path.exists(self.name1))
    	st = Storage.Storage(self.name1)
	self.failUnlessEqual(st.filename, os.path.abspath(self.name1))
    	self.failUnless(not os.path.exists(self.name1))

    def testSave(self):
    	self.failUnless(not os.path.exists(self.name1))
    	st = Storage.Storage(self.name1)

	st.new_file(u"lodju.xml", u"hello, world")
    	self.failUnless(os.path.isdir(self.name1))
    	self.failUnless(os.path.isfile(self.xml1 + u".new"))

    	f = st.new_original(u"42")
	self.failUnless(f != None)
	self.failUnless(isfile(self.orignew1, u"42.new"))
    	st.close_original(u"42", f)
	self.failUnless(isfile(self.orignew1, u"42"))
	
	f = st.new_thumbnail(u"69")
	self.failUnless(f != None)
	self.failUnless(isfile(self.thumbnew1, u"69.new"))
    	st.close_thumbnail(u"69", f)
	self.failUnless(isfile(self.thumbnew1, u"69"))

	st.save()

    	self.failUnless(os.path.isdir(self.name1))
    	self.failUnless(os.path.isfile(self.xml1))
    	self.failUnless(os.path.isdir(self.orig1))
    	self.failUnless(os.path.isdir(self.orignew1))
    	self.failUnless(os.path.isdir(self.thumb1))
    	self.failUnless(os.path.isdir(self.thumbnew1))
    	self.failUnlessEqual(os.listdir(self.orignew1), [])
    	self.failUnlessEqual(os.listdir(self.thumbnew1), [])
	self.failUnless(isfile(self.orig1, u"42"))
	self.failUnless(isfile(self.thumb1, u"69"))

    def testSaveAs(self):
    	self.failIf(os.path.exists(self.name1))
    	self.failIf(os.path.exists(self.name2))
	st = Storage.Storage(self.name1)

	st.new_file(u"lodju.xml", u"hello, world")
    	self.failUnless(os.path.isdir(self.name1))
    	self.failUnless(os.path.isfile(self.xml1 + u".new"))

    	f = st.new_original(u"42")
	st.close_original(u"42", f)
	
	f = st.new_thumbnail(u"69")
	st.close_thumbnail(u"69", f)

	st.save()

    	self.failUnless(os.path.exists(self.name1))
    	self.failIf(os.path.exists(self.name2))
	
	self.failUnlessEqual(st.filename, os.path.abspath(self.name1))
	st.save_as(self.name2)
	self.failUnlessEqual(st.filename, os.path.abspath(self.name2))
	
    	self.failUnless(os.path.exists(self.name1))
    	self.failUnless(os.path.exists(self.name2))
	
    	self.failUnless(os.path.isfile(self.xml2))
    	self.failUnless(isfile(self.orig2, u"42"))
    	self.failUnless(isfile(self.thumb2, u"69"))

    def testDiscardUnnamed(self):
    	st = Storage.Storage(u"")
	self.failIfEqual(st.filename, u"")
	self.failUnless(os.path.isdir(st.filename))
	st.new_file(u"lodju.xml", u"hello, world")
	st.discard()
	self.failIf(os.path.isdir(st.filename))
	
    def testDiscardExistingClean(self):
    	st = Storage.Storage(self.name1)
	st.new_file(u"lodju.xml", u"hello, world")
	f = st.new_original(u"42")
	st.close_original(u"42", f)
	st.save()
	self.failUnless(os.path.isdir(self.name1))
	# We now have an existing storage structure.
	
	st2 = Storage.Storage(self.name1)
	st2.discard()
	self.failUnless(os.path.isdir(self.name1))
	
	st3 = Storage.Storage(self.name1)
	st3.new_file(u"lodju.xml", u"eek")
	st3.discard()
	self.failUnless(os.path.isdir(self.name1))
	self.failIf(os.path.exists(os.path.join(self.name1, u"lodju.xml.new")))
	f = open(os.path.join(self.name1, u"lodju.xml"))
	self.failUnlessEqual(u"hello, world", f.read())
	f.close()

    def testReadFile(self):
    	st = Storage.Storage(self.name1)
	self.failUnlessEqual(st.get_file(u"lodju.xml"), u"")
	st.new_file(u"lodju.xml", u"hello, world")
	self.failUnlessEqual(st.get_file(u"lodju.xml"), u"hello, world")
	st.save()
	self.failUnlessEqual(st.get_file(u"lodju.xml"), u"hello, world")

    def testGetOriginal(self):
    	st = Storage.Storage(self.name1)
	self.failUnlessEqual(st.get_original(u"42"), None)
    	f = st.new_original(u"42")
	st.close_original(u"42", f)
	self.failIfEqual(st.get_original(u"42"), None)
	st.save()
	self.failIfEqual(st.get_original(u"42"), None)

    def testGetThumbnail(self):
    	st = Storage.Storage(self.name1)
	self.failUnlessEqual(st.get_thumbnail(u"42"), None)
    	f = st.new_thumbnail(u"42")
	st.close_thumbnail(u"42", f)
	self.failIfEqual(st.get_thumbnail(u"42"), None)
	st.save()
	self.failIfEqual(st.get_thumbnail(u"42"), None)

    def testRemove(self):
    	st = Storage.Storage(self.name1)
    	f = st.new_original(u"42")
	st.close_original(u"42", f)
	self.failUnless(isfile(self.orignew1, u"42"))
	st.remove(u"42")
	self.failUnless(isfile(self.orignew1, u"42"))
	st.save()
	self.failIf(isfile(self.orig1, u"42"))

    	f = st.new_original(u"69")
	st.close_original(u"69", f)
	self.failUnless(isfile(self.orignew1, u"69"))
	st.save()
	self.failUnless(isfile(self.orig1, u"69"))
	st.remove(u"69")
	self.failUnless(isfile(self.orig1, u"69"))
	st.save()
	self.failIf(isfile(self.orig1, u"69"))

    def testRename(self):
    	self.failIf(os.path.exists(self.name1))
    	self.failIf(os.path.exists(self.name2))
    	st = Storage.Storage(self.name1)
	st.save()

    	self.failUnless(os.path.exists(self.name1))
    	self.failIf(os.path.exists(self.name2))
	st.rename(self.name2)
    	self.failIf(os.path.exists(self.name1))
    	self.failUnless(os.path.exists(self.name2))
	self.failUnlessEqual(st.filename, self.name2)
