/*
 * Tag vocabulary access
 *
 * Copyright (C) 2003,2004,2005  Enrico Zini <enrico@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <apt-front/updater.h>
#include <apt-front/cache/component/tags.h>
#include <apt-front/utils/debdbparser.h>
#include <apt-front/utils/paths.h>
#include <apt-front/utils/vocabularymerger.h>

#include <tagcoll/StdioParserInput.h>
#include <tagcoll/MemParserInput.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/mman.h>

#include <errno.h>

using namespace aptFront;
using namespace cache;
using namespace component;
using namespace std;
using namespace Tagcoll;

//#define TRACE
#ifdef TRACE
#include <iostream>
static bool trace = false;
#endif

std::string Tags::componentName() { return "Tags"; }
time_t Tags::computeTimestamp()
{
	return utils::Path::timestamp(utils::Path::vocabularyIndex());
}

int Tags::FacetIndex::id(const char* name) const
{
	if (size() == 0) return -1;
	int begin, end;

	/* Binary search */
	begin = -1, end = size();
	while (end - begin > 1)
	{
		int cur = (end + begin) / 2;
		if (strcmp(item(cur)->name, name) > 0)
			end = cur;
		else
			begin = cur;
	}

	if (begin == -1 || strcmp(item(begin)->name, name) != 0)
		//throw NotFoundException(string("looking for the ID of string ") + str);
		return -1;
	else
		return begin;
}

static inline int tagcmp(const char* tag1, const char* tag2)
{
	const char* tsep1 = strstr(tag1, "::");
	if (tsep1 == NULL) return strcmp(tag1, tag2);
	const char* tsep2 = strstr(tag2, "::");
	if (tsep2 == NULL) return strcmp(tag1, tag2);

	// See what is the length of the shortest facet
	int len1 = tsep1 - tag1;
	int len2 = tsep2 - tag2;
	int minlen = len1 < len2 ? len1 : len2;

	int res = strncmp(tag1, tag2, minlen);
	if (res != 0)
		// Different facets
		return res;

	if (len1 == len2)
		// If the facet is the same, compare the tags
		return strcmp(tsep1 + 2, tsep2 + 2);
	else
		// Two facets with similar prefixes
		return len1 < len2 ? -1 : 1;
}

int Tags::TagIndex::id(const char* name) const
{
	if (size() == 0) return -1;
	int begin, end;

	/* Binary search */
	begin = -1, end = size();
	while (end - begin > 1)
	{
		int cur = (end + begin) / 2;
		if (tagcmp(item(cur)->name, name) > 0)
			end = cur;
		else
			begin = cur;
	}

	if (begin == -1 || tagcmp(item(begin)->name, name) != 0)
		//throw NotFoundException(string("looking for the ID of string ") + str);
		return -1;
	else
		return begin;
}

Tags::Tags()
	: voc_fd(-1), voc_size(0), voc_buf(0)
{
	//init(utils::Path::vocabulary());
}

Tags::~Tags()
{
	// Unmap and close the file
	if (voc_buf)
		munmap((void*)voc_buf, voc_size);
	if (voc_fd != -1)
		close(voc_fd);
}

void Tags::init(Cache& c)
{
	Updater::updateTags();

	m_timestamp = computeTimestamp();

	string vocfname = utils::Path::vocabulary();
	string idxfname = utils::Path::vocabularyIndex();

	mastermmap.init(idxfname);

	// Initialize the facet and tag indexes
	findex.init(mastermmap, 0);
	tindex.init(mastermmap, 1);

	// MMap the vocabulary

	// Open the file
	voc_fname = vocfname;
	if ((voc_fd = open(voc_fname.c_str(), O_RDONLY)) == -1)
		throw SystemException(errno, "opening vocabulary file " + voc_fname);

	off_t size = lseek(voc_fd, 0, SEEK_END);
	if (size == (off_t)-1)
		throw SystemException(errno, "reading the size of vocabulary file " + voc_fname);
	voc_size = size;
	
	// Map the file into memory
	if ((voc_buf = (const char*)mmap(0, voc_size, PROT_READ, MAP_PRIVATE, voc_fd, 0)) == MAP_FAILED)
		throw SystemException(errno, string("mmapping vocabulary file ") + voc_fname);
}

entity::Facet Tags::facetByID(int id) const
{
	return entity::Facet(this, id);
}

entity::Tag Tags::tagByID(int id) const
{
	return entity::Tag(this, id);
}

void Tags::parseVocBuf(std::map<std::string, std::string>& res, size_t ofs, size_t len) const
{
	// Access the right part of the mmapped buffer
	MemParserInput in(voc_fname + "+" + stringf::fmt(ofs) + "-" + stringf::fmt(len), voc_buf + ofs, len);
	utils::DebDBParser parser(in);
	// Parse the raw string data and store it in the cache vector
	parser.nextRecord(res);

	string desc = res["Description"];
	if (!desc.empty())
	{
		size_t pos = desc.find('\n');
		if (pos == string::npos)
			res["_SD_"] = desc;
		else
			res["_SD_"] = desc.substr(0, pos);
	}
}

std::string Tags::tagShortName(int id) const
{
	const char* fullname = tindex.name(id);
	char* sub = strstr(fullname, "::");
	if (sub != NULL)
		return sub + 2;
	else
		return fullname;
}

const std::map<std::string, std::string>& Tags::facetData(int id) const
{
	if (id < 0) return emptyData;

	// Enlarge the cache vector if needed
	if ((unsigned)id >= m_facetData.size())
		m_facetData.resize(id + 1);

	if (m_facetData[id].empty())
		parseVocBuf(m_facetData[id], findex.offset(id), findex.size(id));
		
	return m_facetData[id];
}

const std::map<std::string, std::string>& Tags::tagData(int id) const
{
	if (id < 0) return emptyData;

	// Enlarge the cache vector if needed
	if ((unsigned)id >= m_tagData.size())
		m_tagData.resize(id + 1);

	if (m_tagData[id].empty())
		parseVocBuf(m_tagData[id], tindex.offset(id), tindex.size(id));

	return m_tagData[id];
}
		


#ifdef COMPILE_TESTSUITE

#include <tests/test-utils.h>

namespace tut {

struct cache_component_tags_shar {
    cache_component_tags_shar() {
        aptInit();
        c.open( Cache::OpenDefault | Cache::OpenTags | Cache::OpenReadOnly );
    }
    Cache c;
};
TESTGRP(cache_component_tags);

template<> template<>
void to::test< 1 >()
{
    c.tags(); // this will throw if the open above didn't work
}

template<> template<>
void to::test< 2 >()
{
    ensure( c.tags().hasFacet( "works-with" ) );
    ensure( !c.tags().hasFacet( "blah" ) );
}

template<> template<>
void to::test< 3 >()
{
    ensure( c.tags().hasTag( "works-with::people" ) );
    ensure( !c.tags().hasTag( "works-with::midgets" ) );
}

template<> template<>
void to::test< 4 >()
{
    entity::Tag people = c.tags().tagByName( "works-with::people" ),
                midgets = c.tags().tagByName( "works-with::midgets" ),
                blahg = c.tags().tagByName( "works-with::blahg" ),
                text = c.tags().tagByName( "works-with::text" ),
                people2 = c.tags().tagByName( "works-with::people" );
    ensure( people != midgets );
    ensure( people != text );
    ensure( people != blahg );
    ensure( midgets == blahg );
    ensure( midgets == midgets );
    ensure( people == people2 );
    ensure( people == people );
}

template<> template<>
void to::test< 5 >()
{
    entity::Tag a = c.tags().tagByName( "works-with::people" ),
                b = c.tags().tagByName( "works-with::midgets" );
    OpSet< entity::Tag > s = c.tags().tags(),
                         f = c.tags().tags( "works-with" ),
                         n = c.tags().tags( "nonsense" );
    ensure( s.contains( a ) );
    ensure( f.contains( a ) );
    ensure( s.contains( f ) );
    ensure( !s.contains( b ) );
    ensure( !f.contains( b ) );
    ensure( n.empty() );
}

template<> template<>
void to::test< 6 >()
{
	entity::Facet f = c.tags().facetByName( "works-with" );
    entity::Tag t = c.tags().tagByName( "works-with::people" );
	ensure_equals(f.name(), "works-with");
	ensure_equals(t.name(), "people");
	ensure_equals(t.fullname(), "works-with::people");
}

template<> template<>
void to::test< 7 >()
{
    entity::Facet f = c.tags().facetByName( "works-with" );
    OpSet< entity::Tag > x = c.tags().tags( "works-with" );
    ensure( x == f.tags() );
}

template<> template<>
void to::test< 8 >()
{
    entity::Facet f = c.tags().facetByName( "does-not-work-with" );
    int x = 1;
    try {
        f.tags();
        x = 2;
    } catch (...) {
        x = 3;
    }
    ensure_equals( x, 3 );
}

template<> template<>
void to::test< 9 >()
{
    entity::Facet f = c.tags().facetByName( "legacy" );
    ensure_equals(f.shortDescription(), "");
    ensure_equals(f.longDescription(), "");
    //ensure_equals(f.shortDescription( "weehee" ), "weehee");
}

template<> template<>
void to::test< 10 >()
{
	// ensure that one-character tag names are parsed correctly
	ensure( c.tags().hasTag( "langdevel::c" ) );
}

template<> template<>
void to::test< 11 >()
{
	// ensure that all tags are somehow working
	Tagcoll::OpSet<entity::Facet> facets = c.tags().facets();

	for (Tagcoll::OpSet<entity::Facet>::const_iterator i = facets.begin();
			i != facets.end(); i++)
	{
		i->name(string("foo"));
		i->shortDescription(string("foo"));
		i->longDescription(string("foo"));
		i->tags();
	}
}

template<> template<>
void to::test< 12 >()
{
	// ensure that all tags are somehow working
	Tagcoll::OpSet<entity::Tag> tags = c.tags().tags();

	for (Tagcoll::OpSet<entity::Tag>::const_iterator i = tags.begin();
			i != tags.end(); i++)
	{
		i->name(string("foo"));
		i->fullname(string("foo"));
		i->shortDescription(string("foo"));
		i->longDescription(string("foo"));
	}
}

// Check for correctness of the first and last tag in the vocabulary
template<> template<>
void to::test< 13 >()
{
	component::Tags& tags = c.tags();

	entity::Tag first = tags.tagByName("accessibility::TODO");
	ensure(first != entity::Tag());
	ensure_equals(first.fullname(), string("accessibility::TODO"));
	ensure_equals(first.name(), string("TODO"));
	ensure_equals(first.shortDescription(), string("Need an extra tag"));

	entity::Tag last = tags.tagByName("x11::xserver");
	ensure(last != entity::Tag());
	ensure_equals(last.fullname(), string("x11::xserver"));
	ensure_equals(last.name(), string("xserver"));
	ensure_equals(last.shortDescription(), string("X Servers"));
}

}

#endif

// vim:set ts=4 sw=4:
