// Copyright (c) 2002 Rob Kaper <cap@capsi.com>
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License version 2.1 as published by the Free Software Foundation.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public License
// along with this library; see the file COPYING.LIB.  If not, write to
// the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
// Boston, MA 02111-1307, USA.

#include <sys/types.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <netinet/in.h>

#include <fcntl.h>
#include <netdb.h>
#include <unistd.h>

#include "listenport.h"

#define	MAXLINE	1024

#ifdef USE_INET_ATON
#define inet_pton(a, b, c) inet_aton(b, c)
#endif

extern int errno;

ListenPort::ListenPort(const std::string ip, const unsigned int port)
{
	m_ipAddr = ip;
	m_port = port;
	m_fd = socket(AF_INET, SOCK_STREAM, 0);
	m_isBound = false;

	struct sockaddr_in servaddr;
	memset(&servaddr, 0, sizeof(servaddr));
	servaddr.sin_family = AF_INET;
	inet_pton(AF_INET, m_ipAddr.c_str(), &servaddr.sin_addr);
//	servaddr.sin_addr.s_addr = htonl(INADDR_ANY); // any host may connect
	servaddr.sin_port = htons(m_port);

	struct hostent *host;
	if((host = gethostbyaddr((char *)&servaddr.sin_addr, sizeof(servaddr.sin_addr), servaddr.sin_family)) != NULL)
		m_fqdn = host->h_name;
	else
		m_fqdn = m_ipAddr;

	// release the socket after program crash, avoid TIME_WAIT
	int reuse = 1;
	if(setsockopt(m_fd, SOL_SOCKET, SO_REUSEADDR, &reuse, sizeof(reuse)) == -1)
	{
		close(m_fd);
		return;
	}

	if( (bind(m_fd, (struct sockaddr *) &servaddr, sizeof(servaddr))) == -1)
	{
		close(m_fd);
		return;
	}
	m_isBound = true;

	if(listen(m_fd, LISTENQ) == -1)
	{
		close(m_fd);
		return;
	}

	// get current socket flags
	int flags;
	if ((flags=fcntl(m_fd, F_GETFL)) == -1)
		return;

	// set socket to non-blocking
	flags |= O_NDELAY;
	if (fcntl(m_fd, F_SETFL, flags) == -1)
		return;
}

bool ListenPort::isBound() const
{
	return m_isBound;
}
