##
## This module finds if the CLAW (C++ Library Absolutely Wonderful) is
## installed and determines where the include files and libraries
## are. It also determines what the name of the library is. This code
## sets the following variables:
##
## ------------------------------------------------------------------
##
## CLAW_FOUND             = system has CLAW
##
## CLAW_DEFINITIONS       = compiler definitions (-D)
##
## CLAW_INCLUDE_DIRECTORY = where to find headers "claw/claw.hpp"
##
## CLAW_LINK_DIRECTORY    = link directory
##
## CLAW_LIBRARIES         = claw libraries to link to
##
## ------------------------------------------------------------------
##
## Author Julien Jorge (10/2006)
##
## ------------------------------------------------------------------
## 
## status: 
## tested with:
##   -cmake 2.4, Linux (Ubuntu Edgy 6.10), gcc 4.0.3
##   -cmake 2.4, Windows XP, MinGW 5.0.3, gcc 3.3
## 

IF( CLAW_FOUND )
  SET(CLAW_FIND_QUIETLY TRUE)
ENDIF( CLAW_FOUND )

IF (UNIX) 
  ## ######################################################################
  ## 
  ## UNIX/Linux specific:
  ## 
  ## use backquoted claw-config to query and set flags and libs
  ## 
  
  STRING(REPLACE : " " SEARCH_PATH $ENV{PATH})

  # claw-config should be in your path anyhow
  FIND_PROGRAM(CMAKE_CLAW_CLAWCONFIG_EXECUTABLE claw-config ${SEARCH_PATH})
  
  # check wether claw-config was found:
  IF(CMAKE_CLAW_CLAWCONFIG_EXECUTABLE)    

    # set CXXFLAGS to be fed into CMAKE_DEFINITIONS by the user:
    EXEC_PROGRAM( ${CMAKE_CLAW_CLAWCONFIG_EXECUTABLE}
      ARGS "--cxxflags"
      OUTPUT_VARIABLE CLAW_DEFINITIONS )

    # evaluate claw-config output to separate linker flags and linkdirs for
    # rpath:
    EXEC_PROGRAM(${CMAKE_CLAW_CLAWCONFIG_EXECUTABLE}
      ARGS "--libs"
      OUTPUT_VARIABLE CLAW_CONFIG_LINK_DIRECTORY_WITH_PREFIX )
    
    ## extract linkdirs (-L)
    ## use regular expression to match wildcard equivalent "-L*<endchar>"
    ## with <endchar> is a space or a semicolon
    STRING( REGEX REPLACE
      "[-][L](.+)$" "\\1"
      CLAW_LINK_DIRECTORY
      "${CLAW_CONFIG_LINK_DIRECTORY_WITH_PREFIX}" )
    
    # set CXXFLAGS to be fed into CMAKE_DEFINITIONS by the user:
    EXEC_PROGRAM( ${CMAKE_CLAW_CLAWCONFIG_EXECUTABLE}
      ARGS "--include"
      OUTPUT_VARIABLE CLAW_CONFIG_INCLUDE_DIRECTORIES_WITH_PREFIX )

    ## extract include dirs (-I) for rpath
    ## use regular expression to match wildcard equivalent "-I*<endchar>"
    ## with <endchar> is a space or a semicolon
    STRING( REGEX REPLACE
      "[-][I](.+)$" "\\1"
      CLAW_INCLUDE_DIRECTORY
      "${CLAW_CONFIG_INCLUDE_DIRECTORIES_WITH_PREFIX}" )
    
    # we really need claw-config...
  ELSE(CMAKE_CLAW_CLAWCONFIG_EXECUTABLE)    
    MESSAGE(SEND_ERROR "Cannot find claw-config anywhere on the system. Please put the file into your path or specify it in CMAKE_CLAW_CLAWCONFIG_EXECUTABLE.")
  ENDIF(CMAKE_CLAW_CLAWCONFIG_EXECUTABLE)

  MARK_AS_ADVANCED(CMAKE_CLAW_CLAWCONFIG_EXECUTABLE)
ENDIF(UNIX)

IF(WIN32)
  FIND_PATH(CLAW_ROOT_DIRECTORY "include/claw/application.hpp"
    "C:/claw/"
    "D:/claw/"
    "C:/Program Files/claw/"
    "D:/Program Files/claw/"
    )

  SET( CLAW_INCLUDE_DIRECTORY "${CLAW_ROOT_DIRECTORY}/include" )
  SET( CLAW_LINK_DIRECTORY "${CLAW_ROOT_DIRECTORY}/lib" )

  FIND_PATH( CLAW_CONFIG_WIN_CXXFILE "win/definitions.txt"
    "${CLAW_ROOT_DIRECTORY}" )

  IF( CLAW_CONFIG_WIN_CXXFILE )
    FILE( READ "${CLAW_CONFIG_WIN_CXXFILE}/win/definitions.txt"
      CLAW_DEFINITIONS )
  ENDIF( CLAW_CONFIG_WIN_CXXFILE )
ENDIF(WIN32)

IF( CLAW_INCLUDE_DIRECTORY
    AND CLAW_LINK_DIRECTORY
    AND CLAW_DEFINITIONS
    )
  SET(CLAW_FOUND 1)
ELSE( CLAW_INCLUDE_DIRECTORY AND CLAW_LINK_DIRECTORY AND CLAW_DEFINITIONS )
  SET(CLAW_FOUND 0)
ENDIF( CLAW_INCLUDE_DIRECTORY AND CLAW_LINK_DIRECTORY AND CLAW_DEFINITIONS )

#-------------------------------------------------------------------------------
# claw_logger
FIND_LIBRARY(
  CLAW_LOGGER_LIBRARIES
  NAMES claw_logger
  PATHS ${CLAW_LINK_DIRECTORY} )

IF( NOT CLAW_LOGGER_LIBRARIES )
  MESSAGE( FATAL_ERROR "Could not find claw_logger library" )
  SET(CLAW_FOUND 0)
ENDIF( NOT CLAW_LOGGER_LIBRARIES )

#-------------------------------------------------------------------------------
# claw_application
FIND_LIBRARY(
  CLAW_APPLICATION_LIB
  NAMES claw_application
  PATHS ${CLAW_LINK_DIRECTORY} )

IF( NOT CLAW_APPLICATION_LIB )
  MESSAGE( FATAL_ERROR "Could not find claw_application library" )
  SET(CLAW_FOUND 0)
ELSE( NOT CLAW_APPLICATION_LIB )
  SET(
    CLAW_APPLICATION_LIBRARIES
    ${CLAW_APPLICATION_LIB}
    ${CLAW_LOGGER_LIBRARIES} )
ENDIF( NOT CLAW_APPLICATION_LIB )

#-------------------------------------------------------------------------------
# claw_configuration_file
FIND_LIBRARY(
  CLAW_CONFIGURATION_FILE_LIBRARIES
  NAMES claw_configuration_file
  PATHS ${CLAW_LINK_DIRECTORY} )

IF( NOT CLAW_CONFIGURATION_FILE_LIBRARIES )
  MESSAGE( FATAL_ERROR "Could not find claw_configuration_file library" )
  SET(CLAW_FOUND 0)
ENDIF( NOT CLAW_CONFIGURATION_FILE_LIBRARIES )

#-------------------------------------------------------------------------------
# claw_dynamic_library
FIND_LIBRARY(
  CLAW_DYNAMIC_LIBRARY_LIB
  NAMES claw_dynamic_library
  PATHS ${CLAW_LINK_DIRECTORY} )

IF( NOT CLAW_DYNAMIC_LIBRARY_LIB )
  MESSAGE( FATAL_ERROR "Could not find claw_dynamic_library library" )
  SET(CLAW_FOUND 0)
ELSE( NOT CLAW_DYNAMIC_LIBRARY_LIB )

  SET(
    CLAW_DYNAMIC_LIBRARY_LIBRARIES
    ${CLAW_DYNAMIC_LIBRARY_LIB} )

  IF(UNIX)
      SET(
        CLAW_DYNAMIC_LIBRARY_LIBRARIES
        ${CLAW_DYNAMIC_LIBRARY_LIBRARIES}
        ${CMAKE_DL_LIBS} )
  ENDIF(UNIX)

ENDIF( NOT CLAW_DYNAMIC_LIBRARY_LIB )

#-------------------------------------------------------------------------------
# claw_graphic
FIND_LIBRARY(
  CLAW_GRAPHIC_LIB
  NAMES claw_graphic
  PATHS ${CLAW_LINK_DIRECTORY} )

IF( NOT CLAW_GRAPHIC_LIB )
  MESSAGE( FATAL_ERROR "Could not find claw_graphic library" )
  SET(CLAW_FOUND 0)
ELSE( NOT CLAW_GRAPHIC_LIB )

  SET( CLAW_GRAPHIC_LIBRARIES ${CLAW_GRAPHIC_LIB} )

  STRING( REGEX MATCH "CLAW_JPEG_SUPPORT"
    CLAW_CONFIG_USES_JPEG
    "${CLAW_DEFINITIONS}" )
  STRING( REGEX MATCH "CLAW_PNG_SUPPORT"
    CLAW_CONFIG_USES_PNG
    "${CLAW_DEFINITIONS}" )

  IF ( CLAW_CONFIG_USES_JPEG )
    INCLUDE(FindJPEG)

    IF( JPEG_FOUND )
      SET(
        CLAW_GRAPHIC_LIBRARIES
        ${CLAW_GRAPHIC_LIBRARIES}
        ${JPEG_LIBRARIES} )
      SET( CLAW_GRAPHIC_INCLUDE_DIRECTORY
        ${CLAW_GRAPHIC_INCLUDE_DIRECTORY}
        ${JPEG_INCLUDE_DIR} )
      SET( CLAW_GRAPHIC_LINK_DIRECTORY
        ${CLAW_GRAPHIC_LINK_DIRECTORY}
        ${JPEG_LIBRARY} )
    ELSE( JPEG_FOUND )
      SET( CLAW_FOUND 0 )
    ENDIF( JPEG_FOUND )
  ENDIF( CLAW_CONFIG_USES_JPEG )

  IF ( CLAW_CONFIG_USES_PNG )
    INCLUDE(FindPNG)

    IF( PNG_FOUND )
      SET(
        CLAW_GRAPHIC_LIBRARIES
        ${CLAW_GRAPHIC_LIBRARIES}
        ${PNG_LIBRARIES} )
      SET( CLAW_GRAPHIC_INCLUDE_DIRECTORY
        ${CLAW_GRAPHIC_INCLUDE_DIRECTORY}
        ${PNG_INCLUDE_DIR} )
      SET( CLAW_GRAPHIC_LINK_DIRECTORY
        ${CLAW_GRAPHIC_LINK_DIRECTORY}
        ${PNG_LIBRARY} )
    ELSE( PNG_FOUND )
      SET( CLAW_FOUND 0 )
    ENDIF( PNG_FOUND )
  ENDIF( CLAW_CONFIG_USES_PNG )

ENDIF( NOT CLAW_GRAPHIC_LIB )

#-------------------------------------------------------------------------------
# claw_net
FIND_LIBRARY(
  CLAW_NET_LIB
  NAMES claw_net
  PATHS ${CLAW_LINK_DIRECTORY} )

IF( NOT CLAW_NET_LIB )
  MESSAGE( FATAL_ERROR "Could not find claw_net library" )
  SET(CLAW_FOUND 0)
ELSE( NOT CLAW_NET_LIB )

  SET(
    CLAW_NET_LIBRARIES
    ${CLAW_NET_LIB} )

  IF(WIN32)
    FIND_LIBRARY(
      CLAW_WSOCK32_LIB
      NAMES wsock32 )

    IF( NOT CLAW_WSOCK32_LIB )
      MESSAGE( FATAL_ERROR "Could not find wsock32 library" )
      SET(CLAW_FOUND 0)
    ELSE( NOT CLAW_WSOCK32_LIB )
      SET(
        CLAW_NET_LIBRARIES
        ${CLAW_NET_LIBRARIES}
        ${CLAW_WSOCK32_LIB} )
    ENDIF( NOT CLAW_WSOCK32_LIB )
  ENDIF(WIN32)

ENDIF( NOT CLAW_NET_LIB )

#-------------------------------------------------------------------------------
MARK_AS_ADVANCED(
  CLAW_DEFINITIONS
  CLAW_INCLUDE_DIRECTORY
  CLAW_LINK_DIRECTORY

  CLAW_APPLICATION_LIB
  CLAW_DYNAMIC_LIBRARY_LIB
  CLAW_GRAPHIC_LIB
  CLAW_NET_LIB

  CLAW_WSOCK32_LIB

  CLAW_CONFIG_USES_JPEG
  CLAW_CONFIG_USES_PNG

  CLAW_GRAPHIC_INCLUDE_DIRECTORY
  CLAW_GRAPHIC_LINK_DIRECTORY

  CLAW_LOGGER_LIBRARIES
  CLAW_APPLICATION_LIBRARIES
  CLAW_CONFIGURATION_FILE_LIBRARIES
  CLAW_DYNAMIC_LIBRARY_LIBRARIES
  CLAW_GRAPHIC_LIBRARIES
  CLAW_NET_LIBRARIES
  )

IF( CLAW_FOUND )
  IF( NOT CLAW_FIND_QUIETLY )
    MESSAGE(STATUS "Found Claw library")
  ENDIF( NOT CLAW_FIND_QUIETLY )
ELSE( CLAW_FOUND )
  MESSAGE(FATAL_ERROR "Could not find Claw library")
ENDIF( CLAW_FOUND )
