/** \file coveb-tree.h */

/*
 * coveb : An open source van Emde Boas Priority Queue library
 * by Rudi Cilibrasi (cilibrar@cilibrar.com)
 *
 * Distributed under the BSD license.  Copyright 2008 Rudi Cilibrasi.
 *
 * v 0.71
 *
 * Uses stratified trees inspired by van Emde Boas to hold a priority queue
 * of unsigned 32-bit integer values.  Standard operations are supported.
 * Speed and memory use are optimized for large numbers of elements and
 * high queue throughput.  The data structure is designed to take advantage
 * of multilevel caches of any size; this property is called being
 * "cache oblivious".
 *
 * Instantiation:
 *
 * struct coveb_bq *q;
 *
 *   q = coveb_bq_new();
 *
 * This allocates a new cache-oblivious bitwise priority queue. To free, use:
 *
 * Deallocation:
 *
 *   coveb_bq_free(q);
 *
 * Insertion of 32-bit unsigned integer values:
 *
 * void coveb_bq_insert(struct coveb_bq *q, uint32_t x);
 *
 *    places a new value into the queue, or does nothing if it's already there.
 *
 * Removal of values:
 *
 * uint32_t coveb_bq_extractmin(struct coveb_bq *q);
 *
 *    removes the smallest value (min) from the queue and returns it.  It
 *    is illegal to try to remove a value from an empty (size = 0) queue.
 *    This is the fastest way to remove elements.
 *
 * uint32_t coveb_bq_remove(struct coveb_bq *q, uint32_t x);
 *
 *    This generic function removes any specific value from the queue.
 *    It returns 0 if the object was already in the queue, and 1 if not.
 *
 * Queue status monitoring:
 *
 *  When the queue is completely full (containing four 'binary' gigasamples),
 *  it consumes about one gigabyte of memory.  This means that it uses about 2
 *  bits per sample on average when full.  This is only about twice as big as
 *  the literal storage cost of a typical queue state.  Because this size
 *  is not reportable as a 32-bit integer, the following function is
 *  available to detect the full queue condition:
 *
 *  uint32_t coveb_bq_addresses_full(const struct coveb_bq *q);
 *
 *  Returns 1 if all 2 ** 32 elements are in the queue, or 0 otherwise.
 *  When the queue is not full, the normal size function is exactly accurate:
 *
 * uint32_t coveb_bq_size(const struct coveb_bq *q);
 *
 *    returns the count of integers currently stored in the queue.  This
 *  can be used to see if an element can be extracted safely (size > 0).
 *  Note that in the special case where the queue is completely full and
 *  therefore the number of elements in the queue is one greater than the
 *  maximum value of an unsigned 32-bit integer, the size function will
 *  return instead one less than the actual number of elements.  To
 *  detect this uncommon condition use coveb_bq_addresses_full.
 *
 * Extrema functions
 *
 * uint32_t coveb_bq_max(const struct coveb_bq *q);
 *
 *  returns the largest integer stored in the queue without removal.
 *
 * uint32_t coveb_bq_min(const struct coveb_bq *q);
 *
 *  returns the smallest integer stored in the queue without removal.
 *
 * Rangefinding functions
 *
 * void coveb_bq_locate_smallest_not_less_than(const struct coveb_bq *q,
 * uint32_t incl_lower_bound, uint32_t *result_x, uint32_t *gotresult);
 *
 *   searches for an integer stored in the queue that is not less than
 *   incl_lower_bound.  If there is more than one such integer, the smallest
 *   is returned.  If an integer is found, *gotresult will be set to 1 and
 *   *result_x will hold the integer that was found.  If no such integer is
 *   found in the queue, then *gotresult with be set to 0 and *result_x will
 *   be untouched.  The queue is not modified in any case.
 *
 * void coveb_bq_successor(const struct coveb_bq *q,
 * uint32_t excl_lower_bound, uint32_t *result_x, uint32_t *gotresult);
 *
 *   This function scans upward for an integer that is greater than
 *   excl_lower_bound.  It is otherwise the same as the
 *   coveb_bq_locate_smallest_not_less_than function.
 */

#ifndef __COVEB_QUEUE_H
#define __COVEB_QUEUE_H

#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <stdlib.h>
#include <stdint.h>

/* Set to 0 to use plain malloc and free instead. */
#define USE_FAST_ALLOCATOR 1

/* Maximum number of size slots in custom allocator. Just 5 or so needed.  */
#define MAXBLOCKSIZES 32

/* The runtime static type used to look things up in the tree. */
#define COVEB_KEYTYPE uint32_t

/* Flag indicating tree has filled up completely with 2 ** 32 elements. */
#define MASK_TOTFULL    0x00000010

/* Used for maps */
#define MASK_TOPSIDE    0x00000020

/* Highest number of bits supported directly by a 32-bit integer bit array. */
#define MAXDIRECTK 5

/* Tree node structures. */
struct FastAllocFreeNode;
struct FastAllocController;
struct FastAllocArena;
struct vEBPNode;

struct FastAllocFreeNode {
  struct FastAllocFreeNode *nxt;
};

struct FastAllocArena {
  uint32_t blocksize, blocksperchunk, maxbpc;
  struct FastAllocFreeNode *freelist;
  struct FastAllocFreeNode *bigchunks;
};

struct FastAllocController {
  struct FastAllocArena faa[MAXBLOCKSIZES];
  int faa_tot;
};

struct vEBPNodeTypeIndicator {
  uint32_t _size;
  uint16_t _fullbitsize;
  uint16_t _flags;
  struct FastAllocController *_fac;
};

struct vEBMPNodeTypeIndicator {
  uint32_t _size;
  uint16_t _fullbitsize;
  uint8_t _flags;
  uint8_t _payload_bits_minus_one;
  struct FastAllocController *_fac;
};

struct coveb_bq {
  struct FastAllocController *fac;
  struct vEBPNode *tree;
};

struct coveb_mq {
  struct FastAllocController *fac;
  struct vEBMPNode *tree;
  uint32_t payload_bits;
};

struct vEBFullNode {
  COVEB_KEYTYPE _min, _max;
  struct vEBPNode *_top;
  struct vEBPNode **_bot;
};

/**
 * \struct vEBMPayloadStatic
 * \brief payload of up to 32 bytes
 *
 * This structure contains a payload ranging in size from 1 bit to 256 bits.
 *
 * There are 8 32-bit unsigned integers in this structure called _pl.
 */
struct vEBMPayloadStatic {
/** array of 8 unsigned 32-bit integers with bits starting in the LSB of entry with index 0 */
  uint32_t _pl[8]; /* Maximum 8 bits/byt * 8 words * 4 byt / word = 64 bytes */
};

struct vEBSingleNode {
  COVEB_KEYTYPE _x;
};

/**
 * \struct vEBMPDatum
 * \brief 32-bit unsigned integer with payload of up to 32 bytes
 *
 * This structure contains a 32-bit unsigned integer key field called _x.
 * It also contains an array of 8 such integers in _p._pl[0] through
 * _p._pl[7].  This structure contains a combination of an integer key and a
 * payload of 1 to 256 bits.  It holds the maximum space (32 bytes) as a static
 * type even if the coveb_mq queue actually carries only a smaller number of
 * bits for the payload.
 */
struct vEBMPDatum {
/** 32-bit unsigned integer key */
  COVEB_KEYTYPE _x;

 /** payload of 1 to 256 bits, as an array of 8 unsigned 32-bit integers */
  struct vEBMPayloadStatic _p;
};

struct vEBMSingleNode {
  struct vEBMPDatum _d;
};

struct vEBMFullNode {
  struct vEBMPDatum _min, _max;
  struct vEBMPNode *_top;
  struct vEBMPNode **_bot;
};

struct vEBSimpleBitTable {
  uint32_t _bits;
};

struct vEBPNode
 {
  struct vEBPNodeTypeIndicator t;
  union {
    struct vEBSingleNode sn;
    struct vEBFullNode fn;
    struct vEBSimpleBitTable sbt;
  };
};

struct vEBMPNode
 {
  struct vEBMPNodeTypeIndicator t;
  union {
    struct vEBMSingleNode sn;
    struct vEBMFullNode fn;
    struct vEBSimpleBitTable sbt;
  };
};

/* Advanced searching results for future expansion. */

struct vEBPSearchResult {
  COVEB_KEYTYPE x;
  uint32_t is_notfound;
  struct vEBPNode *sn;
  struct vEBPNode *atmin;
  struct vEBPNode *ondirectdata;
};

struct vEBMPSearchResult {
  COVEB_KEYTYPE x;
  uint32_t is_notfound;
  struct vEBMPNode *sn;
  struct vEBMPNode *atmin;
  struct vEBMPNode *ondirectdata;
};

/**************************************/
/* PRIMARY USER-LEVEL QUEUE FUNCTIONS */
/**************************************/

/* Allocation and freeing */
struct coveb_bq *coveb_bq_new(void);
struct coveb_bq *coveb_bq_clone(const struct coveb_bq *q);
void coveb_bq_free(struct coveb_bq *q);

/* Mutation */
void coveb_bq_insert(struct coveb_bq *q, uint32_t x);
uint32_t coveb_bq_extractmin(struct coveb_bq *q);
uint32_t coveb_bq_remove(struct coveb_bq *q, uint32_t m);

/* Inspection */
uint32_t coveb_bq_size(const struct coveb_bq *q);
uint32_t coveb_bq_contains(const struct coveb_bq *q, uint32_t x);
uint32_t coveb_bq_addresses_full(const struct coveb_bq *q);

/* Scanning */
void coveb_bq_locate_smallest_not_less_than(const struct coveb_bq *q, uint32_t incl_lower_bound, uint32_t *result_x, uint32_t *gotresult);
void coveb_bq_successor(const struct coveb_bq *q, uint32_t excl_lower_bound, uint32_t *result_x, uint32_t *gotresult);
uint32_t coveb_bq_max(const struct coveb_bq *q);
uint32_t coveb_bq_min(const struct coveb_bq *q);


/* Error reporting */
void coveb_failout(const char *msg);

/* The following functions create a mapping priority queue.
 * In a mapping priority queue, each entry has a payload attached.
 * The payload may be 1, 2, 4, 8, 16, 32, 64, 128, or 256 bits per entry.
 * Regardless of the payload size, the type passed is always the largest.
 *
 * All functions behave equivalently to the simple bitwise queue except
 * that payloads are carried for each entry.
 */
struct coveb_mq *coveb_mq_new(uint32_t payload_bits);
struct coveb_mq *coveb_mq_clone(const struct coveb_mq *q);
void coveb_mq_free(struct coveb_mq *q);
void coveb_mq_insert(struct coveb_mq *q, struct vEBMPDatum x);
struct vEBMPDatum coveb_mq_extractmin(struct coveb_mq *q);
struct vEBMPDatum coveb_mq_min(const struct coveb_mq *q);
struct vEBMPDatum coveb_mq_max(const struct coveb_mq *q);
void coveb_mq_locate_smallest_not_less_than(const struct coveb_mq *q, uint32_t incl_lower_bound, struct vEBMPDatum *result_x, uint32_t *gotresult);
struct vEBMPDatum fetch_from_bottom_table(struct vEBMPNode *v, uint32_t ind);
void store_to_bottom_table(struct vEBMPNode *v, struct vEBMPDatum d, uint32_t ind);
uint32_t coveb_mq_remove(struct coveb_mq *q, uint32_t x);
uint32_t coveb_mq_addresses_full(const struct coveb_mq *q);
uint32_t coveb_mq_size(const struct coveb_mq *q);
uint32_t coveb_mq_contains(const struct coveb_mq *q, uint32_t x);
void coveb_mq_successor(const struct coveb_mq *q, uint32_t num, struct vEBMPDatum *result_x, uint32_t *gotresult);


#endif
