/*

    DSP operations, header
    Copyright (C) 1998-2001 Jussi Laako

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/


#include <math.h>
#include <float.h>
#ifdef DSP_HAVE_FFTW
    #ifdef DSP_FLOAT_FFTW
        #include <sfftw.h>
        #include <srfftw.h>
    #else
        #include <dfftw.h>
        #include <drfftw.h>
    #endif
#endif

#include <Alloc.hh>

#include "dsp/DSPConfig.hh"
#include "dsp/Transform4.hh"
#include "dsp/Transform8.hh"
#include "dsp/TransformS.hh"
 

#ifndef DSPOP_HH
    #define DSPOP_HH

    // To help compilers common subexpression elimination optimization
    #ifdef __GNUG__
        #define CONSTFUNC       __const__
    #else
        #define CONSTFUNC
    #endif

    // Borland C++ support (it doesn't conform with ISO 9x standard)
    #ifdef __BCPLUSPLUS__
        #define fmax(x, y)      ((x >= y) ? x : y)
        #define fmin(x, y)      ((x <= y) ? x : y)
        #define fmaxf(x, y)     fmax(x, y)
        #define fminf(x, y)     fmin(x, y)
        #define fabsf(x)        fabs(x)
        #define powf(x, y)      pow(x, y)
        #define sqrtf(x)        sqrt(x)
        #define expf(x)         exp(x)
        #define log10f(x)       log10(x)
        #define sinf(x)         sin(x)
        #define cosf(x)         cos(x)
        #define atanf(x)        atan(x)
        #define atan2f(x, y)    atan2(x, y)
    #endif

    // Maximum iterations for modified zero order Bessel function of first kind
    #define DSP_MAXBESSEL       32L

    // Filename for FFTW wisdom
    #define DSP_WISDOM_FILE     "fftw.wisdom"

    // Complex datatypes, type prefix removed from members to prevent need
    // of source code changes when precision is changed.
    typedef struct _sSCplx
    {
        float R, I;
    } stSCplx, *stpSCplx;

    typedef struct _sDCplx
    {
        double R, I;
    } stDCplx, *stpDCplx;

    typedef struct _sSPolar
    {
        float M, P;
    } stSPolar, *stpSPolar;

    typedef struct _sDPolar
    {
        double M, P;
    } stDPolar, *stpDPolar;

    typedef union _uSCoord
    {
        stSCplx C;
        stSPolar P;
    } utSCoord, *utpSCoord;

    typedef union _uDCoord
    {
        stDCplx C;
        stDPolar P;
    } utDCoord, *utpDCoord;
    

    class clDSPAlloc : public clAlloc
    {
        public:
            clDSPAlloc () {}
            clDSPAlloc (long lAllocSize) : clAlloc(lAllocSize) {}
            operator stSCplx *()
                { return ((stpSCplx) vpPtr); }
            operator stDCplx *()
                { return ((stpDCplx) vpPtr); }
            operator stSPolar *()
                { return ((stpSPolar) vpPtr); }
            operator stDPolar *()
                { return ((stpDPolar) vpPtr); }
    };

    class clDSPOp
    {
            long lPrevSrcCount;
            long lPrevDestCount;
            float fPI;
            double dPI;
            // --- Dynamically allocated arrays
            // FIR
            long lFIRLength;
            clDSPAlloc FIRCoeff;
            clDSPAlloc FIRBuf;
            // FFT (and other transforms)
            bool bFFTInitialized;
            bool bRealTransform;
            long lFFTLength;
            float fFFTScale;
            double dFFTScale;
            long *lpSBitRevWork;
            long *lpDBitRevWork;
            float *fpCosSinTable;
            double *dpCosSinTable;
            clDSPAlloc SBitRevWork;
            clDSPAlloc DBitRevWork;
            clDSPAlloc SCosSinTable;
            clDSPAlloc DCosSinTable;
            #ifdef DSP_HAVE_FFTW
                fftw_real fftwrScale;
                fftw_plan fftwpPlan;
                fftw_plan fftwpIPlan;
                rfftw_plan rfftwpRPlan;
                rfftw_plan rfftwpIRPlan;
            #endif
            #ifdef DSP_USE_RADIX4
                clTransform4 Tfrm;
            #elif DSP_USE_RADIX8
                clTransform8 Tfrm;
            #else  // Split-radix
                clTransformS Tfrm;
            #endif
        protected:
            void Cart2Polar (float *, float *, float, float);
            void Cart2Polar (double *, double *, double, double);
            void Cart2Polar (float *, float *, const stpSCplx);
            void Cart2Polar (double *, double *, const stpDCplx);
            void Cart2Polar (stpSPolar, const stpSCplx);
            void Cart2Polar (stpDPolar, const stpDCplx);
            void Cart2Polar (utpSCoord);
            void Cart2Polar (utpDCoord);
            void Polar2Cart (float *, float *, float, float);
            void Polar2Cart (double *, double *, double, double);
            void Polar2Cart (stpSCplx, float, float);
            void Polar2Cart (stpDCplx, double, double);
            void Polar2Cart (stpSCplx, const stpSPolar);
            void Polar2Cart (stpDCplx, const stpDPolar);
            void Polar2Cart (utpSCoord);
            void Polar2Cart (utpDCoord);
            void CplxAdd (stpSCplx, const stpSCplx);
            void CplxAdd (stpDCplx, const stpDCplx);
            void CplxAdd (stpSCplx, const stpSCplx, const stpSCplx);
            void CplxAdd (stpDCplx, const stpDCplx, const stpDCplx);
            void CplxSub (stpSCplx, const stpSCplx);
            void CplxSub (stpDCplx, const stpDCplx);
            void CplxSub (stpSCplx, const stpSCplx, const stpSCplx);
            void CplxSub (stpDCplx, const stpDCplx, const stpDCplx);
            void CplxMul (stpSCplx, float);
            void CplxMul (stpDCplx, double);
            void CplxMul (stpSCplx, const stpSCplx);
            void CplxMul (stpDCplx, const stpDCplx);
            void CplxMul (stpSCplx, const stpSCplx, const stpSCplx);
            void CplxMul (stpDCplx, const stpDCplx, const stpDCplx);
            void CplxMulC (stpSCplx, const stpSCplx);
            void CplxMulC (stpDCplx, const stpDCplx);
            void CplxMulC (stpSCplx, const stpSCplx, const stpSCplx);
            void CplxMulC (stpDCplx, const stpDCplx, const stpDCplx);
            void CplxDiv (stpSCplx, const stpSCplx);
            void CplxDiv (stpDCplx, const stpDCplx);
            void CplxDiv (stpSCplx, const stpSCplx, const stpSCplx);
            void CplxDiv (stpDCplx, const stpDCplx, const stpDCplx);
            // In-place is allowed for following
            void CplxExp (stpSCplx, const stpSCplx);
            void CplxExp (stpDCplx, const stpDCplx);
            void CplxLog (stpSCplx, const stpSCplx);
            void CplxLog (stpDCplx, const stpDCplx);
            void CplxPow (stpSCplx, const stpSCplx, float);
            void CplxPow (stpDCplx, const stpDCplx, double);
            void CplxRoot (stpSCplx, const stpSCplx, float);
            void CplxRoot (stpDCplx, const stpDCplx, double);
            void CplxConj (stpSCplx spCplx)
                { spCplx->I = -(spCplx->I); }
            void CplxConj (stpDCplx spCplx)
                { spCplx->I = -(spCplx->I); }
            void CplxConj (stpSCplx, const stpSCplx);
            void CplxConj (stpDCplx, const stpDCplx);
            double Multiple (long);
            float ModZeroBessel (float);
            double ModZeroBessel (double);
        public:
            clDSPOp();
            ~clDSPOp();
            signed long Round (float);
            signed long Round (double);
            // Add single value to vector
            // (source&destination vector, source, length)
            void Add (float *, float, long);
            void Add (double *, double, long);
            void Add (stpSCplx, stSCplx, long);
            void Add (stpDCplx, stDCplx, long);
            // Add two vectors
            // (source&destination vector, source vector, length)
            void Add (float *, const float *, long);
            void Add (double *, const double *, long);
            void Add (stpSCplx, const stpSCplx, long);
            void Add (stpDCplx, const stpDCplx, long);
            // Add two vectors and return result in third
            // (destination vector, source vector, source vector, length)
            void Add (float *, const float *, const float *, long);
            void Add (double *, const double *, const double *, long);
            void Add (stpSCplx, const stpSCplx, const stpSCplx, long);
            void Add (stpDCplx, const stpDCplx, const stpDCplx, long);
            // Substract single value from vector
            // (source&destination vector, source, length)
            void Sub (float *, float, long);
            void Sub (double *, double, long);
            void Sub (stpSCplx, stSCplx, long);
            void Sub (stpDCplx, stDCplx, long);
            // Substract two vectors
            // (source&destination vector, source vector, length)
            void Sub (float *, const float *, long);
            void Sub (double *, const double *, long);
            void Sub (stpSCplx, const stpSCplx, long);
            void Sub (stpDCplx, const stpDCplx, long);
            // Substract two vectors and return result in third
            // (destination vector, source vector, source vector, length)
            void Sub (float *, const float *, const float *, long);
            void Sub (double *, const double *, const double *, long);
            void Sub (stpSCplx, const stpSCplx, const stpSCplx, long);
            void Sub (stpDCplx, const stpDCplx, const stpDCplx, long);
            // Multiply vector with single value in-place
            // (source&destination vector, value, length)
            void Mul (float *, float, long);
            void Mul (double *, double, long);
            void Mul (stpSCplx, float, long);
            void Mul (stpDCplx, double, long);
            void Mul (stpSCplx, stSCplx, long);
            void Mul (stpDCplx, stDCplx, long);
            // Multiply vector with single value out-of-place
            // (destination vector, source vector, value, length)
            void Mul (float *, const float *, float, long);
            void Mul (double *, const double *, double, long);
            // Multiply two vectors
            // (source&destination vector, source vector, length)
            void Mul (float *, const float *, long);
            void Mul (double *, const double *, long);
            void Mul (stpSCplx, const float *, long);
            void Mul (stpDCplx, const double *, long);
            void Mul (stpSCplx, const stpSCplx, long);
            void Mul (stpDCplx, const stpDCplx, long);
            // Multiply two vectors and return result in third
            // (destination vector, source vector, source vector, length)
            void Mul (float *, const float *, const float *, long);
            void Mul (double *, const double *, const double *, long);
            void Mul (stpSCplx, const stpSCplx, const stpSCplx, long);
            void Mul (stpDCplx, const stpDCplx, const stpDCplx, long);
            // Multiply vector with others complex conjugate
            void MulC (stpSCplx, const stpSCplx, long);
            void MulC (stpDCplx, const stpDCplx, long);
            void MulC (stpSCplx, const stpSCplx, const stpSCplx, long);
            void MulC (stpDCplx, const stpDCplx, const stpDCplx, long);
            // Multiply two vectors in-place with third vector
            // (source&destination vector, source&destination vector,
            // source vector, length)
            void Mul2 (float *, float *, const float *, long);
            void Mul2 (double *, double *, const double *, long);
            // Multiply two vectors out-of-place with third vector and
            // return results in separate two vectors
            // (destination vector, destination vector, source vector,
            // source vector, value vector, length)
            void Mul2 (float *, float *, const float *, const float *,
                const float *, long);
            void Mul2 (double *, double *, const double *, const double *,
                const double *, long);
            // Divide vector with single value
            // (source&destination vector, value, length)
            void Div (float *, float, long);
            void Div (double *, double, long);
            void Div (stpSCplx, stSCplx, long);
            void Div (stpDCplx, stDCplx, long);
            // Divide two vectors
            // (source&destination vector, source vector, length)
            void Div (float *, const float *, long);
            void Div (double *, const double *, long);
            void Div (stpSCplx, const stpSCplx, long);
            void Div (stpDCplx, const stpDCplx, long);
            // Divide two vectors and return result in third
            // (destination vector, source vector, source vector, length)
            void Div (float *, const float *, const float *, long);
            void Div (double *, const double *, const double *, long);
            void Div (stpSCplx, const stpSCplx, const stpSCplx, long);
            void Div (stpDCplx, const stpDCplx, const stpDCplx, long);
            // Calculate 1 / value in-place
            // (source&destination vector, length)
            void Div1x (float *, long);
            void Div1x (double *, long);
            // Calculate 1 / value out-of-place
            // (destination vector, source vector, length)
            void Div1x (float *, const float *, long);
            void Div1x (double *, const double *, long);
            // Get absolute value in-place
            // (source&destination vector, length)
            void Abs (float *, long);
            void Abs (double *, long);
            // Get absolute value out-of-place
            // (destination vector, source vector, length)
            void Abs (float *, const float *, long);
            void Abs (double *, const double *, long);
            // Calculate square-roots in-place
            // (source&destination vector, length)
            void Sqrt (float *, long);
            void Sqrt (double *, long);
            // Calculate squre-roots out-of-place
            // (destination vector, source vector, length)
            void Sqrt (float *, const float *, long);
            void Sqrt (double *, const double *, long);
            // Set all vector elements to zero
            // (destination vector, length)
            void Zero (float *, long);
            void Zero (double *, long);
            void Zero (stpSCplx, long);
            void Zero (stpDCplx, long);
            // Set all vector elements to specified value
            // (destination vector, length)
            void Set (float *, float, long);
            void Set (double *, double, long);
            void Set (stpSCplx, stSCplx, long);
            void Set (stpDCplx, stDCplx, long);
            // Set specified vector elements to specified value with
            // boundary check
            // (destination vector, value, start index, n elements, length)
            void Set (float *, float, long, long, long);
            void Set (double *, double, long, long, long);
            void Set (stpSCplx, stSCplx, long, long, long);
            void Set (stpDCplx, stDCplx, long, long, long);
            // Clip vector in-place to specified value
            // (source&destination vector, value, length)
            void Clip (float *, float, long);
            void Clip (double *, double, long);
            // Clip vector out-of-place to specified value
            // (destination vector, source vector, value, length)
            void Clip (float *, const float *, float, long);
            void Clip (double *, const double *, double, long);
            // Clip vector in-place to fit between specified values
            // (source&destination vector, min value, max value, length)
            void Clip (float *, float, float, long);
            void Clip (double *, double, double, long);
            // Clip vector out-of-place to fit between specified values
            // (destination vector, source vector, min, max, length)
            void Clip (float *, const float *, float, float, long);
            void Clip (double *, const double *, double, double, long);
            // Clip smallest values in vector to zero (in-place)
            // (source&destination vector, length)
            void ClipZero (float *, long);
            void ClipZero (double *, long);
            // Clip smallest values in vector to zero (out-of-place)
            // (destination vector, source vector, length)
            void ClipZero (float *, const float *, long);
            void ClipZero (double *, const double *, long);
            // Copy source vector to destination vector(s)
            // (destination vector, source vector, length)
            // (destination vector, destination vector, source vector, length)
            void Copy (float *, const float *, long);
            void Copy (double *, const double *, long);
            void Copy (stpSCplx, const stpSCplx, long);
            void Copy (stpDCplx, const stpDCplx, long);
            void Copy (float *, float *, const float *, long);
            void Copy (double *, double *, const double *, long);
            // Convolve two vectors (same length, finite) w/ and w/o lag
            // Note: circular convolution, result is not scaled
            // (source vector, source vector, length) = conv
            // (destination vector, source vector, source vector, length)
            float Convolve (const float *, const float *, long);
            double Convolve (const double *, const double *, long);
            void Convolve (float *, const float *, const float *, long);
            void Convolve (double *, const double *, const double *, long);
            // Correlate two vectors (same length, finite) w/ and w/o lag
            // Note: circular correlation, result is scaled
            // (source vector, source vector, length) = corr
            // (destination vector, source vector, source vector, length)
            float Correlate (const float *, const float *, long);
            double Correlate (const double *, const double *, long);
            void Correlate (float *, const float *, const float *, long);
            void Correlate (double *, const double *, const double *, long);
            // Autocorrelate vector
            // Note: circular, result is scaled
            // (source vector, length) = energy
            // (destination vector, source vector, length)
            float AutoCorrelate (const float *, long);
            double AutoCorrelate (const double *, long);
            void AutoCorrelate (float *, const float *, long);
            void AutoCorrelate (double *, const double *, long);
            // Dot product of two vectors
            // (source vector, source vector, length) = dotprod
            float DotProduct (const float *, const float *, long);
            double DotProduct (const double *, const double *, long);
            // Find minimum and maximum values of vector
            // (min, max, source vector, length)
            void MinMax (float *, float *, const float *, long);
            void MinMax (double *, double *, const double *, long);
            // Calculate mean of vector
            // (source vector, length) = mean
            float Mean (const float *, long);
            double Mean (const double *, long);
            // Calculate median of vector
            // (source vector, length) = median
            float Median (const float *, long);
            double Median (const double *, long);
            // Negate vector elements
            // (source&destination vector, length)
            // (destination vector, source vector, length)
            void Negate (float *, long);
            void Negate (double *, long);
            void Negate (float *, const float *, long);
            void Negate (double *, const double *, long);
            // Normalize vector elements
            // (source&destination vector, length)
            // (destination vector, source vector, length)
            void Normalize (float *, long);
            void Normalize (double *, long);
            void Normalize (float *, const float *, long);
            void Normalize (double *, const double *, long);
            // Product of vector elements
            // (source vector, length) = product
            float Product (const float *, long);
            double Product (const double *, long);
            // Reverse vector
            // (destination vector, source vector, length)
            void Reverse (float *, const float *, long);
            void Reverse (double *, const double *, long);
            // Scale vector to range [-1:1]
            // (source&destination vector, length)
            // (destination vector, source vector, length)
            void Scale (float *, long);
            void Scale (double *, long);
            void Scale (float *, const float *, long);
            void Scale (double *, const double *, long);
            // Scale vector to range [0:1]
            // (source&destination vector, length)
            // (destination vector, source vector, length)
            void Scale01 (float *, long);
            void Scale01 (double *, long);
            void Scale01 (float *, const float *, long);
            void Scale01 (double *, const double *, long);
            // Sort vector elements (in-place)
            // (source&destination vector, length)
            void Sort (float *, long);
            void Sort (double *, long);
            // Calculate standard deviation and mean of vector
            // (stddev, mean, source vector, length)
            void StdDev (float *, float *, const float *, long);
            void StdDev (double *, double *, const double *, long);
            // Calculate sum of vector elements
            // (source vector, length) = sum
            float Sum (const float *, long);
            double Sum (const double *, long);
            // Square vector in-place
            // (source&destination vector, length)
            void Square (float *, long);
            void Square (double *, long);
            // Square vector out-of-place
            // (destination vector, source vector, length)
            void Square (float *, const float *, long);
            void Square (double *, const double *, long);
            // Vector datatype conversion
            // Note: 12-bit data in LSBs, 24-bit data in MSBs
            // (destination vector, source vector, length)
            // (destination vector, source vector, length, 12bit)
            // (destination vector, source vector, length, 24bit)
            void Convert (float *, const unsigned char *, long);
            void Convert (float *, const signed short *, long, bool);
            void Convert (float *, const signed int *, long, bool);
            void Convert (float *, const double *, long);
            void Convert (double *, const unsigned char *, long);
            void Convert (double *, const signed short *, long, bool);
            void Convert (double *, const signed int *, long, bool);
            void Convert (double *, const float *, long);
            void Convert (unsigned char *, const float *, long);
            void Convert (unsigned char *, const double *, long);
            void Convert (signed short *, const float *, long, bool);
            void Convert (signed short *, const double *, long, bool);
            void Convert (signed int *, const float *, long, bool);
            void Convert (signed int *, const double *, long, bool);
            // --- More specialized operations ---
            // Convert cartesian to polar vectors (out-of-place)
            // (destination vector, source vector, length)
            void CartToPolar (float *, float *, const float *, 
                const float *, long);
            void CartToPolar (double *, double *, const double *, 
                const double *, long);
            void CartToPolar (float *, float *, const stpSCplx, long);
            void CartToPolar (double *, double *, const stpDCplx, long);
            void CartToPolar (stpSPolar, const stpSCplx, long);
            void CartToPolar (stpDPolar, const stpDCplx, long);
            // Convert cartesian to polar vectors (in-place)
            // (source&destination vector, length)
            void CartToPolar (utpSCoord, long);
            void CartToPolar (utpDCoord, long);
            // Convert polar to cartesian vectors (out-of-place)
            // (destination vector, source vector, length)
            void PolarToCart (float *, float *, const float *,
                const float *, long);
            void PolarToCart (double *, double *, const double *,
                const double *, long);
            void PolarToCart (stpSCplx, const float *, const float *, long);
            void PolarToCart (stpDCplx, const double *, const double *, long);
            void PolarToCart (stpSCplx, const stpSPolar, long);
            void PolarToCart (stpDCplx, const stpDPolar, long);
            // Convert cartesian to polar vectors (in-place)
            // (source&destination vector, length)
            void PolarToCart (utpSCoord, long);
            void PolarToCart (utpDCoord, long);
            // Calculate normalized cross correlation of two vectors w/o delay
            // (source vector, source vector, length) = crosscorr
            float CrossCorr (const float *, const float *, long);
            double CrossCorr (const double *, const double *, long);
            // Calculate normalized cross correlation of two vector w/ delay
            // in second (will be done for dataset length - delay)
            // (source vector, source vector, delay, length) = cross corr
            float DelCrossCorr (const float *, const float *, long, long);
            double DelCrossCorr (const double *, const double *, long, long);
            // Calculate normalized cross correlation of two vectors
            // with delays specified in vector 
            // (with maximum dataset for each delay)
            // (destination vector, source vector, source vector,
            //     source length, delay vector, delay length)
            void DelCrossCorr (float *, const float *, const float *, long,
                const long *, long);
            void DelCrossCorr (double *, const double *, const double *, long,
                const long *, long);
            // Calculates energy (square) of vector
            // (source vector, length) = power
            float Energy (const float *, long);
            double Energy (const double *, long);
            // Calculates magnitudes (linear) of vector
            // (destination vector, source vector, length)
            void Magnitude (float *, const stpSCplx, long);
            void Magnitude (double *, const stpDCplx, long);
            // Calculates powers (in dB) of vector
            // (destination vector, source vector, length)
            void Power (float *, const stpSCplx, long);
            void Power (double *, const stpDCplx, long);
            // Calculated phases (in rad) of vector
            // (destination vector, source vector, length)
            void Phase (float *, const stpSCplx, long);
            void Phase (double *, const stpDCplx, long);
            // Calculates powers (in dB) and phases (in rad) of vector
            // (power vector, phase vector, source vector, length)
            void PowerPhase (float *, float *, const stpSCplx, long);
            void PowerPhase (double *, double *, const stpDCplx, long);
            // Decimate vector without average
            // Note: This can be used in-place also
            // (destination vector, source vector, factor, length)
            void Decimate (float *, const float *, long, long);
            void Decimate (double *, const double *, long, long);
            // Decimate vector with average
            // (destination vector, source vector, factor, length)
            void DecimateAvg (float *, const float *, long, long);
            void DecimateAvg (double *, const double *, long, long);
            // Iterpolate vector without average (nulling)
            // (destination vector, source vector, factor, source length)
            void Interpolate (float *, const float *, long, long);
            void Interpolate (double *, const double *, long, long);
            // Interpolate vector with average (linear)
            // (destination vector, source vector, factor, source length)
            void InterpolateAvg (float *, const float *, long, long);
            void InterpolateAvg (double *, const double *, long, long);
            // Calculate RMS (root mean square) of vector
            // (source vector, length) = rms
            float RMS (const float *, long);
            double RMS (const double *, long);
            // Calculate variance and mean of vector
            // Note: variance and mean pointers can be NULL
            // (variance, mean, source vector, length) = variance
            float Variance (float *, float *, const float *, long);
            double Variance (double *, double *, const double *, long);
            // Find peak level of vector and return result in dB
            // (source vector, length) = peak level
            float PeakLevel (const float *, long);
            double PeakLevel (const double *, long);
            // Calculate Bartlett (triangle) window coefficients
            // (destination vector, length)
            void WinBartlett (float *, long);
            void WinBartlett (double *, long);
            // Calculate Blackman window coefficients
            // (destination vector, length)
            void WinBlackman (float *, long);
            void WinBlackman (double *, long);
            // Calculate Blackman-Harris window coefficients
            // (destination vector, length)
            void WinBlackmanHarris (float *, long);
            void WinBlackmanHarris (double *, long);
            // Calculate cosine tapered window coefficients
            // (destination vector, length)
            void WinCosTapered (float *, long);
            void WinCosTapered (double *, long);
            // Apply cosine tapered window to vector (in-place)
            // (source&destination vector, length)
            void WinCosTaperedA (float *, long);
            void WinCosTaperedA (double *, long);
            // Apply cosine tapered window to vector (out-of-place)
            // (destination vector, source vector, length)
            void WinCosTaperedA (float *, const float *, long);
            void WinCosTaperedA (double *, const double *, long);
            // Calculate exact Blackman window coefficients
            // (destination vector, length)
            void WinExactBlackman (float *, long);
            void WinExactBlackman (double *, long);
            // Calculate exponential window coefficients
            // (destination vector, final value, length)
            void WinExp (float *, float, long);
            void WinExp (double *, double, long);
            // Calculate flat top window coefficients
            // (destination vector, length)
            void WinFlatTop (float *, long);
            void WinFlatTop (double *, long);
            // Calculate generic cosine window coefficients
            // (destination vector, length, coefficient vector, coef length)
            void WinGenericCos (float *, long, const float *, long);
            void WinGenericCos (double *, long, const double *, long);
            // Calculate Hamming window coefficients
            // (destination vector, length)
            void WinHamming (float *, long);
            void WinHamming (double *, long);
            // Calculate Hanning (Hann?) window coefficients
            // (destination vector, length)
            void WinHanning (float *, long);
            void WinHanning (double *, long);
            // Calculate Kaiser window coefficients
            // (destination vector, beta, length)
            void WinKaiser (float *, float, long);
            void WinKaiser (double *, double, long);
            // Calculate Kaiser-Bessel window coefficients
            // (destination vector, alpha, length)
            void WinKaiserBessel (float *, float, long);
            void WinKaiserBessel (double *, double, long);
            // Calculate Tukey window coefficients
            // (destination vector, length)
            void WinTukey (float *, long);
            void WinTukey (double *, long);
            // Mix two channels interleaved in vector
            // (destination vector, source vector, destination length)
            void Mix (float *, const float *, long);
            void Mix (double *, const double *, long);
            // Mix two channels in separate vectors
            // (destination vector, source vector, source vector, length)
            void Mix (float *, const float *, const float *, long);
            void Mix (double *, const double *, const double *, long);
            // Mix n channels interleaved in vector
            // (destination vector, source vector, channel count, dest length)
            void Mix (float *, const float *, long, long);
            void Mix (double *, const double *, long, long);
            // Spatialize one channel to two channels by inversing others phase
            // (for playing mono sound on stereo headphones)
            // (destination vector, destination vector, source vector, length)
            void Spatialize (float *, float *, const float *, long);
            void Spatialize (double *, double *, const double *, long);
            // (interleaved destination vector, source vector, source length)
            void Spatialize (float *, const float *, long);
            void Spatialize (double *, const double *, long);
            // Extract channel n from N channel interleaved input
            // (destination vector, source vector, channel, channel count,
            // source length)
            void Extract (float *, const float *, long, long, long);
            void Extract (double *, const double *, long, long, long);
            // Pack channel n to N channel interleaved output
            // (destination vector, source vector, channel, channel count,
            // source length)
            void Pack (float *, const float *, long, long, long);
            void Pack (double *, const double *, long, long, long);
            // Rebuffer to different sized buffer
            // (destination vector, source vector, destination size,
            // source size) = 0 (no complete result), 1 (complete result),
            // 2 (multiple results)
            long ReBuffer (float *, const float *, long, long);
            long ReBuffer (double *, const double *, long, long);
            // Convert degrees to radians
            // (degrees) = radians
            double DegToRad (double dSource)
                { return ((dPI / 180.0) * dSource); }
            float DegToRad (float fSource)
                { return ((fPI / 180.0F) * fSource); }
            // Convert radians to degrees
            // (radians) = degrees
            double RadToDeg (double dSource)
                { return ((180.0 / dPI) * dSource); }
            float RadToDeg (float fSource)
                { return ((180.0F / fPI) * fSource); }
            // Convert Real-FFTW complex output to our Cplx
            // Note: destination length is source length / 2 + 1
            // (destination vector, source vector, source length)
            void FFTWConvert (stpSCplx, const float *, long);
            void FFTWConvert (stpDCplx, const float *, long);
            void FFTWConvert (stpSCplx, const double *, long);
            void FFTWConvert (stpDCplx, const double *, long);
            // Convert our Cplx to Real-FFTW complex input
            // (destination vector, source vector, destination length)
            void FFTWConvert (float *, const stpSCplx, long);
            void FFTWConvert (float *, const stpDCplx, long);
            void FFTWConvert (double *, const stpSCplx, long);
            void FFTWConvert (double *, const stpDCplx, long);
            // -*- Methods with dynamic data -*-
            // Allocate FIR filter
            // (coefficients vector, length)
            void FIRAllocate (const float *, long);
            void FIRAllocate (const double *, long);
            // Filter using FIR filter
            // (destination&source vector, length)
            void FIRFilter (float *, long);
            void FIRFilter (double *, long);
            // (destination vector, source vector, length)
            void FIRFilter (float *, const float *, long);
            void FIRFilter (double *, const double *, long);
            // Fast version, new data at source[FIRlength], must have
            // FIRlength size scratch pad at start of source vector
            // (destination vector, source vector, length)
            void FIRFilterF (float *, float *, long);
            void FIRFilterF (double *, double *, long);
            // Design FIR filter 
            // NOTE: Not implemented yet!
            // (type, cutoff frequency, samplerate)
            void FIRDesign () { return; }
            // Free FIR filter
            void FIRFree ();
            // Initialize FFT (and other transforms)
            // (transform size, real?)
            void FFTInitialize (long, bool);
            // Uninitialize FFT
            void FFTUninitialize ();
            // FFT half in-place, source vector is modified
            // Note! Output is FFT size / 2 + 1
            // (destination vector, source vector)
            void FFTi (stpSCplx, float *);
            void FFTi (stpDCplx, double *);
            // FFT out-of-place (uses FFTW if available)
            // Note! Output is FFT size / 2 + 1 for real input
            // (destination vector, source vector)
            void FFTo (stpSCplx, const float *);
            void FFTo (stpDCplx, const double *);
            void FFTo (stpSCplx, const stpSCplx);
            void FFTo (stpDCplx, const stpDCplx);
            // IFFT out-of-place (uses FFTW if available)
            // Note! Input is FFT size / 2 + 1 for real output
            // (destination vector, source vector)
            void IFFTo (float *, const stpSCplx);
            void IFFTo (double *, const stpDCplx);
            void IFFTo (stpSCplx, const stpSCplx);
            void IFFTo (stpDCplx, const stpDCplx);
    };

#endif

