/* Copyright (C) 1999 Hans Petter K. Jansson
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * You can contact the library's author by sending e-mail to <hpj@styx.net>.
 */

#ifndef _COMM_H
#define _COMM_H 1

#ifndef LIBFLUX_BUILD
#  include <flux/sock.h>
#  include <flux/tt.h>
#else
#  include "sock.h"
#  include "tt.h"
#endif /* LIBFLUX_BUILD */

/* FIXME: The comment format and language in this file is not entirely
 * consistent. */

#define BLOCK_NODES_MAX_DEF 256
#define BLOCK_SIZE_MAX_DEF  8192
#define BLOCK_DEPTH_MAX_DEF 128


enum comm_state
{
  COMM_STATE_NONE = 0, COMM_STATE_BLOCK_HEADER, COMM_STATE_NODE_HEADER,
  COMM_STATE_NODE_DATA
};


enum comm_order
{
  COMM_ORDER_DEPTH_FIRST = 0, COMM_ORDER_BREADTH_FIRST
};


typedef struct _comm
{
  SOCK *s;
  TT *queue_in, *queue_out;

  TT *root_in, *root_out;
  TT *node_in, *node_out;

  u32 block_in_size, block_out_size;
  u32 block_in_bytes_left, block_out_bytes_left;
  u32 node_in_bytes_left, node_out_bytes_left;
  u32 block_in_level, block_out_level;  /* Breadth-first traversal */

  u16 state_in : 2;  /* None, block header, node header, node data */
  u16 state_out : 2;

  u16 order_in : 1;
  u16 order_out : 1;
  
  u16 trans_in, trans_out;
  
  u32 block_nodes_max, block_size_max, block_depth_max;
  u16 trans_last;
}
COMM;

struct comm_block_header
{
  u32 bytes;
  u32 serial;
  u16 transaction;
  u16 flags;
};

struct comm_node_header
{
  u32 size;
  u32 children;
};

typedef struct _comm_reg
{
  int pad;
}
COMM_HD;


/* --- Structure of a communications handler registration entry --- */

typedef struct _comm_hd_reg
{
  char *root;  /* Points to a string which identifies a block's root node */
  unsigned short trans;  /* Associated transaction number - 0 for any */
  unsigned short flags;  /* Um, yeah. We might need these */
  int (*func)(SOCK *s, TT *t, unsigned short trans, unsigned char *root,
              unsigned long flags);  /* Points to callback */
}
COMM_HD_REG;


/* --- */

COMM *_comm_alloc();
COMM *comm_new();
COMM *comm_pipe_new();
COMM *comm_new_with_port(int port);
COMM *comm_accepting_new(int port);
COMM *comm_new_with_sock(SOCK *s);
void comm_del(COMM *c);

void comm_enqueue_to(COMM *c, TT *queue, TT *block, u16 trans, u16 flags);
TT *comm_dequeue_from(COMM *c, TT *queue, u16 *trans, u16 *flags);

#define comm_enqueue(c, block, trans, flags) \
  comm_enqueue_to(c, c->queue_out, block, trans, flags)

#define comm_dequeue(c, trans) \
  comm_dequeue_from(c, c->queue_in, trans, 0)

#define comm_queue_out_len(c) tt_count_children(((COMM *) c)->queue_out)
#define comm_queue_in_len(c) tt_count_children(((COMM *) c)->queue_in)

#define comm_buf_out_size(c) sock_buf_out_size(((COMM *) c)->s)
#define comm_buf_in_size(c) sock_buf_in_size(((COMM *) c)->s)

#define comm_get_sock(c) (((COMM *) c)->s)
#define comm_get_fd_read(c) sock_get_fd_read(comm_get_sock((COMM *) c))
#define comm_get_fd_write(c) sock_get_fd_write(comm_get_sock((COMM *) c))

#define comm_get_trans_in(c) (((COMM *) c)->trans_in)
#define comm_get_trans_out(c) (((COMM *) c)->trans_out)

#define comm_is_connecting(c) sock_is_connecting(((COMM *) c)->s)
#define comm_is_connected(c) sock_is_connected(((COMM *) c)->s)
#define comm_is_accepting(c) sock_is_accepting(((COMM *) c)->s)
#define comm_is_active(c) (comm_is_connected(c) | comm_is_accepting(c))

COMM *comm_accept(COMM *c);

void comm_limits(COMM *c, unsigned long nodes, unsigned long size, unsigned long depth);

int comm_pull(COMM *c, TT **ret_node);
TT *comm_pull_one(COMM *c);
#define comm_pull_many(c) comm_pull(c, 0);
int comm_push(COMM *c);

#if 0

/* --- Block handler functions --- */

/* comm_hd_reg_list() and comm_hd_unreg*() return the number of handlers
 * (un)registered.
 * 
 * comm_hd_reg() returns the COMM_HD handle of the handler registered.
 * 
 * comm_hd_loop() takes the flag COMM_ONE_HANDLER, in which case any incoming
 * block will only invoke one callback (the first that is encountered in the
 * internal list). Otherwise, all appropriate handlers are invoked in turn.
 * It returns 0 if the socket was closed by peer, or the return value of the
 * last callback if it was negative.
 * 
 * NOTE: A block handler list is private to a socket, and can not be shared
 * with others (you may use the same structures to initialize both, though). */

/* Communications loop. Triggers handlers based on incoming blocks. Frees the
 * received ttree before processing next block, so be sure to make copies if
 * you're going to use (parts of) it later */
int comm_hd_loop(SOCK *s, unsigned long flags);

/* Register a handler for transaction number/block type */
COMM_HD *comm_hd_reg(SOCK *s, unsigned short trans, unsigned char *root, unsigned short flags, int (*func)(SOCK *s, TT *t, unsigned short trans, unsigned char *root, unsigned long flags));

/* Unregister a handler */
int comm_hd_unreg(SOCK *s, COMM_HD *hd);

/* Unregister all handlers exactly matching transaction and block type */
int comm_hd_unreg_matches(SOCK *s, unsigned short trans, unsigned char *root);

/* Unregister all handlers handling the given trans, any block type */
int comm_hd_unreg_trans(SOCK *s, unsigned short trans);

/* Unregister all handlers handling the given block type, any trans */
int comm_hd_unreg_type(SOCK *s, unsigned char *root);

/* Register a number of handlers, given by a list of COMM_HD_REG, where the
 * last entry must have a 0 in the (*func) field */
int comm_hd_reg_list(SOCK *s, COMM_HD_REG *hd_list);

/* Unregister a number of handlers, given by a list of COMM_HD_REG, where the
 * last entry must have a 0 in the (*func) field */
int comm_hd_unreg_list(SOCK *s, COMM_HD_REG *hd_list);

/* Unregister all handlers on a socket */
int comm_hd_unreg_all(SOCK *s);


/* --- Transaction management function --- */

/* Allocate a new transaction number (for initiating a transaction). Returns
 * the transaction number immediately */
unsigned short comm_trans_new(SOCK *s);

/* Frees all handlers and structures associated with a transaction for a
 * given socket */
int comm_trans_end(SOCK *s, unsigned short trans);

/* Registers block handlers from a list, just like comm_hd_reg_list(), but
 * overrides the transaction field with a provided number */
int comm_trans_hd_reg_list(SOCK *s, unsigned short trans, COMM_HD_REG *hd_list);

/* Frees block handlers from a list, just like comm_hd_unreg_list(), but
 * overrides the transaction field with a provided number */
int comm_trans_hd_unreg_list(SOCK *s, unsigned short trans, COMM_HD_REG *hd_list);



/* --- Internal function prototypes --- */

int _comm_send_r(SOCK *s, TT *node);
TT *_comm_recv_r(SOCK *s, TT *parent, unsigned long n, unsigned int depth, int store);
int _comm_recv_mem(SOCK *s, TT *node, unsigned long bytes);
int _comm_recv_file(SOCK *s, TT *node, unsigned long bytes);
int _comm_recv_none(SOCK *s, TT *node, unsigned long bytes);
int _comm_hd_call(SOCK *s, TT *t, unsigned short trans, unsigned long flags);

#endif

#endif /* _COMM_H */
