/**
 * ===========================================
 * LibFonts : a free Java font reading library
 * ===========================================
 *
 * Project Info:  http://reporting.pentaho.org/libfonts/
 *
 * (C) Copyright 2006-2008, by Pentaho Corporation and Contributors.
 *
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation;
 * either version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * [Java is a trademark or registered trademark of Sun Microsystems, Inc.
 * in the United States and other countries.]
 *
 * ------------
 * Iso8859_1.java
 * ------------
 */
package org.pentaho.reporting.libraries.fonts.encoding.manual;

import java.util.Locale;

import org.pentaho.reporting.libraries.fonts.encoding.ByteBuffer;
import org.pentaho.reporting.libraries.fonts.encoding.CodePointBuffer;
import org.pentaho.reporting.libraries.fonts.encoding.Encoding;
import org.pentaho.reporting.libraries.fonts.encoding.EncodingErrorType;
import org.pentaho.reporting.libraries.fonts.encoding.EncodingException;

/**
 * This is a lucky case, as ISO-8859-1 can be transformed directly. There is no
 * lookup step needed.
 *
 * @author Thomas Morgner
 */
public final class Iso8859_1 implements Encoding
{
  public Iso8859_1()
  {
  }

  public String getName()
  {
    return "ISO-8859-1";
  }

  public String getName(final Locale locale)
  {
    return getName();
  }

  public boolean isUnicodeCharacterSupported(final int c)
  {
    return (c & 0xFFFFFF00) == 0;
  }

  /**
   * Encode, but ignore errors.
   *
   * @param text
   * @param buffer
   * @return
   */
  public ByteBuffer encode(final CodePointBuffer text, ByteBuffer buffer)
  {
    final int textLength = text.getLength();
    if (buffer == null)
    {
      buffer = new ByteBuffer(textLength);
    }
    else if (buffer.getLength() < textLength)
    {
      buffer.ensureSize(textLength);
    }

    final byte[] targetArray = buffer.getData();
    final int[] sourceArray = text.getData();

    int targetIdx = buffer.getOffset();
    final int endPos = text.getCursor();
    for (int i = text.getOffset(); i < endPos; i++)
    {
      final int sourceItem = sourceArray[i];
      if (isUnicodeCharacterSupported(sourceItem))
      {
        targetArray[targetIdx] = (byte) (sourceItem & 0xff);
        targetIdx += 1;
      }
    }

    buffer.setCursor(targetIdx);
    return buffer;
  }

  public CodePointBuffer decode(final ByteBuffer text, CodePointBuffer buffer)
  {
    final int textLength = text.getLength();
    if (buffer == null)
    {
      buffer = new CodePointBuffer(textLength);
    }
    else if (buffer.getLength() < textLength)
    {
      buffer.ensureSize(textLength);
    }

    final int[] targetArray = buffer.getData();
    final byte[] sourceArray = text.getData();

    int targetIdx = buffer.getOffset();
    final int endPos = text.getCursor();
    for (int i = text.getOffset(); i < endPos; i++)
    {
      targetArray[targetIdx] = (sourceArray[i] & 0xff);
      targetIdx += 1;
    }

    buffer.setCursor(targetIdx);
    return buffer;
  }

  public ByteBuffer encode(final CodePointBuffer text,
                           ByteBuffer buffer,
                           final EncodingErrorType errorHandling)
          throws EncodingException
  {
    final int textLength = text.getLength();
    if (buffer == null)
    {
      buffer = new ByteBuffer(textLength);
    }
    else if (buffer.getLength() < textLength)
    {
      buffer.ensureSize(textLength);
    }

    final byte[] targetArray = buffer.getData();
    final int[] sourceArray = text.getData();

    int targetIdx = buffer.getOffset();
    final int endPos = text.getCursor();
    for (int i = text.getOffset(); i < endPos; i++)
    {
      final int sourceItem = sourceArray[i];
      if (isUnicodeCharacterSupported(sourceItem))
      {
        targetArray[targetIdx] = (byte) (sourceItem & 0xff);
        targetIdx += 1;
      }
      else
      {
        if (EncodingErrorType.REPLACE.equals(errorHandling))
        {
          targetArray[targetIdx] = (byte) ('?' & 0xff);
          targetIdx += 1;
        }
        else if (EncodingErrorType.FAIL.equals(errorHandling))
        {
          throw new EncodingException();
        }
      }
    }

    buffer.setCursor(targetIdx);
    return buffer;
  }

  public CodePointBuffer decode(final ByteBuffer text,
                                CodePointBuffer buffer,
                                final EncodingErrorType errorHandling)
          throws EncodingException
  {
    final int textLength = text.getLength();
    if (buffer == null)
    {
      buffer = new CodePointBuffer(textLength);
    }
    else if (buffer.getLength() < textLength)
    {
      buffer.ensureSize(textLength);
    }

    final int[] targetArray = buffer.getData();
    final byte[] sourceArray = text.getData();

    int targetIdx = buffer.getOffset();
    final int endPos = text.getCursor();
    for (int i = text.getOffset(); i < endPos; i++)
    {
      targetArray[targetIdx] = (sourceArray[i] & 0xff);
      targetIdx += 1;
    }

    buffer.setCursor(targetIdx);
    return buffer;
  }

}
