/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 4 c-style: "K&R" -*- */

/*-----------------------------------------------------------------------------
   libgpiv - library for Particle Image Velocimetry

   Copyright (C) 2002, 2003, 2004, 2005
   Gerber van der Graaf <gerber_graaf@users.sourceforge.net>

   This file is part of libgpiv.

   Libgpiv is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  




-------------------------------------------------------------------------------
FILENAME:               dac.h
LIBRARY:                libgpiv:
EXTERNAL FUNCTIONS:
SOURCE:                 dac_par.c
                        gpiv_img_parameters_logic
                        gpiv_img_read_header
                        gpiv_img_fread_header
                        gpiv_img_check_header_read
                        gpiv_img_test_header
                        gpiv_img_print_header
                        gpiv_img_fprint_header
                        gpiv_img_fread_parameters
                        gpiv_img_fread_hdf5_parameters
                        gpiv_img_fwrite_hdf5_parameters
                        gpiv_img_fread_davis_parameters
			 
SOURCE:                 dac.c
                        gpiv_dac_get_camvar
                        gpiv_dac_free_camvar
                        gpiv_dac_openrtfs


LAST MODIFICATION DATE:  $Id: dac.h,v 1.6 2006/01/31 13:27:22 gerber Exp $
---------------------------------------------------------------------------- */

#ifndef __LIBGPIV_DAC_H__
#define __LIBGPIV_DAC_H__



#include <rtai.h>
#include <fcntl.h>

/*
 * Default values of GpivEvalPar
 */
#define GPIV_DACPAR_DEFAULT__TTIME_CAP 250
#define GPIV_DACPAR_DEFAULT__TTIME_LPW 0.02
#define GPIV_DACPAR_DEFAULT__TTIME_T2L 0.19
#define GPIV_DACPAR_DEFAULT__TTIME_DT 10
#define GPIV_DACPAR_DEFAULT__TTIME_MODE 2
#define GPIV_DACPAR_DEFAULT__TTIME_CYCLES 1
#define GPIV_DACPAR_DEFAULT__TTIME_INCR_DT 1

#define GPIV_DACPAR_DEFAULT__FNAME "gpiv_img"


#define GPIV_DACPAR_TTIME_CAP_MIN 100
#define GPIV_DACPAR_TTIME_LPW_MIN 0.02
#define GPIV_DACPAR_TTIME_T2L_MIN 0.19
#define GPIV_DACPAR_TTIME_DT_MIN 0.10
#define GPIV_DACPAR_TTIME_MODE_MIN 1
#define GPIV_DACPAR_TTIME_CYCLES_MIN 1
#define GPIV_DACPAR_TTIME_INCR_DT_MIN 0.01

#define GPIV_DACPAR_TTIME_CAP_MAX 1000
#define GPIV_DACPAR_TTIME_LPW_MAX 0.02
#define GPIV_DACPAR_TTIME_T2L_MAX 0.19
#define GPIV_DACPAR_TTIME_DT_MAX 100
#define GPIV_DACPAR_TTIME_MODE_MAX 6
#define GPIV_DACPAR_TTIME_CYCLES_MAX 1
#define GPIV_DACPAR_TTIME_INCR_DT_MAX 20

#define GPIV_NANO2MILI 0.001 * 0.001
#define GPIV_MILI2NANO 1000 * 1000



enum GpivFifo {
    GPIV_FIFO_TIMING_PARAMETER = 1,
    GPIV_FIFO_START_COMMAND,
    GPIV_FIFO_STOP_COMMAND,
    GPIV_FIFO_ERROR,
    GPIV_FIFO_JITTER,
};



enum GpivTimingMode {
    GPIV_TIMER_MODE__PERIODIC = 1,
    GPIV_TIMER_MODE__DURATION,
    GPIV_TIMER_MODE__ONE_SHOT_IRQ,
    GPIV_TIMER_MODE__TRIGGER_IRQ,
    GPIV_TIMER_MODE__INCREMENT,
    GPIV_TIMER_MODE__DOUBLE,
};
/*     GPIV_TIMER_MODE__SNAPSHOT, */
/*     GPIV_TIMER_MODE__EXIT, */



typedef struct __GpivTrigTime GpivTrigTime;
struct __GpivTrigTime {
    RTIME cam_acq_period;       /* camera acquisition period time in ns (period) */
    RTIME laser_trig_pw;        /* laser trigger pulse width in ns (tp) */
    RTIME time2laser;           /* time from laser trigger->Hi (tl)
				   until laser pulse: 0.19 ms in ns */
    RTIME dt;                   /* time between 2 laser exposures in ns (dt) */
    enum GpivTimingMode mode;   /* operating mode (indefinite/interrupt/definite) */
    gint cycles;                /* number of cycles (equal to number of images to be recorded?) */
    RTIME increment;            /* increment in dt (ms) */
};
  

  
typedef struct __GpivDacPar GpivDacPar;
struct __GpivDacPar {
    GpivTrigTime ttime;         /* camera and laser trigger timings */
    gchar fname[GPIV_MAX_CHARS];/* image file name */

    gboolean ttime__cam_acq_period_logic;
    gboolean ttime__laser_trig_pw_logic;
    gboolean ttime__time2laser_logic;
    gboolean ttime__dt_logic;
    gboolean ttime__mode_logic;
    gboolean ttime__cycles_logic;
    gboolean ttime__increment_logic;
    gboolean fname_logic;
};



typedef struct __GpivCamVar GpivCamVar;
struct __GpivCamVar {
    gint numNodes;
    gint numCameras;
    gint maxspeed;
    gint port;
    raw1394handle_t handle;
    nodeid_t *camera_nodes;

    dc1394_cameracapture *capture;
    dc1394_camerainfo *camera;
    dc1394_feature_info *feature_info;
    dc1394_feature_set *feature_set;
    dc1394_miscinfo *misc_info;
};



char *
gpiv_dac_get_camvar (GpivCamVar *cam_var
                   );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Get varaiables of connected cameras using firewire
 *
 * INPUTS:
 *
 * OUTPUTS:
 *     dac_var:         structure of camera variables
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



char *
gpiv_dac_free_camvar (GpivCamVar *cam_var
                      );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Free memory variables of connected cameras using firewire
 *
 * INPUTS:
 *
 * OUTPUTS:
 *     cam_var:         structure of camera variables
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



int 
gpiv_dac_openrtfs (int *init, 
                   int *trig, 
                   int *stop, 
                   int *error
                   );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Opens communication channels to camlasco-rtl kernel module
 *
 * INPUTS:
 *     init:            initialization or uploading of trigger parameters
 *     trig:            ?
 *     stop:            stop signal
 *     error:           error signal
 *
 * OUTPUTS:
 *     piv_dac_par_default:     structure of data aquisition parameters
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



void
gpiv_dac_default_parameters(GpivDacPar *dac_par_default,
                            gboolean force
                            );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Sets default GpivDacPar parameter values
 *
 * INPUTS:
 *
 * OUTPUTS:
 *     piv_dac_par_default:   structure of dac parameters
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



void 
gpiv_dac_read_parameters(FILE * fp_par, 
                         GpivDacPar * dac_par, 
                         int print_par
                         );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Read all GpivDacPar parameters
 *
 * INPUTS:
 *      fp_par:        file pointer to parameter file
 *      print_par:     flag to print parametrs to stdout
 *
 * OUTPUTS:
 *      dac_par:       Data Aquisition parameters
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



char *
gpiv_dac_check_parameters_read(GpivDacPar *dac_par,
                               GpivDacPar dac_par_default
                               );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Check out if all GpivDacPar parameters have been read
 *
 * INPUTS:
 *     dac_par_default:
 *
 * OUTPUTS:
 *     dac_par:         Data Aquisition parameters
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



void
gpiv_dac_parameters_logic(GpivDacPar * dac_par,
                          gboolean flag
                          );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Sets flags for _logic variables of GpivDacPar 
 *
 * INPUTS:
 *     dac_par_default:
 *
 * OUTPUTS:
 *     dac_par:         Data Aquisition parameters
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



char *
gpiv_dac_test_parameter(GpivDacPar * dac_par
                        );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Testing GpivDacPar parameters on valid values and initializing derived 
 *     variables 
 *
 * INPUTS:
 *     dac_par:      
 *     fname_logic:    
 *
 * OUTPUTS:
 *     dac_par:         Data Aquisition parameters
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 * --------------------------------------------------------------------------*/



void 
gpiv_dac_print_parameters(GpivDacPar dac_par
                          );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Prints GpivDacPar parameters to stdout
 *
 * INPUTS:
 *      dac_par:        Data Aquisition parameters
 *
 * OUTPUTS:
 *
 * RETURNS:
 *
 *---------------------------------------------------------------------------*/


void 
gpiv_dac_fprint_parameters(FILE * fp_par_out, 
                           GpivDacPar dac_par
                           );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Prints GpivDacPar parameters to fp_par_out
 *
 * INPUTS:
 *      fp_par_out:     
 *      dac_par:        Data Aquisition parameters
 *
 * OUTPUTS:
 *
 * RETURNS:
 *
 *---------------------------------------------------------------------------*/



#endif /*__LIBGPIV_DAC_H__ */
