/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 4 c-style: "K&R" -*- */

/*
   libgpiv - library for Particle Image Velocimetry

   Copyright (C) 2002, 2003, 2004 Gerber van der Graaf

   This file is part of libgpiv.

   Libgpiv is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  




-----------------------------------------------------------------------
FILENAME:               valid.h
LIBRARY:                libgpiv
EXTERNAL FUNCTIONS:
SOURCE:                 valid_par.c
                        gpiv_valid_parameters_logic
                        gpiv_valid_default_parameters
                        gpiv_valid_read_parameters
                        gpiv_valid_check_parameters_read
                        gpiv_valid_test_parameter
                        gpiv_valid_print_parameters
                        gpiv_valid_fprint_parameters
                        gpiv_valid_fread_hdf5_parameters
                        gpiv_valid_fwrite_hdf5_parameters

SOURCE:                 valid.c
                        gpiv_valid_residu
                        gpiv_valid_residu_stats
                        gpiv_valid_peaklck
                        gpiv_valid_errvec
                        gpiv_valid_gradient
                        gpiv_cumhisto_eqdatbin_gnuplot
                        gpiv_valid_threshold

LAST MODIFICATION DATE: $Id: valid.h,v 1.18 2006/01/31 13:27:22 gerber Exp $
 ------------------------------------------------------------------- */
#ifndef __LIBGPIV_VALID_H__
#define __LIBGPIV_VALID_H__

#define GPIV_GRADIENT_THRESHOLD 2.0     /* Default threshold of displacement gradient */
#define GPIV_THRESHOLD_DEFAULT  1.2     /* Default residu threshold */
#define GPIV_VALID_MAX_SWEEP 9      /* maximum number of validation sweeps,
                                  starting from zero */
#define GPIV_RESIDU_MAX_NORMMEDIAN 2.0  /* residu threshold if normalized median is used */
/*
 * Default and boundary values for GpivValidPar
 */
#define GPIV_VALIDPAR_DEFAULT__AUTO_THRESH 0
#define GPIV_VALIDPAR_DEFAULT__DATA_YIELD 0.95
#define GPIV_VALIDPAR_DEFAULT__RESIDU_MAX 2.0
#define GPIV_VALIDPAR_DEFAULT__RES_STATS 0
#define GPIV_VALIDPAR_DEFAULT__RESIDU_TYPE 2
#define GPIV_VALIDPAR_DEFAULT__SUBST_TYPE 2
#define GPIV_VALIDPAR_DEFAULT__HISTO_TYPE 0
#define GPIV_VALIDPAR_DEFAULT__ARRAY_LENGTH 3
#define GPIV_VALIDPAR_DEFAULT__NEIGHBORS 3

#define GPIV_VALIDPAR_MAX__NEIGHBORS 9

typedef struct __GpivLinRegData GpivLinRegData;
struct __GpivLinRegData {       /* Data structure of linear regression data */
    double c0;                  /* zero order constant */
    double c1;                  /* first order constant */
    double cov00;               /* variance of zero order constant */
    double cov01;               /* correlation bewteen first and 2nd const. */
    double cov11;               /* variance of zero order constant */
    double sumsq;               /* sum of squared residuals */
};



enum ResiduType {       /* Type of residu to detect outliers */
    GPIV_VALID_RESIDUTYPE__SNR,
    GPIV_VALID_RESIDUTYPE__MEDIAN,
    GPIV_VALID_RESIDUTYPE__NORMMEDIAN   /* normalized median, obtained from the ratio
                                           between datapoint under investigation and 
                                           median velocity (from 3x3 array, excluding
                                           current point) and median residu. 
                                           Ref. Westerweel, Scarano, Exp. in Fluids,
                                           39 (2005), pp1096-1100. */
};



enum SubstitutionType {
    GPIV_VALID_SUBSTYPE__NONE,
    GPIV_VALID_SUBSTYPE__L_MEAN,
    GPIV_VALID_SUBSTYPE__MEDIAN,
    GPIV_VALID_SUBSTYPE__COR_PEAK
};



typedef struct __GpivValidPar GpivValidPar;
struct __GpivValidPar {
    int auto_thresh;
    float data_yield;
    float residu_max;
    int neighbors;
    gboolean res_stats;
    enum ResiduType residu_type;
    enum SubstitutionType subst_type;
    int histo_type;
    
    gboolean auto_thresh_logic;
    gboolean data_yield_logic;
    gboolean residu_max_logic;
    gboolean neighbors_logic;
    gboolean res_stats_logic;
    gboolean residu_type_logic;
    gboolean subst_type_logic;
    gboolean histo_type_logic;
};



void
gpiv_valid_parameters_logic(GpivValidPar * piv_valid_par,
                            gboolean flag
                            );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Set flag for piv_valid_par_logic
 *
 * INPUTS:
 *      piv_valid_par:  struct of validation parameters
 *      flag:           boolean to define parameter flag
 *
 * OUTPUTS:
 *      piv_valid_par:  struct of validation parameters
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



void
gpiv_valid_default_parameters(GpivValidPar *piv_valid_par_default,
                              gboolean flag
                              );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Default parameter values
 *
 * INPUTS:
 *     force:                 flag to enforce parameters set to defaults
 *
 * OUTPUTS:
 *      piv_valid_par_default:  struct of validation parameters
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/




void 
gpiv_valid_read_parameters(FILE * fp_par,
			   GpivValidPar * piv_valid_par,
			   gboolean print_par
                           );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Reads validation parameters from file
 *
 * INPUTS:
 *      fp_par:         file pointer to file to be read
 *      print_par:      boolean to print parameters to stdout
 *
 * OUTPUTS:
 *      piv_valid_par:  struct of validation parameters
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



char *
gpiv_valid_check_parameters_read(GpivValidPar * piv_valid_par,
                                 GpivValidPar piv_valid_par_default
                                 );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Check out if all parameters have been read
 *
 * INPUTS:
 *      piv_valid_par:  struct of validation parameters
 *      piv_valid_par_default:  struct of validation parameters
 *
 * OUTPUTS:
 *      piv_valid_par:  struct of validation parameters
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/





void 
gpiv_valid_test_parameter(GpivValidPar * piv_valid_par
                          );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      tests if all validation parameters have been loaded
 *
 * INPUTS:
 *      piv_valid_par:  struct of validation parameters
 *
 * OUTPUTS:
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



void 
gpiv_valid_print_parameters(GpivValidPar piv_valid_par
                            );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      print all validation parameters to stdout
 *
 * INPUTS:
 *      piv_valid_par:  struct of validation parameters
 *
 * OUTPUTS:
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



void 
gpiv_valid_fprint_parameters(FILE * fp,
			     GpivValidPar piv_valid_par
                             );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      print all validation parameters to fp ASCII file
 *
 * INPUTS:
 *      fp:             file pointer to printed file
 *      piv_valid_par:  struct of validation parameters
 *
 * OUTPUTS:
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/




char *
gpiv_valid_fread_hdf5_parameters(char * fname, 
                                 GpivValidPar * piv_valid_par
                                 );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Reads validation parameters from hdf5 data file
 *
 * INPUTS:
 *      fname:          file name of data file
 *
 * OUTPUTS:
 *      piv_valid_par:  struct of validation parameters
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



char *
gpiv_valid_fwrite_hdf5_parameters(char *fname, 
                                  GpivValidPar * piv_valid_par
                                  );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Writes validation parameters to an existing hdf5 data file
 *
 * INPUTS:
 *      fname:          file name of data file
 *      piv_valid_par:  struct of validation parameters
 *
 * OUTPUTS:
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/




void 
gpiv_valid_residu(GpivPivData in_data, 
                  GpivPivData * out_data,
                  GpivValidPar piv_valid_par,
                  gboolean incl_point
                  );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Calculates residu values and apply to snr member of out_data structure
 *
 * PROTOTYPE LOCATATION:
 *     valid.h
 *
 * INPUTS:
 *     in_data:        piv dataset
 *     piv_valid_par:  validation parameters
 *     incl_point:     flag to include data point under investigation
 *
 * OUTPUTS:
 *     out_data:       piv dataset containing residu values in snr
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



char *
gpiv_valid_residu_stats(GpivLinRegData * linreg, 
                        GpivPivData * out_data, 
                        GpivBinData * klass
                        );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Calculates cumulative histogram of residus
 *
 * PROTOTYPE LOCATATION:
 *     valid.h
 *
 * INPUTS:
 *     linreg:         linear regression data structure
 *     klass:          histogram data
 *
 * OUTPUTS:
 *     out_data:       piv dataset containing residu values in snr
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



void 
gpiv_valid_peaklck(GpivPivData in_data,
		   GpivBinData * klass
                   );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Piv data evaluation function on peak-locking effects
 *
 * INPUTS:
 *      in_data:	input piv data
 *      klass:	        struct containing number of bins for histogram      
 *
 * OUTPUTS:
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



char *
gpiv_valid_errvec(GpivImagePar image_par,
                  GpivEvalPar piv_eval_par,
                  GpivValidPar piv_valid_par,
                  GpivPivData in_data,
                  GpivPivData * out_data,
                  GpivBinData * klass,
                  GpivLinRegData * linreg,
                  guint16 **img1,
                  guint16 **img2,
                  gboolean interrogate_valid
                  );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Piv data validation function on outliers
 *
 * INPUTS:
 *      image_par:      struct of image parameters
 *      piv_eval_par:   struct of piv image evaluation parameters
 *      piv_valid_par:  struct of validation parameters
 *      in_data:        input piv data
 *      klass:          histogram data
 *      linreg:         linear regression data
 *      exist_img:      boolean weather image has been loaded
 *      img1:           first image of PIV image pair
 *      img2:           second image of PIV image pair
 *      interrogate_valid:      validation during (iterative) interrogation 
 *                              process
 *
 * OUTPUTS:
 *      out_data:       output piv data
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



void 
gpiv_valid_gradient(GpivValidPar piv_valid_par, 
		    GpivEvalPar piv_eval_par, 
		    GpivPivData * piv_data
                    );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *       Searches vectors in a PIV data set that exceed the maximum gradient 
 *      (dUxdt/int_size > 0.05)
 *
 * INPUTS:
 *      piv_valid_par:  struct of validation parameters
 *      piv_eval_par:   struct of piv image evaluation parameters
 *      piv_data:       PIV data struct
 *
 * OUTPUTS:
 *      piv_data:       PIV dataset containing peak_no = -1 for exceeded maxima
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



void
gpiv_cumhisto_eqdatbin_gnuplot(char *fname_out, 
                               char *title, 
                               const char *GNUPLOT_DISPLAY_COLOR,
                               const int GNUPLOT_DISPLAY_SIZE,
                               GpivLinRegData * linreg
                               );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Plots cumulative histogram with an equal number of date 
 *      per bin of klass on screen with gnuplot
 *
 * INPUTS:
 *      fname_out:      output file name
 *      title:          title of plot
 *      GNUPLOT_DISPLAY_COLOR:  color of gnuplot display
 *      GNUPLOT_DISPLAY_SIZE:   size of gnuplot display
 *      linreg:         : linear regression data
 *
 * OUTPUTS:
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



float 
gpiv_valid_threshold(GpivEvalPar piv_eval_par,
                     GpivValidPar piv_valid_par,
                     GpivLinRegData linreg
                     );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Calculates threshold value (residu_max) from residus. 
 *      Will need int_size_1 from the GpivEvalPar struct
 *
 * INPUTS:
 *      piv_eval_par:   struct of piv image evaluation parameters
 *      piv_valid_par:  struct of validation parameters
 *      linreg:         : linear regression data
 *
 * OUTPUTS:
 *
 * RETURNS:
 *      threshold value
 *---------------------------------------------------------------------------*/





#endif /* __LIBGPIV_VALID_H__ */

