/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 4 c-style: "K&R" -*- */

/*
   libgpiv - library for Particle Image Velocimetry

   Copyright (C) 2002,2003, 2004 Gerber van der Graaf

   This file is part of libgpiv.

   Libgpiv is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  



------------------------------------------------------------------------------
FILENAME:                eval_par.c
LIBRARY:                 libgpiv
EXTERNAL FUNCTIONS:      
                         gpiv_piv_parameters_logic
                         gpiv_piv_default_parameters
                         gpiv_piv_read_parameters
                         gpiv_piv_check_parameters_read
		         gpiv_piv_test_parameters
                         gpiv_piv_print_parameters
                         gpiv_piv_fprint_parameters
                         gpiv_piv_cp_parameters
                         gpiv_piv_fread_hdf5_parameters
                         gpiv_piv_fwrite_hdf5_parameters

LAST MODIFICATION DATE:  $Id: eval_par.c,v 1.1 2006/01/13 10:05:54 gerber Exp $
--------------------------------------------------------------------- */

#include <gpiv.h>


static herr_t 
attr_info(hid_t loc_id, 
          const char *name, 
          GpivEvalPar * piv_eval_par
          )
/*-----------------------------------------------------------------------------
 * Operator function.
 */
{
    hid_t attribute_id, atype;
    herr_t status;



/*
 * Image piv parametes
 */
    if (strcmp(name, "int_geo") == 0) {
        attribute_id = H5Aopen_name(loc_id, name);
        status = H5Aread(attribute_id, H5T_NATIVE_INT, 
                         &piv_eval_par->int_geo); 
        status = H5Aclose(attribute_id); 
        piv_eval_par->int_geo_logic = TRUE;
    }


    if (strcmp(name, "col_start") == 0) {
        attribute_id = H5Aopen_name(loc_id, name);
        status = H5Aread(attribute_id, H5T_NATIVE_INT, 
                         &piv_eval_par->col_start); 
        status = H5Aclose(attribute_id); 
        piv_eval_par->col_start_logic = TRUE;
    }


    if (strcmp(name, "col_end") == 0) {
        attribute_id = H5Aopen_name(loc_id, name);
        status = H5Aread(attribute_id, H5T_NATIVE_INT, 
                         &piv_eval_par->col_end); 
        status = H5Aclose(attribute_id); 
        piv_eval_par->col_end_logic = TRUE;
    }


    if (strcmp(name, "row_start") == 0) {
        attribute_id = H5Aopen_name(loc_id, name);
        status = H5Aread(attribute_id, H5T_NATIVE_INT, 
                         &piv_eval_par->row_start); 
        status = H5Aclose(attribute_id); 
        piv_eval_par->row_start_logic = TRUE;
    }


    if (strcmp(name, "row_end") == 0) {
        attribute_id = H5Aopen_name(loc_id, name);
        status = H5Aread(attribute_id, H5T_NATIVE_INT, 
                         &piv_eval_par->row_end); 
        status = H5Aclose(attribute_id); 
        piv_eval_par->row_end_logic = TRUE;
    }

/*
 * line
 */
    if (strcmp(name, "int_line_row") == 0) {
        attribute_id = H5Aopen_name(loc_id, name);
        status = H5Aread(attribute_id, H5T_NATIVE_INT, 
                         &piv_eval_par->int_line_row); 
        status = H5Aclose(attribute_id); 
        piv_eval_par->int_line_row_logic = TRUE;
    }


    if (strcmp(name, "int_line_row_start") == 0) {
        attribute_id = H5Aopen_name(loc_id, name);
        status = H5Aread(attribute_id, H5T_NATIVE_INT, 
                         &piv_eval_par->int_line_row_start); 
        status = H5Aclose(attribute_id); 
        piv_eval_par->int_line_row_start_logic = TRUE;
    }


    if (strcmp(name, "int_line_row_end") == 0) {
        attribute_id = H5Aopen_name(loc_id, name);
        status = H5Aread(attribute_id, H5T_NATIVE_INT, 
                         &piv_eval_par->int_line_row_end); 
        status = H5Aclose(attribute_id); 
        piv_eval_par->int_line_row_end_logic = TRUE;
    }


    if (strcmp(name, "int_line_col") == 0) {
        attribute_id = H5Aopen_name(loc_id, name);
        status = H5Aread(attribute_id, H5T_NATIVE_INT, 
                         &piv_eval_par->int_line_col); 
        status = H5Aclose(attribute_id); 
        piv_eval_par->int_line_col_logic = TRUE;
    }


    if (strcmp(name, "int_line_col_start") == 0) {
        attribute_id = H5Aopen_name(loc_id, name);
        status = H5Aread(attribute_id, H5T_NATIVE_INT, 
                         &piv_eval_par->int_line_col_start); 
        status = H5Aclose(attribute_id); 
        piv_eval_par->int_line_col_start_logic = TRUE;
    }


    if (strcmp(name, "int_line_col_end") == 0) {
        attribute_id = H5Aopen_name(loc_id, name);
        status = H5Aread(attribute_id, H5T_NATIVE_INT, 
                         &piv_eval_par->int_line_col_end); 
        status = H5Aclose(attribute_id); 
        piv_eval_par->int_line_col_end_logic = TRUE;
    }


    if (strcmp(name, "autokey") == 0) {
        attribute_id = H5Aopen_name(loc_id, name);
        status = H5Aread(attribute_id, H5T_NATIVE_INT, 
                         &piv_eval_par->autokey); 
        status = H5Aclose(attribute_id); 
        piv_eval_par->autokey_logic = TRUE;
    }

/*
 * Interrogation
 */
    if (strcmp(name, "int_size_1") == 0) {
        attribute_id = H5Aopen_name(loc_id, name);
        status = H5Aread(attribute_id, H5T_NATIVE_INT, 
                         &piv_eval_par->int_size_1); 
        status = H5Aclose(attribute_id); 
        piv_eval_par->int_size_1_logic = TRUE;
    }


    if (strcmp(name, "int_size_2") == 0) {
        attribute_id = H5Aopen_name(loc_id, name);
        status = H5Aread(attribute_id, H5T_NATIVE_INT, 
                         &piv_eval_par->int_size_2); 
        status = H5Aclose(attribute_id); 
        piv_eval_par->int_size_2_logic = TRUE;
    }


    if (strcmp(name, "int_shift") == 0) {
        attribute_id = H5Aopen_name(loc_id, name);
        status = H5Aread(attribute_id, H5T_NATIVE_INT, 
                         &piv_eval_par->int_shift); 
        status = H5Aclose(attribute_id); 
        piv_eval_par->int_shift_logic = TRUE;
    }


    if (strcmp(name, "e_shift_col") == 0) {
        attribute_id = H5Aopen_name(loc_id, name);
        status = H5Aread(attribute_id, H5T_NATIVE_INT, 
                         &piv_eval_par->pre_shift_col); 
        status = H5Aclose(attribute_id); 
        piv_eval_par->pre_shift_col_logic = TRUE;
    }


    if (strcmp(name, "e_shift_row") == 0) {
        attribute_id = H5Aopen_name(loc_id, name);
        status = H5Aread(attribute_id, H5T_NATIVE_INT, 
                         &piv_eval_par->pre_shift_row); 
        status = H5Aclose(attribute_id); 
        piv_eval_par->pre_shift_row_logic = TRUE;
    }

/*
 * Sub-pixel interrogation
 */
    if (strcmp(name, "ifit") == 0) {
        attribute_id = H5Aopen_name(loc_id, name);
        status = H5Aread(attribute_id, H5T_NATIVE_INT, 
                         &piv_eval_par->ifit); 
        status = H5Aclose(attribute_id); 
        piv_eval_par->ifit_logic = TRUE;
    }


    if (strcmp(name, "peak") == 0) {
        attribute_id = H5Aopen_name(loc_id, name);
        status = H5Aread(attribute_id, H5T_NATIVE_INT, 
                         &piv_eval_par->peak); 
        status = H5Aclose(attribute_id); 
        piv_eval_par->peak_logic = TRUE;
    }


    if (strcmp(name, "int_scheme") == 0) {
        attribute_id = H5Aopen_name(loc_id, name);
        status = H5Aread(attribute_id, H5T_NATIVE_INT, 
                         &piv_eval_par->int_scheme); 
        status = H5Aclose(attribute_id); 
        piv_eval_par->int_scheme_logic = TRUE;
    }


    if (strcmp(name, "print_piv") == 0) {
        attribute_id = H5Aopen_name(loc_id, name);
        status = H5Aread(attribute_id, H5T_NATIVE_INT, 
                         &piv_eval_par->print_piv); 
        status = H5Aclose(attribute_id); 
        piv_eval_par->print_piv_logic = TRUE;
    }


    if (strcmp(name, "old_piv") == 0) {
        attribute_id = H5Aopen_name(loc_id, name);
        status = H5Aread(attribute_id, H5T_NATIVE_INT, 
                         &piv_eval_par->old_piv); 
        status = H5Aclose(attribute_id); 
        piv_eval_par->old_piv_logic = TRUE;
    }


    return 0;
}



/*
 * Public functions
 */

void
gpiv_piv_parameters_logic(GpivEvalPar * piv_eval_par,
                           gboolean flag
                           )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Set flags for all piv_eval_par_logic
 *
 * INPUTS:
 *      flag:           true (1) or false (0)
 *
 * OUTPUTS:
 *      piv_eval_par:   PIV evaluation parameters
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/
{
    piv_eval_par->autokey_logic = flag;
    piv_eval_par->col_start_logic = flag;
    piv_eval_par->col_end_logic = flag;
    piv_eval_par->int_geo_logic = flag;
    piv_eval_par->ifit_logic = flag;
    piv_eval_par->int_line_row_logic = flag;
    piv_eval_par->int_line_row_start_logic = flag;
    piv_eval_par->int_line_row_end_logic = flag;
    piv_eval_par->int_line_col_logic = flag;
    piv_eval_par->int_line_col_start_logic = flag;
    piv_eval_par->int_line_col_end_logic = flag;
    piv_eval_par->int_point_col_logic = flag;
    piv_eval_par->int_point_row_logic = flag;
    piv_eval_par->int_size_1_logic = flag;
    piv_eval_par->int_size_2_logic = flag;
    piv_eval_par->int_shift_logic = flag;
    piv_eval_par->old_piv_logic = flag;
    piv_eval_par->pre_shift_col_logic = flag;
    piv_eval_par->pre_shift_row_logic = flag; 
    piv_eval_par->peak_logic = flag;
    piv_eval_par->print_par_logic = flag;
    piv_eval_par->print_piv_logic = flag;
    piv_eval_par->print_cov_logic = flag;
    piv_eval_par->row_start_logic = flag;
    piv_eval_par->row_end_logic = flag;
    piv_eval_par->ad_int_logic = flag;
    piv_eval_par->int_scheme_logic = flag;

}



void
gpiv_piv_default_parameters(GpivEvalPar *piv_eval_par_default,
                            gboolean force
                            )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Sets default parameter values
 *
 * INPUTS:
 *     force:                 flag to enforce parameters set to defaults
 *
 * OUTPUTS:
 *     piv_eval_par_default:   structure of piv evaluation parameters
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    if (!piv_eval_par_default->autokey_logic || force)
        piv_eval_par_default->autokey = GPIV_EVALPAR_DEFAULT__AUTOKEY;
    if (!piv_eval_par_default->int_geo_logic || force)
        piv_eval_par_default->int_geo = GPIV_EVALPAR_DEFAULT__INT_GEO;
    if (!piv_eval_par_default->col_start_logic || force)
        piv_eval_par_default->col_start = GPIV_EVALPAR_DEFAULT__COL_START;
    if (!piv_eval_par_default->col_end_logic || force)
        piv_eval_par_default->col_end = GPIV_EVALPAR_DEFAULT__COL_END;
    if (!piv_eval_par_default->row_start_logic || force)
        piv_eval_par_default->row_start = GPIV_EVALPAR_DEFAULT__ROW_START;
    if (!piv_eval_par_default->row_end_logic || force)
        piv_eval_par_default->row_end = GPIV_EVALPAR_DEFAULT__ROW_END;
    if (!piv_eval_par_default->int_line_row_start_logic || force)
        piv_eval_par_default->int_line_row_start = GPIV_EVALPAR_DEFAULT__INT_LINE_ROW_START;
    if (!piv_eval_par_default->int_line_row_end_logic || force)
        piv_eval_par_default->int_line_row_end = GPIV_EVALPAR_DEFAULT__INT_LINE_ROW_END;
    if (!piv_eval_par_default->int_line_col_start_logic || force)
        piv_eval_par_default->int_line_col_start = GPIV_EVALPAR_DEFAULT__INT_LINE_COL_START;
    if (!piv_eval_par_default->int_line_col_end_logic || force)
        piv_eval_par_default->int_line_col_end = GPIV_EVALPAR_DEFAULT__INT_LINE_COL_END;
    if (!piv_eval_par_default->int_point_col_logic || force)
        piv_eval_par_default->int_point_col = GPIV_EVALPAR_DEFAULT__INT_POINT_COL;
    if (!piv_eval_par_default->int_point_row_logic || force)
        piv_eval_par_default->int_point_row = GPIV_EVALPAR_DEFAULT__INT_POINT_ROW;
    if (!piv_eval_par_default->int_size_1_logic || force)
        piv_eval_par_default->int_size_1 = GPIV_EVALPAR_DEFAULT__INT_SIZE_1;
    if (!piv_eval_par_default->int_size_2_logic || force)
        piv_eval_par_default->int_size_2 = GPIV_EVALPAR_DEFAULT__INT_SIZE_1;
    if (!piv_eval_par_default->int_shift_logic || force)
        piv_eval_par_default->int_shift = GPIV_EVALPAR_DEFAULT__INT_SHIFT;
    if (!piv_eval_par_default->pre_shift_col_logic || force)
        piv_eval_par_default->pre_shift_col = GPIV_EVALPAR_DEFAULT__PRE_SHIFT_COL;
    if (!piv_eval_par_default->pre_shift_row_logic || force)
        piv_eval_par_default->pre_shift_row = GPIV_EVALPAR_DEFAULT__PRE_SHIFT_ROW;
    if (!piv_eval_par_default->ad_int_logic || force)
        piv_eval_par_default->ad_int = GPIV_EVALPAR_DEFAULT__AD_INT;
    if (!piv_eval_par_default->ifit_logic || force)
        piv_eval_par_default->ifit = GPIV_EVALPAR_DEFAULT__IFIT;
    if (!piv_eval_par_default->peak_logic || force)
        piv_eval_par_default->peak = GPIV_EVALPAR_DEFAULT__PEAK;
    if (!piv_eval_par_default->int_scheme_logic || force)
        piv_eval_par_default->int_scheme = GPIV_EVALPAR_DEFAULT__INT_SCHEME;
    if (!piv_eval_par_default->old_piv_logic || force)
        piv_eval_par_default->old_piv = GPIV_EVALPAR_DEFAULT__OLD_PIV;


    gpiv_piv_parameters_logic(piv_eval_par_default, TRUE);
}



void 
gpiv_piv_read_parameters(FILE * fp_par, 
                         GpivEvalPar * piv_eval_par, 
                         int print_par
                         )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Read all parameters for PIV evaluation
 *
 * INPUTS:
 *      fp_par:        file pointer to parameter file
 *      print_par:     flag to print parametrs to stdout
 *
 * OUTPUTS:
 *     piv_eval_par
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    char line[GPIV_MAX_CHARS], par_name[GPIV_MAX_CHARS];

    while (fgets(line, GPIV_MAX_CHARS, fp_par) != NULL) {
        if (line[0] != '#' && line[0] != '\n' && line[0] != ' ' 
            && line[0] != '\t') {
            sscanf(line,"%s", par_name);

            if (piv_eval_par->int_geo_logic == FALSE) {
                piv_eval_par->int_geo_logic = 
                    gpiv_scan_iph(GPIV_EVAL_PAR_KEY, line, par_name, 
                                  "Int_geo:",
                                  &piv_eval_par->int_geo, 
                                  print_par, 1);
            }

            if (piv_eval_par->col_start_logic == FALSE) {
                piv_eval_par->col_start_logic = 
                    gpiv_scan_iph(GPIV_EVAL_PAR_KEY, line, par_name, 
                                  "Col_start:",
                                  &piv_eval_par->col_start, print_par, 1);
            }

            if (piv_eval_par->col_end_logic == FALSE) {
                piv_eval_par->col_end_logic =  
                    gpiv_scan_iph(GPIV_EVAL_PAR_KEY, line, par_name, 
                                  "Col_end:", 
                                  &piv_eval_par->col_end, print_par, 1);
            }

            if (piv_eval_par->row_start_logic == FALSE) {
                piv_eval_par->row_start_logic = 
                    gpiv_scan_iph(GPIV_EVAL_PAR_KEY, line, par_name, 
                                  "Row_start:", 
                                  &piv_eval_par->row_start, print_par, 1);
            }

            if (piv_eval_par->row_end_logic == FALSE) {
                piv_eval_par->row_end_logic = 
                    gpiv_scan_iph(GPIV_EVAL_PAR_KEY, line, par_name, 
                                  "Row_end:", 
                                  &piv_eval_par->row_end, print_par, 1);
            }

            if (piv_eval_par->int_line_row_logic == FALSE) {
                piv_eval_par->int_line_row_logic = 
                    gpiv_scan_iph(GPIV_EVAL_PAR_KEY, line, par_name, 
                                  "Int_line_row:",
                                  &piv_eval_par->int_line_row, 
                                  print_par, 1);
            }

            if (piv_eval_par->int_line_row_start_logic == FALSE) {
                piv_eval_par->int_line_row_start_logic = 
                    gpiv_scan_iph(GPIV_EVAL_PAR_KEY, line, par_name, 
                                  "Int_line_row_start:",
                                  &piv_eval_par->int_line_row_start, 
                                  print_par, 1);
            }

            if (piv_eval_par->int_line_row_end_logic == FALSE) {
                piv_eval_par->int_line_row_end_logic = 
                    gpiv_scan_iph(GPIV_EVAL_PAR_KEY, line, par_name, 
                              "Int_line_row_end:",
                              &piv_eval_par->int_line_row_end, 
                                  print_par, 1);
            }

            if (piv_eval_par->int_line_col_logic == FALSE) {
                piv_eval_par->int_line_col_logic = 
                    gpiv_scan_iph(GPIV_EVAL_PAR_KEY, line, par_name, 
                                  "Int_line_col:",
                                  &piv_eval_par->int_line_col, 
                                  print_par, 1);
            }

            if (piv_eval_par->int_line_col_start_logic == FALSE) {
                piv_eval_par->int_line_col_start_logic = 
                    gpiv_scan_iph(GPIV_EVAL_PAR_KEY, line, par_name, 
                                  "Int_line_col_start:",
                                  &piv_eval_par->int_line_col_start, 
                                  print_par, 1);
            }

            if (piv_eval_par->int_line_col_end_logic == FALSE) {
                piv_eval_par->int_line_col_end_logic = 
                    gpiv_scan_iph(GPIV_EVAL_PAR_KEY, line, par_name, 
                                  "Int_line_col_end:",
                                  &piv_eval_par->int_line_col_end, 
                                  print_par, 1);
            }

            if (piv_eval_par->int_point_col_logic == FALSE) {
                piv_eval_par->int_point_col_logic = 
                    gpiv_scan_iph(GPIV_EVAL_PAR_KEY, line, par_name, 
                                  "Int_point_col:",
                                  &piv_eval_par->int_point_col, print_par, 1);
            }

            if (piv_eval_par->int_point_row_logic == FALSE) {
                piv_eval_par->int_point_row_logic = 
                    gpiv_scan_iph(GPIV_EVAL_PAR_KEY, line, par_name, 
                                  "Int_point_row:",
                                  &piv_eval_par->int_point_row, print_par, 1);
            }



/*             if (piv_eval_par->ad_int_logic == FALSE) { */
/*                 piv_eval_par->ad_int_logic =  */
/*                     gpiv_scan_iph(GPIV_EVAL_PAR_KEY, line, par_name,  */
/*                                   "Ad_int:",  */
/*                                   &piv_eval_par->ad_int, print_par, 1); */
/*             } */

            if (piv_eval_par->int_size_1_logic == FALSE) {
                piv_eval_par->int_size_1_logic = 
                    gpiv_scan_iph(GPIV_EVAL_PAR_KEY, line, par_name, 
                                  "Int_size_1:", 
                                  &piv_eval_par->int_size_1, print_par, 1);
            }

            if (piv_eval_par->int_size_2_logic == FALSE) {
                piv_eval_par->int_size_2_logic = 
                    gpiv_scan_iph(GPIV_EVAL_PAR_KEY, line, par_name, 
                                  "Int_size_2:", 
                                  &piv_eval_par->int_size_2, print_par, 1);
            }

            if (piv_eval_par->int_shift_logic == FALSE) {
                piv_eval_par->int_shift_logic = 
                    gpiv_scan_iph(GPIV_EVAL_PAR_KEY, line, par_name, 
                                  "Int_shift:", 
                                  &piv_eval_par->int_shift, print_par, 1);
            }

            if (piv_eval_par->pre_shift_col_logic == FALSE) {
                piv_eval_par->pre_shift_col_logic = 
                    gpiv_scan_iph(GPIV_EVAL_PAR_KEY, line, par_name, 
                                  "Pre_shift_col:", 
                                  &piv_eval_par->pre_shift_col, print_par, 
                                  1);
            }

            if (piv_eval_par->pre_shift_row_logic == FALSE) {
                piv_eval_par->pre_shift_row_logic = 
                    gpiv_scan_iph(GPIV_EVAL_PAR_KEY, line, par_name, 
                                  "Pre_shift_row:", 
                                  &piv_eval_par->pre_shift_row, print_par, 
                                  1);
            }

            if (piv_eval_par->int_scheme_logic == FALSE) {
                piv_eval_par->int_scheme_logic = 
                    gpiv_scan_iph(GPIV_EVAL_PAR_KEY, line, par_name, 
                                  "Int_scheme:", 
                                  &piv_eval_par->int_scheme, print_par, 1);
            }

            if (piv_eval_par->peak_logic == FALSE) {
                piv_eval_par->peak_logic = 
                    gpiv_scan_iph(GPIV_EVAL_PAR_KEY, line, par_name, 
                                  "Peak:", 
                                  &piv_eval_par->peak, print_par, 1);
            }
            
            if (piv_eval_par->ifit_logic == FALSE) {
                piv_eval_par->ifit_logic = 
                    gpiv_scan_iph(GPIV_EVAL_PAR_KEY, line, par_name, 
                                  "Ifit:", 
                                  &piv_eval_par->ifit, print_par, 1);
            }

            if (piv_eval_par->print_cov_logic == FALSE) {
                piv_eval_par->print_cov_logic = 
                    gpiv_scan_iph(GPIV_EVAL_PAR_KEY, line, par_name, 
                                  "Print_cov:", 
                                  &piv_eval_par->print_cov, print_par, 1);
            }
            
            if (piv_eval_par->old_piv_logic == FALSE) {
                piv_eval_par->old_piv_logic = 
                    gpiv_scan_iph(GPIV_EVAL_PAR_KEY, line, par_name, 
                                  "Old_piv:", 
                                  &piv_eval_par->old_piv, print_par, 1);
            }
            

/*             if (piv_eval_par->autokey_logic == FALSE) { */
/*                 piv_eval_par->autokey_logic =  */
/*                     gpiv_scan_iph(GPIV_EVAL_PAR_KEY, line, par_name,  */
/*                                   "Autokey:",  */
/*                                   &piv_eval_par->autokey, print_par, 1); */
/*             } */
            
        }
    }
}


char *
gpiv_piv_check_parameters_read(GpivEvalPar *piv_eval_par,
                               GpivEvalPar piv_eval_par_default
                               )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Check out if all parameters have been read
 *
 * INPUTS:
 *     piv_eval_par_default:
 *
 * OUTPUTS:
 *     piv_eval_par:
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/
{
    char *err_msg = NULL;


/*     if (piv_eval_par->ad_int_logic == FALSE) { */
/* 	err_msg = "GPIV_EVAL_PAR_KEY.Ad_int not set", GPIV_EVAL_PAR_KEY); */
/*         return err_msg; */
/*     } */

    if (piv_eval_par->int_geo_logic == FALSE) {
        err_msg = "Using default:";
        piv_eval_par->int_geo = piv_eval_par_default.int_geo;
        gpiv_warning("%s\nEVAL.Int_geo: %d", err_msg, 
                     piv_eval_par_default.int_geo);
/*         return err_msg; */
    }

    if (piv_eval_par->int_geo == GPIV_AOI && piv_eval_par->col_start_logic == FALSE) {
	err_msg = "Using default:";
        piv_eval_par->col_start = piv_eval_par_default.col_start;
        gpiv_warning("%s\nEVAL.: %d", err_msg, 
                     piv_eval_par_default.col_start);
/*         return err_msg; */
    }

    if (piv_eval_par->int_geo == GPIV_AOI && piv_eval_par->col_end_logic == FALSE) {
	err_msg = "Using default:";
        piv_eval_par->col_end = piv_eval_par_default.col_end;
        gpiv_warning("%s\nEVAL.Col_end: %d", err_msg, 
                     piv_eval_par_default.col_end);
/*         return err_msg; */
    }

    if (piv_eval_par->int_geo == GPIV_AOI && piv_eval_par->row_start_logic == FALSE) {
	err_msg = "Using default:";
        piv_eval_par->row_start = piv_eval_par_default.row_start;
        gpiv_warning("%s\nEVAL.Row_start: %d", err_msg,
                     piv_eval_par_default.row_start);
/*         return err_msg; */
    }

    if (piv_eval_par->int_geo == GPIV_AOI && piv_eval_par->row_end_logic == FALSE) {
	err_msg = "Using default:";
        piv_eval_par->row_end = piv_eval_par_default.row_end;
        gpiv_warning("%s\nEVAL.Row_end: %d", err_msg, 
                     piv_eval_par_default.row_end);
/*         return err_msg; */
    }

    if (piv_eval_par->int_geo == GPIV_LINE_R 
        && piv_eval_par->int_line_row_logic == FALSE) {
        err_msg = "Using default:";
        piv_eval_par->int_line_row = piv_eval_par_default.int_line_row;
       gpiv_warning("%s\nEVAL.Int_line_row: %d", err_msg, 
                     piv_eval_par_default.int_line_row);
/*         return err_msg; */
    }

    if (piv_eval_par->int_geo == GPIV_LINE_C 
        && piv_eval_par->int_line_row_start_logic == FALSE) {
        err_msg = "Using default:";
        piv_eval_par->int_line_row_start = piv_eval_par_default.int_line_row_start;
        gpiv_warning("%s\nEVAL.Int_line_row_start: %d", err_msg, 
                     piv_eval_par_default.int_line_row_start);
/*         return err_msg; */
    }

    if (piv_eval_par->int_geo == GPIV_LINE_C 
        && piv_eval_par->int_line_row_end_logic == FALSE) {
        err_msg = "Using default:";
        piv_eval_par->int_line_row_end = piv_eval_par_default.int_line_row_end;
        gpiv_warning("%s\nEVAL.Int_line_row_end: %d", err_msg, 
                     piv_eval_par_default.int_line_row_end);
/*         return err_msg; */
    }

    if (piv_eval_par->int_geo == GPIV_LINE_C 
        && piv_eval_par->int_line_col_logic == FALSE) {
        err_msg = "Using default:";
        piv_eval_par->int_line_col = piv_eval_par_default.int_line_col;
        gpiv_warning("%s\nEVAL.Int_line_col: %d", err_msg, 
                     piv_eval_par_default.int_line_col);
/*         return err_msg; */
    }

    if (piv_eval_par->int_geo == GPIV_LINE_R 
        && piv_eval_par->int_line_col_start_logic == FALSE) {
        err_msg = "Using default:";
        piv_eval_par->int_line_col_start = piv_eval_par_default.int_line_col_start;
        gpiv_warning("%s\nEVAL.Int_line_col_start: %d", err_msg, 
                     piv_eval_par_default.int_line_col_start);
/*         return err_msg; */
    }

    if (piv_eval_par->int_geo == GPIV_LINE_R 
        && piv_eval_par->int_line_col_end_logic == FALSE) {
        err_msg = "Using default:";
        piv_eval_par->int_line_col_end = piv_eval_par_default.int_line_col_end;
        gpiv_warning("%s\nEVAL.Int_line_col_end: %d", err_msg, 
                     piv_eval_par_default.int_line_col_end);
/*         return err_msg; */
    }

    if (piv_eval_par->int_geo == GPIV_POINT 
        && piv_eval_par->int_point_col_logic == FALSE) {
        err_msg = "Using default:";
        piv_eval_par->int_point_col = piv_eval_par_default.int_point_col;
        gpiv_warning("%s\nEVAL.: %d", err_msg, 
                     piv_eval_par_default.int_point_col);
/*         return err_msg; */
    }

    if (piv_eval_par->int_geo == GPIV_POINT 
        && piv_eval_par->int_point_row_logic == FALSE) {
        err_msg = "Using default:";
        piv_eval_par->int_point_row = piv_eval_par_default.int_point_row;
        gpiv_warning("%s\nEVAL.Int_point_row: %d", err_msg, 
                     piv_eval_par_default.int_point_row);
/*        return err_msg; */
    }



    if (piv_eval_par->int_size_1_logic == FALSE) {
	err_msg = "Using default:";
        piv_eval_par->int_size_1 = piv_eval_par_default.int_size_1;
        gpiv_warning("%s\nEVAL.Int_size_1: %d", err_msg, 
                     piv_eval_par_default.int_size_1);
/*        return err_msg; */
    }

    if (piv_eval_par->int_size_2_logic == FALSE) {
	err_msg = "Using default:";
        piv_eval_par->int_size_2 = piv_eval_par_default.int_size_2;
        gpiv_warning("%s\nEVAL.Int_size_2: %d", err_msg, 
                     piv_eval_par_default.int_size_2);
/*        return err_msg; */
    }

    if (piv_eval_par->int_shift_logic == FALSE) {
	err_msg = "Using default:";
        piv_eval_par->int_shift = piv_eval_par_default.int_shift;
        gpiv_warning("%s\nEVAL.Int_shift: %d", err_msg, 
                     piv_eval_par_default.int_shift);
/*        return err_msg; */
    }

    if (piv_eval_par->pre_shift_col_logic == FALSE) {
	err_msg = "Using default:";
        piv_eval_par->pre_shift_col = piv_eval_par_default.pre_shift_col;
        gpiv_warning("%s\nEVAL.Pre_shift_col: %d", err_msg, 
                     piv_eval_par_default.pre_shift_col);
/*        return err_msg; */
    }

    if (piv_eval_par->pre_shift_row_logic == FALSE) {
	err_msg = "Using default:";
        piv_eval_par->pre_shift_row = piv_eval_par_default.pre_shift_row;
        gpiv_warning("%s\nEVAL.Pre_shift_row: %d", err_msg, 
                     piv_eval_par_default.pre_shift_row);
/*        return err_msg; */
    }

    if (piv_eval_par->peak_logic == FALSE) {
	err_msg = "Using default:";
        piv_eval_par->peak = piv_eval_par_default.peak;
        gpiv_warning("%s\nEVAL.Peak: %d", err_msg, 
                     piv_eval_par_default.peak);
/*        return err_msg; */
    }

    if (piv_eval_par->int_scheme_logic == FALSE) {
	err_msg = "Using default:";
        piv_eval_par->int_scheme = piv_eval_par_default.int_scheme;
        gpiv_warning("%s\nEVAL.Int_scheme: %d", err_msg, 
                     piv_eval_par_default.int_scheme);
    }

    if (piv_eval_par->ifit_logic == FALSE) {
	err_msg = "Using default:";
        piv_eval_par->ifit = piv_eval_par_default.ifit;
        gpiv_warning("%s\nEVAL.Ifit: %d", err_msg, 
                     piv_eval_par_default.ifit);
/*        return err_msg; */
    }

    if (piv_eval_par->print_cov_logic == FALSE) {
	err_msg = "Using default:";
        piv_eval_par->print_cov = piv_eval_par_default.print_cov;
        gpiv_warning("%s\nEVAL.Print_cov: %d", err_msg, 
                     piv_eval_par_default.print_cov);
/*        return err_msg; */
    }

    if (piv_eval_par->old_piv_logic == FALSE) {
	err_msg = "Using default:";
        piv_eval_par->old_piv = piv_eval_par_default.old_piv;
        gpiv_warning("%s\nEVAL.Old_piv: %d", err_msg, 
                     piv_eval_par_default.old_piv);
/*         return err_msg; */
    }



/* 	if (piv_eval_par->autokey_logic == FALSE) */
/* 	    printf("\n#%s warning: Rr.Autokey not set (optional)", */
/* 		   GPIV_EVAL_PAR_KEY); */
/*      } */

    err_msg = NULL;
    return err_msg;
}



char *
gpiv_piv_test_parameter (GpivImagePar * image_par, 
                         GpivEvalPar * piv_eval_par/* , */
                         /*                          int fname_logic */
                         )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Testing parameters on valid values and initializing derived variables 
 *
 * INPUTS:
 *     image_par:      
 *     fname_logic:    
 *
 * OUTPUTS:
 *     piv_eval_par:   
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 * --------------------------------------------------------------------------*/
{
    char * err_msg = NULL;
/*
 * Check if col_end and row_end are within the image size
 */
    if (piv_eval_par->col_end > 
        image_par->ncolumns - piv_eval_par->pre_shift_col - 1) {
	err_msg = "GPIV_PIV_TEST_PARAMETER: last column to be analysed is out of \
image region;\nRr.Col_end > Ncolumns-Rr.Pre_shift_col-1";
         gpiv_warning("%s", err_msg);
       return err_msg;
    }

    if (piv_eval_par->row_end > image_par->nrows - piv_eval_par->pre_shift_row - 1) {
	err_msg = "GPIV_PIV_TEST_PARAMETER: last row to be analysed is out of image "
                   "region;\nRr.Row_end > Nrows-Rr.Pre_shift_row-1";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }

/*
 * Check if size of Interrogation Area's is allowed
 */
    if (piv_eval_par->int_size_1 > GPIV_MAX_INTERR_SIZE || 
        piv_eval_par->int_size_1 > image_par->ncolumns ||
	piv_eval_par->int_size_1 > image_par->nrows) {
	err_msg = "GPIV_PIV_TEST_PARAMETER: int_size_1 larger than allowed maximum, ncolumns or nrows";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }

    if (piv_eval_par->int_size_2 > GPIV_MAX_INTERR_SIZE || 
        piv_eval_par->int_size_2 > image_par->ncolumns ||
	piv_eval_par->int_size_2 > image_par->nrows) {
	err_msg = "GPIV_PIV_TEST_PARAMETER: int_size_2 larger than allowed maximum, ncolumns or nrows";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }

    if (piv_eval_par->int_size_2 < piv_eval_par->int_size_1) {
	err_msg = "GPIV_PIV_TEST_PARAMETER: int_size_2 smaller than int_size_1. \n\
Use int_size_2 larger than int_size_1 for adaptive interrogation sizes.";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }

/*
 * Check ifit
 */
    if (piv_eval_par->ifit != 0 
        && piv_eval_par->ifit != 1 
        && piv_eval_par->ifit != 2 && 
        piv_eval_par->ifit != 3) {
	err_msg = "GPIV_PIV_TEST_PARAMETER: invalid value of RR.Ifit (0, 1, 2 or 3)";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }

/*
 * Piv data from previous analyse can only be used with -f option
 */
/*     if ((piv_eval_par->old_piv == 1) && (piv_eval_par->fname_logic == FALSE)) { */
/* 	err_msg = "GPIV_PIV_TEST_PARAMETER: Old_piv can only be used in combination with -f 'filename'"; */
/*         gpiv_warning("%s", err_msg); */
/*         return err_msg; */
/*     } */

/*
 * Some variables depend on parameters
 */
    if (piv_eval_par->int_size_2 > piv_eval_par->int_size_1) {
        piv_eval_par->ad_int = 1;
        if (!piv_eval_par->int_scheme_logic) {
            piv_eval_par->int_scheme = GPIV_ZERO_OFF_CENTRAL;
        }
    } else {
        piv_eval_par->ad_int = 0;
    }


    return err_msg;
}



void 
gpiv_piv_print_parameters(GpivEvalPar piv_eval_par
                          )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Prints parameters to stdout
 *
 * INPUTS:
 *      piv_eval_par:   PIV evaluation parameters
 *
 * OUTPUTS:
 *
 * RETURNS:
 *
 *---------------------------------------------------------------------------*/
{
    if (piv_eval_par.int_geo_logic)
	printf("%s%s %d\n", GPIV_EVAL_PAR_KEY, "Int_geo:", 
               piv_eval_par.int_geo);

    if (piv_eval_par.int_geo_logic
        && piv_eval_par.int_geo == GPIV_AOI 
        && piv_eval_par.col_start_logic)
	printf("%s%s %d\n", GPIV_EVAL_PAR_KEY, "Col_start:", 
               piv_eval_par.col_start);

    if (piv_eval_par.int_geo_logic
        && piv_eval_par.int_geo == GPIV_AOI 
        && piv_eval_par.col_end_logic)
	printf("%s%s %d\n", GPIV_EVAL_PAR_KEY, "Col_end:", 
               piv_eval_par.col_end);

    if (piv_eval_par.int_geo_logic
        && piv_eval_par.int_geo == GPIV_AOI 
        && piv_eval_par.row_start_logic)
	printf("%s%s %d\n", GPIV_EVAL_PAR_KEY, "Row_start:", 
               piv_eval_par.row_start);

    if (piv_eval_par.int_geo_logic
        && piv_eval_par.int_geo == GPIV_AOI 
        && piv_eval_par.row_end_logic)
	printf("%s%s %d\n", GPIV_EVAL_PAR_KEY, "Row_end:", 
               piv_eval_par.row_end);



    if (piv_eval_par.int_geo_logic
        && piv_eval_par.int_geo == GPIV_LINE_R 
        && piv_eval_par.int_line_row_logic)
	printf("%s%s %d\n", GPIV_EVAL_PAR_KEY, "Int_line_row:", 
               piv_eval_par.int_line_row);

    if (piv_eval_par.int_geo_logic
        && piv_eval_par.int_geo == GPIV_LINE_C 
        && piv_eval_par.int_line_row_start_logic)
	printf("%s%s %d\n", GPIV_EVAL_PAR_KEY, "Int_line_row_start:", 
               piv_eval_par.int_line_row_start);

    if (piv_eval_par.int_geo_logic
        && piv_eval_par.int_geo == GPIV_LINE_C
        && piv_eval_par.int_line_row_end_logic)
	printf("%s%s %d\n", GPIV_EVAL_PAR_KEY, "Int_line_row_end:", 
               piv_eval_par.int_line_row_end);



    if (piv_eval_par.int_geo_logic
        && piv_eval_par.int_geo == GPIV_LINE_C 
        && piv_eval_par.int_line_col_logic)
	printf("%s%s %d\n", GPIV_EVAL_PAR_KEY, "Int_line_col:", 
               piv_eval_par.int_line_col);

    if (piv_eval_par.int_geo_logic
        && piv_eval_par.int_geo == GPIV_LINE_R 
        && piv_eval_par.int_line_col_start_logic)
	printf("%s%s %d\n", GPIV_EVAL_PAR_KEY, "Int_line_col_start:", 
	       piv_eval_par.int_line_col_start);

    if (piv_eval_par.int_geo_logic
        && piv_eval_par.int_geo == GPIV_LINE_R 
        && piv_eval_par.int_line_col_end_logic)
	printf("%s%s %d\n", GPIV_EVAL_PAR_KEY, "Int_line_col_end:", 
	       piv_eval_par.int_line_col_end);


    if (piv_eval_par.int_geo_logic
        && piv_eval_par.int_geo == GPIV_POINT 
        && piv_eval_par.int_point_row_logic)
	printf("%s%s %d\n", GPIV_EVAL_PAR_KEY, "Int_point_row:", 
	       piv_eval_par.int_point_row);

    if (piv_eval_par.int_geo_logic
        && piv_eval_par.int_geo == GPIV_POINT 
        && piv_eval_par.int_point_col_logic)
	printf("%s%s %d\n", GPIV_EVAL_PAR_KEY, "Int_point_col:", 
	       piv_eval_par.int_point_col);


    
    if (piv_eval_par.ad_int_logic)
	printf("%s%s %d\n", GPIV_EVAL_PAR_KEY, "Ad_int:", 
               piv_eval_par.ad_int);

    if (piv_eval_par.int_size_1_logic)
	printf("%s%s %d\n", GPIV_EVAL_PAR_KEY, "Int_size_1:", 
               piv_eval_par.int_size_1);

    if (piv_eval_par.int_size_2_logic)
	printf("%s%s %d\n", GPIV_EVAL_PAR_KEY, "Int_size_2:", 
               piv_eval_par.int_size_2);

    if (piv_eval_par.int_shift_logic)
	printf("%s%s %d\n", GPIV_EVAL_PAR_KEY, "Int_shift:", 
               piv_eval_par.int_shift);

    if (piv_eval_par.int_scheme_logic)
	printf("%s%s %d\n", GPIV_EVAL_PAR_KEY, "Int_scheme:", 
               piv_eval_par.int_scheme);

    if (piv_eval_par.pre_shift_col_logic)
	printf("%s%s %d\n", GPIV_EVAL_PAR_KEY, "Pre_shift_col:", 
               piv_eval_par.pre_shift_col);

    if (piv_eval_par.pre_shift_row_logic)
	printf("%s%s %d\n", GPIV_EVAL_PAR_KEY, "Pre_shift_row:", 
               piv_eval_par.pre_shift_row);

    if (piv_eval_par.ifit_logic)
	printf("%s%s %d\n", GPIV_EVAL_PAR_KEY, "Ifit:", 
               piv_eval_par.ifit);

    if (piv_eval_par.peak_logic)
	printf("%s%s %d\n", GPIV_EVAL_PAR_KEY, "Peak:", 
               piv_eval_par.peak);

    if (piv_eval_par.print_piv_logic)
	printf("%s%s %d\n", GPIV_EVAL_PAR_KEY, "Print_piv", 
               piv_eval_par.print_piv);

    if (piv_eval_par.print_cov_logic)
	printf("%s%s %d\n", GPIV_EVAL_PAR_KEY, "Print_cov:", 
               piv_eval_par.print_cov);

    if (piv_eval_par.old_piv_logic)
	printf("%s%s %d\n", GPIV_EVAL_PAR_KEY, "Old_piv:", 
               piv_eval_par.old_piv);
}



void 
gpiv_piv_fprint_parameters(FILE * fp_par_out, 
                           GpivEvalPar piv_eval_par
                           )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Prints parameters to fp_par_out
 *
 * INPUTS:
 *      fp_par_out:     
 *      piv_eval_par:   PIV evaluation parameters
 *
 * OUTPUTS:
 *
 * RETURNS:
 *
 *---------------------------------------------------------------------------*/
{
    if (piv_eval_par.int_geo_logic)
	fprintf(fp_par_out, "%s%s %d\n", GPIV_EVAL_PAR_KEY, "Int_geo:", 
               piv_eval_par.int_geo);

    if (piv_eval_par.col_start_logic)
	fprintf(fp_par_out, "%s%s %d\n", GPIV_EVAL_PAR_KEY, "Col_start:", 
                piv_eval_par.col_start);

    if (piv_eval_par.col_end_logic)
	fprintf(fp_par_out, "%s%s %d\n", GPIV_EVAL_PAR_KEY, "Col_end:", 
                piv_eval_par.col_end);

    if (piv_eval_par.row_start_logic)
	fprintf(fp_par_out, "%s%s %d\n", GPIV_EVAL_PAR_KEY, "Row_start:", 
                piv_eval_par.row_start);

    if (piv_eval_par.row_end_logic)
	fprintf(fp_par_out, "%s%s %d\n", GPIV_EVAL_PAR_KEY, "Row_end:", 
                piv_eval_par.row_end);



    if (piv_eval_par.int_geo_logic
        && piv_eval_par.int_geo == GPIV_LINE_R 
        && piv_eval_par.int_line_row_logic) 
	fprintf(fp_par_out, "%s%s %d\n", GPIV_EVAL_PAR_KEY, "Int_line_row:", 
	       piv_eval_par.int_line_row);

    if (piv_eval_par.int_geo_logic
        && piv_eval_par.int_geo == GPIV_LINE_C 
        && piv_eval_par.int_line_row_start_logic) 
	fprintf(fp_par_out, "%s%s %d\n", GPIV_EVAL_PAR_KEY, "Int_line_row_start:", 
	       piv_eval_par.int_line_row_start);

    if (piv_eval_par.int_geo_logic
        && piv_eval_par.int_geo == GPIV_LINE_C 
        && piv_eval_par.int_line_row_end_logic) 
	fprintf(fp_par_out, "%s%s %d\n", GPIV_EVAL_PAR_KEY, "Int_line_row_end:", 
	       piv_eval_par.int_line_row_end);
    


    if (piv_eval_par.int_geo_logic
        && piv_eval_par.int_geo == GPIV_LINE_C 
        && piv_eval_par.int_line_col_logic) 
	fprintf(fp_par_out, "%s%s %d\n", GPIV_EVAL_PAR_KEY, "Int_line_col:", 
	       piv_eval_par.int_line_col);

    if (piv_eval_par.int_geo_logic
        && piv_eval_par.int_geo == GPIV_LINE_R 
        && piv_eval_par.int_line_col_start_logic) 
	fprintf(fp_par_out, "%s%s %d\n", GPIV_EVAL_PAR_KEY, "Int_line_col_start:", 
	       piv_eval_par.int_line_col_start);

    if (piv_eval_par.int_geo_logic
        && piv_eval_par.int_geo == GPIV_LINE_R 
        && piv_eval_par.int_line_col_end_logic) 
	fprintf(fp_par_out, "%s%s %d\n", GPIV_EVAL_PAR_KEY, "Int_line_col_end:", 
	       piv_eval_par.int_line_col_end);
    


    if (piv_eval_par.int_geo_logic
        && piv_eval_par.int_geo == GPIV_POINT 
        && piv_eval_par.int_point_row_logic)
	fprintf(fp_par_out, "%s%s %d\n",
		GPIV_EVAL_PAR_KEY, "Int_point_row:", 
                piv_eval_par.int_point_row);

    if (piv_eval_par.int_geo_logic
        && piv_eval_par.int_geo == GPIV_POINT 
        && piv_eval_par.int_point_col_logic)
	fprintf(fp_par_out, "%s%s %d\n", GPIV_EVAL_PAR_KEY, "Int_point_col:",
		piv_eval_par.int_point_col);




    if (piv_eval_par.ad_int_logic)
	fprintf(fp_par_out, "%s%s %d\n", GPIV_EVAL_PAR_KEY, "Ad_int:", 
                piv_eval_par.ad_int);

    if (piv_eval_par.int_size_1_logic)
	fprintf(fp_par_out, "%s%s %d\n",
		GPIV_EVAL_PAR_KEY, "Int_size_1:", 
                piv_eval_par.int_size_1);

    if (piv_eval_par.int_size_2_logic)
	fprintf(fp_par_out, "%s%s %d\n",
		GPIV_EVAL_PAR_KEY, "Int_size_2:", 
                piv_eval_par.int_size_2);

    if (piv_eval_par.int_shift_logic)
	fprintf(fp_par_out, "%s%s %d\n", GPIV_EVAL_PAR_KEY, "Int_shift:", 
                piv_eval_par.int_shift);

    if (piv_eval_par.int_scheme_logic)
	fprintf(fp_par_out, "%s%s %d\n", GPIV_EVAL_PAR_KEY, "Int_scheme:", 
                piv_eval_par.int_scheme);

    if (piv_eval_par.pre_shift_col_logic)
	fprintf(fp_par_out, "%s%s %d\n", GPIV_EVAL_PAR_KEY, "Pre_shift_col:", 
                piv_eval_par.pre_shift_col);

    if (piv_eval_par.pre_shift_row_logic)
	fprintf(fp_par_out, "%s%s %d\n", GPIV_EVAL_PAR_KEY, "Pre_shift_row:", 
                piv_eval_par.pre_shift_row);

    if (piv_eval_par.ifit_logic)
	fprintf(fp_par_out, "%s%s %d\n", GPIV_EVAL_PAR_KEY, "Ifit:", 
                piv_eval_par.ifit);



    if (piv_eval_par.peak_logic)
	fprintf(fp_par_out, "%s%s %d\n", GPIV_EVAL_PAR_KEY, "Peak:", 
                piv_eval_par.peak);

    if (piv_eval_par.print_piv_logic)
	fprintf(fp_par_out, "%s%s %d\n", GPIV_EVAL_PAR_KEY, "Print_piv", 
                piv_eval_par.print_piv);

    if (piv_eval_par.print_cov_logic)
	fprintf(fp_par_out, "%s%s %d\n", GPIV_EVAL_PAR_KEY, "Print_cov:", 
                piv_eval_par.print_cov);

    if (piv_eval_par.old_piv_logic)
	fprintf(fp_par_out, "%s%s %d\n", GPIV_EVAL_PAR_KEY, "Old_piv:", 
                piv_eval_par.old_piv);

}



void 
gpiv_piv_cp_parameters(GpivEvalPar piv_eval_par_src, 
                       GpivEvalPar * piv_eval_par_dest,
                       gboolean force,
                       gboolean print_par
                       )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Copies piv parameters from src to dest
 *
 * INPUTS:
 *      piv_eval_par_src:       source piv parameters
 *      force:                  flag to force the copying, even if destination
 *                              already exists
 *      print_par:              verbose output. Not implemented yet.
 *
 * OUTPUTS:
 *      piv_eval_par_dest:     destination piv parameters
 *
 * RETURNS:
 *
 *---------------------------------------------------------------------------*/
{
    if (force 
        || ( piv_eval_par_src.col_start_logic 
        && !piv_eval_par_dest->col_start_logic)) {
        piv_eval_par_dest->col_start = piv_eval_par_src.col_start;
        piv_eval_par_dest->col_start_logic = TRUE;
    }

    if (force 
        || ( piv_eval_par_src.col_end_logic 
        && !piv_eval_par_dest->col_end_logic)) {
        piv_eval_par_dest->col_end = piv_eval_par_src.col_end;
        piv_eval_par_dest->col_end_logic = TRUE;
    }

    if (force 
        || ( piv_eval_par_src.row_start_logic 
        && !piv_eval_par_dest->row_start_logic)) {
        piv_eval_par_dest->row_start = piv_eval_par_src.row_start;
        piv_eval_par_dest->row_start_logic = TRUE;
    }

    if (force 
        || ( piv_eval_par_src.row_end_logic 
        && !piv_eval_par_dest->row_end_logic)) {
        piv_eval_par_dest->row_end = piv_eval_par_src.row_end;
        piv_eval_par_dest->row_end_logic = TRUE;
    }

    
    if (force 
        || ( piv_eval_par_src.int_geo_logic 
        && !piv_eval_par_dest->int_geo_logic)) {
        piv_eval_par_dest->int_geo = piv_eval_par_src.int_geo;
        piv_eval_par_dest->int_geo_logic = TRUE;
    }

    if (force 
        || ( piv_eval_par_src.int_line_col_logic 
        && !piv_eval_par_dest->int_line_col_logic)) {
        piv_eval_par_dest->int_line_col = piv_eval_par_src.int_line_col;
        piv_eval_par_dest->int_line_col_logic = TRUE;
    }

    if (force 
        || ( piv_eval_par_src.int_line_col_start_logic 
        && !piv_eval_par_dest->int_line_col_start_logic)) {
        piv_eval_par_dest->int_line_col_start = 
            piv_eval_par_src.int_line_col_start;
        piv_eval_par_dest->int_line_col_start_logic = TRUE;
    }

    if (force 
        || ( piv_eval_par_src.int_line_col_end_logic 
        && !piv_eval_par_dest->int_line_col_end_logic)) {
        piv_eval_par_dest->int_line_col_end = 
            piv_eval_par_src.int_line_col_end;
        piv_eval_par_dest->int_line_col_end_logic = TRUE;
    }

    if (force 
        || ( piv_eval_par_src.int_line_row_logic 
        && !piv_eval_par_dest->int_line_row_logic)) {
        piv_eval_par_dest->int_line_row = piv_eval_par_src.int_line_row;
        piv_eval_par_dest->int_line_row_logic = TRUE;
    }

    if (force 
        || ( piv_eval_par_src.int_line_row_start_logic 
        && !piv_eval_par_dest->int_line_row_start_logic)) {
        piv_eval_par_dest->int_line_row_start = 
            piv_eval_par_src.int_line_row_start;
        piv_eval_par_dest->int_line_row_start_logic = TRUE;
    }

    if (force 
        || ( piv_eval_par_src.int_line_row_end_logic 
        && !piv_eval_par_dest->int_line_row_end_logic)) {
        piv_eval_par_dest->int_line_row_end = 
            piv_eval_par_src.int_line_row_end;
        piv_eval_par_dest->int_line_row_end_logic = TRUE;
    }

    if (force 
        || ( piv_eval_par_src.int_point_col_logic 
        && !piv_eval_par_dest->int_point_col_logic)) {
        piv_eval_par_dest->int_point_col = piv_eval_par_src.int_point_col;
        piv_eval_par_dest->int_point_col_logic = TRUE;
    }

    if (force 
        || ( piv_eval_par_src.int_point_row_logic 
        && !piv_eval_par_dest->int_point_row_logic)) {
        piv_eval_par_dest->int_point_row = piv_eval_par_src.int_point_row;
        piv_eval_par_dest->int_point_row_logic = TRUE;
    }

    if (force 
        || ( piv_eval_par_src.int_size_1_logic 
        && !piv_eval_par_dest->int_size_1_logic)) {
        piv_eval_par_dest->int_size_1 = piv_eval_par_src.int_size_1;
        piv_eval_par_dest->int_size_1_logic = TRUE;
    }

    if (force 
        || ( piv_eval_par_src.int_size_2_logic 
        && !piv_eval_par_dest->int_size_2_logic)) {
        piv_eval_par_dest->int_size_2 = piv_eval_par_src.int_size_2;
        piv_eval_par_dest->int_size_2_logic = TRUE;
    }

    if (force 
        || ( piv_eval_par_src.int_shift_logic 
        && !piv_eval_par_dest->int_shift_logic)) {
        piv_eval_par_dest->int_shift = 
            piv_eval_par_src.int_shift;
        piv_eval_par_dest->int_shift_logic = TRUE;
    }

    if (force 
        || ( piv_eval_par_src.pre_shift_col_logic 
        && !piv_eval_par_dest->pre_shift_col_logic)) {
        piv_eval_par_dest->pre_shift_col = piv_eval_par_src.pre_shift_col;
        piv_eval_par_dest->pre_shift_col_logic = TRUE;
    }

    if (force 
        || ( piv_eval_par_src.pre_shift_row_logic 
        && !piv_eval_par_dest->pre_shift_row_logic)) {
        piv_eval_par_dest->pre_shift_row = piv_eval_par_src.pre_shift_row;
        piv_eval_par_dest->pre_shift_row_logic = TRUE;
    }

    if (force 
        || ( piv_eval_par_src.old_piv_logic 
        && !piv_eval_par_dest->old_piv_logic)) {
        piv_eval_par_dest->old_piv = piv_eval_par_src.old_piv;
        piv_eval_par_dest->old_piv_logic = TRUE;
    }

    if (force 
        || ( piv_eval_par_src.peak_logic 
        && !piv_eval_par_dest->peak_logic)) {
        piv_eval_par_dest->peak = piv_eval_par_src.peak;
        piv_eval_par_dest->peak_logic = TRUE;
    }

    if (force 
        || ( piv_eval_par_src.print_cov_logic 
        && !piv_eval_par_dest->print_cov_logic)) {
        piv_eval_par_dest->print_cov = piv_eval_par_src.print_cov;
        piv_eval_par_dest->print_cov_logic = TRUE;
    }

    if (force 
        || ( piv_eval_par_src.print_par_logic 
        && !piv_eval_par_dest->print_par_logic)) {
        piv_eval_par_dest->print_par = piv_eval_par_src.print_par;
        piv_eval_par_dest->print_par_logic = TRUE;
    }

    if (force 
        || ( piv_eval_par_src.print_piv_logic 
        && !piv_eval_par_dest->print_piv_logic)) {
        piv_eval_par_dest->print_piv = piv_eval_par_src.print_piv;
        piv_eval_par_dest->print_piv_logic = TRUE;
    }

    if (force 
        || ( piv_eval_par_src.int_scheme_logic 
        && !piv_eval_par_dest->int_scheme_logic)) {
        piv_eval_par_dest->int_scheme = piv_eval_par_src.int_scheme;
        piv_eval_par_dest->int_scheme_logic = TRUE;
    }

    if (force 
        || ( piv_eval_par_src.ifit_logic 
        && !piv_eval_par_dest->ifit_logic)) {
        piv_eval_par_dest->ifit = piv_eval_par_src.ifit;
        piv_eval_par_dest->ifit_logic = TRUE;
    }

    if (force 
        || ( piv_eval_par_src.ad_int_logic 
        && !piv_eval_par_dest->ad_int_logic)) {
        piv_eval_par_dest->ad_int = piv_eval_par_src.ad_int;
        piv_eval_par_dest->ad_int_logic = TRUE;
    }

    if (force 
        || ( piv_eval_par_src.autokey_logic 
        && !piv_eval_par_dest->autokey_logic)) {
        piv_eval_par_dest->autokey = piv_eval_par_src.autokey;
        piv_eval_par_dest->autokey_logic = TRUE;
    }

}



char *
gpiv_piv_fread_hdf5_parameters(char *fname, 
                         GpivEvalPar * piv_eval_par 
                         )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *   Reads piv parameters from hdf5 data file
 *
 * PROTOTYPE LOCATATION:
 *   gpiv_eval.h
 *
 * INPUTS:
 *   group_id:         hdf group identity
 *
 * OUTPUTS:
 *   piv_eval_par:     parameter structure
 *---------------------------------------------------------------------------*/
{
    char *err_msg = NULL;
    int i;
/*
 * HDF declarations
 */
    hid_t       file_id, group_id, attribute_id; 
    herr_t      status;

    if ((i = H5Fis_hdf5(fname)) == 0) {
        err_msg = "gpiv_piv_fread_hdf5_parameters: not an hdf5 file";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }

    file_id = H5Fopen(fname, H5F_ACC_RDONLY, H5P_DEFAULT);
    group_id = H5Gopen (file_id, "PIV");

    H5Aiterate(group_id, NULL, attr_info, piv_eval_par );

   status = H5Gclose (group_id);
   status = H5Fclose(file_id); 
   return err_msg;
}



char *
gpiv_piv_fwrite_hdf5_parameters(char *fname, 
                                GpivEvalPar * piv_eval_par 
                                )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *   Writes piv parameters to an existing hdf5 data file
 *
 * PROTOTYPE LOCATATION:
 *   gpiv_io.h
 *
 * INPUTS:
 *   group_id:         hdf group identity
 *   piv_eval_par:     parameter structure
 *
 * OUTPUTS:
 *---------------------------------------------------------------------------*/
{
    char *err_msg = NULL;
    int i;
/*
 * HDF declarations
 */
    hid_t       file_id, dataspace_id, group_id, attribute_id; 
    hsize_t     dims[1];
    herr_t      status;

    if ((i = H5Fis_hdf5(fname)) == 0) {
        err_msg = "gpiv_piv_fwrite_hdf5_parameters: not an hdf5 file";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }

    file_id = H5Fopen(fname, H5F_ACC_RDWR, H5P_DEFAULT);
    group_id = H5Gopen (file_id, "PIV");



    dims[0] = 1;
    dataspace_id = H5Screate_simple(1, dims, NULL);

/* 
 * Image piv parametes
 */
    if (piv_eval_par->int_geo_logic) {
        attribute_id = H5Acreate(group_id, "int_geo", H5T_NATIVE_INT, 
                                 dataspace_id, H5P_DEFAULT);
        status = H5Awrite(attribute_id, H5T_NATIVE_INT, 
                          &piv_eval_par->int_geo); 
        status = H5Aclose(attribute_id); 
    }

    if (piv_eval_par->int_geo == GPIV_AOI && piv_eval_par->col_start_logic) {
        attribute_id = H5Acreate(group_id, "col_start", H5T_NATIVE_INT, 
                                 dataspace_id, H5P_DEFAULT);
        status = H5Awrite(attribute_id, H5T_NATIVE_INT, 
                          &piv_eval_par->col_start); 
        status = H5Aclose(attribute_id); 
    }

    if (piv_eval_par->int_geo == GPIV_AOI && piv_eval_par->col_end_logic) {
        attribute_id = H5Acreate(group_id, "col_end", H5T_NATIVE_INT, 
                                 dataspace_id, H5P_DEFAULT);
        status = H5Awrite(attribute_id, H5T_NATIVE_INT, 
                          &piv_eval_par->col_end); 
        status = H5Aclose(attribute_id); 
    }

    if (piv_eval_par->int_geo == GPIV_AOI && piv_eval_par->row_start_logic) {
        attribute_id = H5Acreate(group_id, "row_start", H5T_NATIVE_INT, 
                                 dataspace_id, H5P_DEFAULT);
        status = H5Awrite(attribute_id, H5T_NATIVE_INT, 
                          &piv_eval_par->row_start); 
        status = H5Aclose(attribute_id); 
    }

    if (piv_eval_par->int_geo == GPIV_AOI && piv_eval_par->row_end_logic) {
        attribute_id = H5Acreate(group_id, "row_end", H5T_NATIVE_INT, 
                                 dataspace_id, H5P_DEFAULT);
        status = H5Awrite(attribute_id, H5T_NATIVE_INT, 
                          &piv_eval_par->row_end); 
        status = H5Aclose(attribute_id); 
    }

/* 
 * line
 */
    if (piv_eval_par->int_geo == GPIV_LINE_R && piv_eval_par->int_line_row_logic) {
        attribute_id = H5Acreate(group_id, "int_line_row", H5T_NATIVE_INT, 
                                 dataspace_id, H5P_DEFAULT);
        status = H5Awrite(attribute_id, H5T_NATIVE_INT, 
                          &piv_eval_par->int_line_row); 
        status = H5Aclose(attribute_id); 
    }

    if (piv_eval_par->int_geo == GPIV_LINE_C && piv_eval_par->int_line_row_start_logic) {
        attribute_id = H5Acreate(group_id, "int_line_row_start", 
                                 H5T_NATIVE_INT, 
                                 dataspace_id, H5P_DEFAULT);
        status = H5Awrite(attribute_id, H5T_NATIVE_INT, &piv_eval_par->int_line_row_start); 
        status = H5Aclose(attribute_id); 
    }

    if (piv_eval_par->int_geo == GPIV_LINE_C && piv_eval_par->int_line_row_end_logic) {
        attribute_id = H5Acreate(group_id, "int_line_row_end", H5T_NATIVE_INT, 
                                 dataspace_id, H5P_DEFAULT);
        status = H5Awrite(attribute_id, H5T_NATIVE_INT, 
                          &piv_eval_par->int_line_row_end); 
        status = H5Aclose(attribute_id); 
    }

    if (piv_eval_par->int_geo == GPIV_LINE_C && piv_eval_par->int_line_col_logic) {
        attribute_id = H5Acreate(group_id, "int_line_col", H5T_NATIVE_INT, 
                                 dataspace_id, H5P_DEFAULT);
        status = H5Awrite(attribute_id, H5T_NATIVE_INT, 
                          &piv_eval_par->int_line_col); 
        status = H5Aclose(attribute_id); 
    }

    if (piv_eval_par->int_geo == GPIV_LINE_R && piv_eval_par->int_line_col_start_logic) {
        attribute_id = H5Acreate(group_id, "int_line_col_start", 
                                 H5T_NATIVE_INT, 
                                 dataspace_id, H5P_DEFAULT);
        status = H5Awrite(attribute_id, H5T_NATIVE_INT, 
                          &piv_eval_par->int_line_col_start); 
        status = H5Aclose(attribute_id); 
    }

    if (piv_eval_par->int_geo == GPIV_LINE_R && piv_eval_par->int_line_col_end_logic) {
        attribute_id = H5Acreate(group_id, "int_line_col_end", H5T_NATIVE_INT, 
                                 dataspace_id, H5P_DEFAULT);
        status = H5Awrite(attribute_id, H5T_NATIVE_INT, 
                          &piv_eval_par->int_line_col_end); 
        status = H5Aclose(attribute_id); 
    }

/*
 * point
 */


    if (piv_eval_par->int_geo == GPIV_POINT && piv_eval_par->int_point_col_logic) {
        attribute_id = H5Acreate(group_id, "int_point_col", H5T_NATIVE_INT, 
                                 dataspace_id, H5P_DEFAULT);
        status = H5Awrite(attribute_id, H5T_NATIVE_INT, 
                          &piv_eval_par->int_point_col); 
        status = H5Aclose(attribute_id); 
    }

    if (piv_eval_par->int_geo == GPIV_POINT && piv_eval_par->int_point_row_logic) {
        attribute_id = H5Acreate(group_id, "int_point_row", H5T_NATIVE_INT, 
                                 dataspace_id, H5P_DEFAULT);
        status = H5Awrite(attribute_id, H5T_NATIVE_INT, 
                          &piv_eval_par->int_point_row); 
        status = H5Aclose(attribute_id); 
    }

/*         attribute_id = H5Acreate(group_id, "autokey", H5T_NATIVE_INT,  */
/*                                  dataspace_id, H5P_DEFAULT); */
/*         status = H5Awrite(attribute_id, H5T_NATIVE_INT, &piv_eval_par->autokey);  */
/*         status = H5Aclose(attribute_id);  */

/*
 * Interrogation
 */
    if (piv_eval_par->int_size_1_logic) {
        attribute_id = H5Acreate(group_id, "int_size_1", H5T_NATIVE_INT, 
                                 dataspace_id, H5P_DEFAULT);
        status = H5Awrite(attribute_id, H5T_NATIVE_INT, 
                          &piv_eval_par->int_size_1); 
        status = H5Aclose(attribute_id); 
    }

    if (piv_eval_par->int_size_2_logic) {
        attribute_id = H5Acreate(group_id, "int_size_2", H5T_NATIVE_INT, 
                                 dataspace_id, H5P_DEFAULT);
        status = H5Awrite(attribute_id, H5T_NATIVE_INT, 
                          &piv_eval_par->int_size_2); 
        status = H5Aclose(attribute_id); 
    }

    if (piv_eval_par->int_shift_logic) {
        attribute_id = H5Acreate(group_id, "int_shift", H5T_NATIVE_INT, 
                                 dataspace_id, H5P_DEFAULT);
        status = H5Awrite(attribute_id, H5T_NATIVE_INT, 
                          &piv_eval_par->int_shift); 
        status = H5Aclose(attribute_id); 
    }

    if (piv_eval_par->pre_shift_col_logic) {
        attribute_id = H5Acreate(group_id, "pre_shift_col", H5T_NATIVE_INT, 
                                 dataspace_id, H5P_DEFAULT);
        status = H5Awrite(attribute_id, H5T_NATIVE_INT, 
                          &piv_eval_par->pre_shift_col); 
        status = H5Aclose(attribute_id); 
    }

    if (piv_eval_par->pre_shift_row_logic) {
        attribute_id = H5Acreate(group_id, "pre_shift_row", H5T_NATIVE_INT, 
                                 dataspace_id, H5P_DEFAULT);
        status = H5Awrite(attribute_id, H5T_NATIVE_INT, 
                          &piv_eval_par->pre_shift_row); 
        status = H5Aclose(attribute_id); 
    }

/*
 * Sub-pixel interrogation
 */
    if (piv_eval_par->ifit_logic) {
        attribute_id = H5Acreate(group_id, "ifit", H5T_NATIVE_INT, 
                                 dataspace_id, H5P_DEFAULT);
        status = H5Awrite(attribute_id, H5T_NATIVE_INT, &piv_eval_par->ifit); 
        status = H5Aclose(attribute_id); 
    }

    if (piv_eval_par->peak_logic) {
        attribute_id = H5Acreate(group_id, "peak", H5T_NATIVE_INT, 
                                 dataspace_id, H5P_DEFAULT);
        status = H5Awrite(attribute_id, H5T_NATIVE_INT, &piv_eval_par->peak); 
        status = H5Aclose(attribute_id); 
    }

    if (piv_eval_par->int_scheme_logic) {
        attribute_id = H5Acreate(group_id, "int_scheme", H5T_NATIVE_INT, 
                                 dataspace_id, H5P_DEFAULT);
        status = H5Awrite(attribute_id, H5T_NATIVE_INT, 
                          &piv_eval_par->int_scheme); 
        status = H5Aclose(attribute_id); 
    }

    if (piv_eval_par->print_piv_logic) {
        attribute_id = H5Acreate(group_id, "print_piv", H5T_NATIVE_INT, 
                                 dataspace_id, H5P_DEFAULT);
        status = H5Awrite(attribute_id, H5T_NATIVE_INT, 
                          &piv_eval_par->print_piv); 
        status = H5Aclose(attribute_id); 
    }

    if (piv_eval_par->old_piv_logic) {
        attribute_id = H5Acreate(group_id, "old_piv", H5T_NATIVE_INT, 
                                 dataspace_id, H5P_DEFAULT);
        status = H5Awrite(attribute_id, H5T_NATIVE_INT, 
                          &piv_eval_par->old_piv); 
        status = H5Aclose(attribute_id); 
    }


    status = H5Sclose(dataspace_id);
    status = H5Gclose (group_id);
    status = H5Fclose(file_id); 
    return err_msg;
}
