/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 4 c-style: "K&R" -*- */

/*
   libgpiv - library for Particle Image Velocimetry

   Copyright (C) 2002, 2003, 2004 Gerber van der Graaf

   This file is part of libgpiv.
   Libgpiv is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  



-----------------------------------------------------------------------
FILENAME:               post_par.c
LIBRARY:                libgpiv


EXTERNAL FUNCTIONS:     gpiv_post_read_parameters
                        gpiv_post_check_parameters_read
			gpiv_post_test_parameter
			gpiv_post_print_parameters
			gpiv_post_fprint_parameters
                        gpiv_post_fread_hdf_parameters
                        gpiv_post_fwrite_hdf_parameters

LAST MODIFICATION DATE: $Id: post_par.c,v 1.1 2006/01/13 10:05:54 gerber Exp $
 ------------------------------------------------------------------- */

#include <gpiv.h>



/* 
 * Local functions
 */

static herr_t 
attr_info(hid_t loc_id, 
          const char *name, 
          GpivPostPar * piv_post_par
          )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Operator function.
 *
 * PROTOTYPE LOCATATION:
 *     post.h
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    hid_t attribute_id, atype;
    herr_t status;



/*
 * Parameters of manipiv
 */
    if (strcmp(name, "operator_manipiv") == 0) {
        attribute_id = H5Aopen_name(loc_id, name);
        status = H5Aread(attribute_id, H5T_NATIVE_INT, 
                         &piv_post_par->operator_manipiv); 
        status = H5Aclose(attribute_id); 
        piv_post_par->operator_manipiv_logic = TRUE;
    }


/*
 * Global variables and parameters of s-avg
 */
    if (strcmp(name, "subtract") == 0) {
        attribute_id = H5Aopen_name(loc_id, name);
        status = H5Aread(attribute_id, H5T_NATIVE_INT, 
                         &piv_post_par->subtract); 
        status = H5Aclose(attribute_id); 
        piv_post_par->subtract_logic = TRUE;
    }

    if (strcmp(name, "z_off_dx") == 0) {
        attribute_id = H5Aopen_name(loc_id, name);
        status = H5Aread(attribute_id, H5T_NATIVE_FLOAT, 
                         &piv_post_par->z_off_dx); 
        status = H5Aclose(attribute_id); 
        piv_post_par->z_off_dx_logic = TRUE;
    }


    if (strcmp(name, "z_off_dy") == 0) {
        attribute_id = H5Aopen_name(loc_id, name);
        status = H5Aread(attribute_id, H5T_NATIVE_FLOAT, 
                         &piv_post_par->z_off_dy); 
        status = H5Aclose(attribute_id); 
        piv_post_par->z_off_dy_logic = TRUE;
    }


/*
 * Parameters of vorstra
 */
    if (strcmp(name, "diff_type") == 0) {
        attribute_id = H5Aopen_name(loc_id, name);
        status = H5Aread(attribute_id, H5T_NATIVE_INT, 
                         &piv_post_par->diff_type); 
        status = H5Aclose(attribute_id); 
        piv_post_par->diff_type_logic = TRUE;
    }


    if (strcmp(name, "operator_vorstra") == 0) {
        attribute_id = H5Aopen_name(loc_id, name);
        status = H5Aread(attribute_id, H5T_NATIVE_INT, 
                         &piv_post_par->operator_vorstra); 
        status = H5Aclose(attribute_id); 
        piv_post_par->operator_vorstra_logic = TRUE;
    }

/*
 * Parameters of scale
 */
    if (strcmp(name, "scale_type") == 0) {
        attribute_id = H5Aopen_name(loc_id, name);
        status = H5Aread(attribute_id, H5T_NATIVE_INT, 
                         &piv_post_par->scale_type); 
        status = H5Aclose(attribute_id); 
        piv_post_par->scale_type_logic = TRUE;
    }

    return 0;
}



/*
 * Public functions
 */

void
gpiv_post_parameters_logic(GpivPostPar * piv_post_par,
                           gboolean flag
                           )
/*-----------------------------------------------------------------------------
 * Set flag for piv_post_par _logic */
{
    piv_post_par->operator_manipiv_logic = flag;
    piv_post_par->set_logic = flag;
    piv_post_par->set_dx_logic = flag;
    piv_post_par->set_dy_logic = flag;
    piv_post_par->subtract_logic = flag;
    piv_post_par->z_off_dx_logic = flag;
    piv_post_par->z_off_dy_logic = flag;
    piv_post_par->diff_type_logic = flag;
    piv_post_par->operator_vorstra_logic = flag;
    piv_post_par->scale_type_logic = flag;
}



void
gpiv_post_default_parameters(GpivPostPar *piv_post_par_default,
                             gboolean force
                             )
/*-----------------------------------------------------------------------------
 * Default parameter values
 */
{
    if (!piv_post_par_default->operator_manipiv_logic || force)
        piv_post_par_default->operator_manipiv = 
            GPIV_GPIVPOSTPAR_DEFAULT__OPERATOR_MANIPIV;
    if (!piv_post_par_default->set_logic || force)
        piv_post_par_default->set = GPIV_GPIVPOSTPAR_DEFAULT__SET;
    if (!piv_post_par_default->set_dx_logic || force)
        piv_post_par_default->set_dx = GPIV_GPIVPOSTPAR_DEFAULT__SET_DX;
    if (!piv_post_par_default->set_dy_logic || force)
        piv_post_par_default->set_dy = GPIV_GPIVPOSTPAR_DEFAULT__SET_DY;
    if (!piv_post_par_default->subtract_logic || force)
        piv_post_par_default->subtract = GPIV_GPIVPOSTPAR_DEFAULT__SUBTRACT;
    if (!piv_post_par_default->z_off_dx_logic || force) 
        piv_post_par_default->z_off_dx = GPIV_GPIVPOSTPAR_DEFAULT__Z_OFF_DX;
    if (!piv_post_par_default->z_off_dy_logic || force) 
        piv_post_par_default->z_off_dy = GPIV_GPIVPOSTPAR_DEFAULT__Z_OFF_DY;
    if (!piv_post_par_default->diff_type_logic || force)
        piv_post_par_default->diff_type = GPIV_GPIVPOSTPAR_DEFAULT__DIFF_TYPE;
    if (!piv_post_par_default->operator_vorstra_logic || force)
        piv_post_par_default->operator_vorstra = 
            GPIV_GPIVPOSTPAR_DEFAULT__OPERATOR_VORSTRA;
    if (!piv_post_par_default->scale_type_logic || force)
        piv_post_par_default->scale_type = GPIV_GPIVPOSTPAR_DEFAULT__SCALE_TYPE;
    
    gpiv_post_parameters_logic(piv_post_par_default, TRUE);
}



void 
gpiv_post_read_parameters(FILE * fp_par, 
                          GpivPostPar * piv_post_par,
                          int print_par
                          )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Read all parameters for PIV post processing 
 *     if not defined by command line keys
 *
 * PROTOTYPE LOCATATION:
 *     post.h
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    char line[GPIV_MAX_CHARS], par_name[GPIV_MAX_CHARS];


    while (fgets(line, GPIV_MAX_CHARS, fp_par) != NULL) {
        if (line[0] != '#' && line[0] != '\n' && line[0] != ' ' 
            && line[0] != '\t') {
            sscanf(line,"%s",par_name);

/*
 * Parameters for manipulating PIV data
 */
            if (piv_post_par->operator_manipiv_logic == FALSE) {
                piv_post_par->operator_manipiv_logic = 
                    gpiv_scan_iph(GPIV_POST_PAR_KEY, line, par_name, "Operator_manipiv:",
                                  &piv_post_par->operator_manipiv, print_par, 1);
            }
            
            if (piv_post_par->set_logic == FALSE) {
                piv_post_par->set_logic = 
                    gpiv_scan_iph(GPIV_POST_PAR_KEY, line, par_name, "Set:",
                                  &piv_post_par->set, print_par, 1);
            }
            
            if (piv_post_par->set_dx_logic == FALSE) {
                piv_post_par->set_dx_logic = 
                    gpiv_scan_fph(GPIV_POST_PAR_KEY, line, par_name, "Set_dx:",
                                  &piv_post_par->set_dx, print_par, 1);
            }
            
            if (piv_post_par->set_dy_logic == FALSE) {
                piv_post_par->set_dy_logic = 
                    gpiv_scan_fph(GPIV_POST_PAR_KEY, line, par_name, "Set_dy:",
                                  &piv_post_par->set_dy, print_par, 1);
            }

/*
 * Parameters for s-avg
 */
            if (piv_post_par->subtract_logic == FALSE) {
                piv_post_par->subtract_logic = 
                    gpiv_scan_iph(GPIV_POST_PAR_KEY,line,par_name, "Subtract:",
                                  &piv_post_par->subtract, print_par, 1);
            }
            
            if (piv_post_par->z_off_dx_logic == FALSE) {
                piv_post_par->z_off_dx_logic = 
                    gpiv_scan_fph(GPIV_POST_PAR_KEY, line, par_name, 
                                  "Zoff_dx:",
                                  &piv_post_par->z_off_dx, print_par, 1);
            }
            
            if (piv_post_par->z_off_dy_logic == FALSE) {
                piv_post_par->z_off_dy_logic = 
                    gpiv_scan_fph(GPIV_POST_PAR_KEY, line, par_name, 
                                  "Zoff_dy:",
                                  &piv_post_par->z_off_dy, print_par, 1);
            }

/*
 * Parameters for vorticity and strain
 */
            if (piv_post_par->diff_type_logic == FALSE) {
                piv_post_par->diff_type_logic = 
                    gpiv_scan_iph(GPIV_POST_PAR_KEY, line, par_name, 
                                  "Differential_Type:", 
                                  &piv_post_par->diff_type, print_par, 1);
            }
            
            if (piv_post_par->operator_vorstra_logic == FALSE) {
                piv_post_par->operator_vorstra_logic = 
                    gpiv_scan_iph(GPIV_POST_PAR_KEY, line, par_name, 
                                  "Operator_vorstra:", 
                                  &piv_post_par->operator_vorstra, print_par, 1);
            }
            
/*
 * Parameters for scaling PIV data.
 */
            if (piv_post_par->scale_type_logic == FALSE) {
                piv_post_par->scale_type_logic = 
                    gpiv_scan_iph(GPIV_POST_PAR_KEY, line, par_name, 
                                  "Scale_Type:", 
                                  &piv_post_par->scale_type, print_par, 1);
            }
            
        }
    }
    
}



char *
gpiv_post_check_parameters_read(GpivPostPar *piv_post_par,
                                GpivPostPar piv_post_par_default
                                )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Check out if all parameters have been read
 *     Returns: NULL on success or *err_msg on failure
 *
 * PROTOTYPE LOCATATION:
 *     post.h
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    char * err_msg = NULL;

/*
 * Parameters for manipiv
 */
    if (piv_post_par->operator_manipiv_logic == FALSE) {
        piv_post_par->operator_manipiv = piv_post_par_default.operator_manipiv;
	err_msg = "Using default: ";
        gpiv_warning("%s\nPost.Operator_manipiv: %d", 
                     err_msg, piv_post_par->operator_manipiv);
    }


    if (piv_post_par->set_logic == FALSE) {
        piv_post_par->set = piv_post_par_default.set;
	err_msg = "Using default:";
        gpiv_warning("%s\nPost.Set: %d", err_msg, piv_post_par->set);
    }


    if (piv_post_par->set_dx_logic == FALSE) {
        piv_post_par->set_dx = piv_post_par_default.set_dx;
	err_msg = "Using default:";
        gpiv_warning("%s\nPost.Set_dx: %f", err_msg, piv_post_par->set_dx);
    }


    if (piv_post_par->set_dy_logic == FALSE) {
        piv_post_par->set_dy = piv_post_par_default.set_dy;
	err_msg = "Using default:";
        gpiv_warning("%s\nPost.Set_dy: %f", err_msg, piv_post_par->set_dy);
    }


/*
 * Parameters of s-avg
 */
   if (piv_post_par->subtract_logic == FALSE) {
       piv_post_par->subtract = piv_post_par_default.subtract;
       err_msg = "Using default:";
        gpiv_warning("%s\nPost.Subtract: %d", err_msg, piv_post_par->subtract);
   }

   if (piv_post_par->z_off_dx_logic==0) {
        piv_post_par->z_off_dx = piv_post_par_default.z_off_dx;
	err_msg = "Using default: ";
        gpiv_warning("%s\nPost.Zoff_dx: %f", 
                     err_msg, piv_post_par->z_off_dx);
   }

   if (piv_post_par->z_off_dy_logic==0) {
        piv_post_par->z_off_dy = piv_post_par_default.z_off_dy;
	err_msg = "Using default: ";
        gpiv_warning("%s\nPost.Zoff_dy: %f", 
                     err_msg, piv_post_par->z_off_dy);
   }


/*
 * Parameters for vorstra
 */
    if (piv_post_par->diff_type_logic == FALSE) {
        piv_post_par->diff_type = piv_post_par_default.diff_type;
	err_msg = "Using default:";
        gpiv_warning("%s\nPost.Differential_Type: %d", 
                     err_msg, piv_post_par->diff_type);
    }

    if (piv_post_par->operator_vorstra_logic == FALSE) {
        piv_post_par->operator_vorstra = piv_post_par_default.operator_vorstra;
	err_msg = "Using default:";
        gpiv_warning("%s\nPost.Operator_vorstra: %d", 
                     err_msg, piv_post_par->operator_vorstra);
    }

/*
 * Parameters for scale
 */
    if (piv_post_par->scale_type_logic == FALSE) {
        piv_post_par->scale_type = piv_post_par_default.scale_type;
	err_msg = "Using default:";
        gpiv_warning("%s\nPost.Scale_Type: %d", 
                     err_msg, piv_post_par->operator_vorstra);
    }

    err_msg = NULL;
    return err_msg;
}



/* void gpiv_post_test_parameter(PivPostpar piv_post_par) */
/* ----------------------------------------------------------------------------
 * Testing parameters on valid values and initializing derived variables
 */
/* { */

/* } */





void 
gpiv_post_print_parameters(GpivPostPar piv_post_par
                           )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Prints parameters to stdout
 *
 * PROTOTYPE LOCATATION:
 *     post.h
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{

/*
 * Parameters for manipiv
 */
    if (piv_post_par.operator_manipiv_logic)
        printf("%s%s  %d\n", GPIV_POST_PAR_KEY, "Operator_manipiv:", 
               piv_post_par.operator_manipiv);
    
    if (piv_post_par.set_logic == TRUE)
        printf("%s%s  %d\n", GPIV_POST_PAR_KEY, "Set:", piv_post_par.set);
    
    if (piv_post_par.set_dx_logic == TRUE)
        printf("%s%s  %f\n", GPIV_POST_PAR_KEY, "Set_dx:", piv_post_par.set_dx);
    
    if (piv_post_par.set_dy_logic == TRUE)
        printf("%s%s  %f\n", GPIV_POST_PAR_KEY, "Set_dy:", piv_post_par.set_dy);

/*
 * Parameters of s-avg
 */
    if (piv_post_par.subtract_logic) 
        printf("%s%s %d\n",GPIV_POST_PAR_KEY, "Subtract:", 
               piv_post_par.subtract);
    
    if (piv_post_par.z_off_dx_logic) 
        printf("%s%s %f\n",GPIV_POST_PAR_KEY, "Zoff_dx:", 
               piv_post_par.z_off_dx);
    
    if (piv_post_par.z_off_dy_logic) 
        printf("%s%s %f\n",GPIV_POST_PAR_KEY, "Zoff_dy:", 
               piv_post_par.z_off_dy);
    
  
/*
 * Parameters for vorstra
 */
    if (piv_post_par.operator_vorstra_logic)
	printf("%s%s %d\n", GPIV_POST_PAR_KEY, "Operator_vorstra:", 
	       piv_post_par.operator_vorstra);

    if (piv_post_par.diff_type_logic)
	printf("%s%s %d\n", GPIV_POST_PAR_KEY, "Differential_Type:", 
	       piv_post_par.diff_type);


/*
 * Parameter for scale
 */
    if (piv_post_par.scale_type_logic)
	printf("%s%s %d\n", GPIV_POST_PAR_KEY, "Scale_Type:", 
	       piv_post_par.scale_type);


    printf("\n");
}



void 
gpiv_post_fprint_parameters(FILE * fp,
                            GpivPostPar piv_post_par
                            )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Prints parameters to fp
 *
 * PROTOTYPE LOCATATION:
 *     post.h
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{

/*
 * Parameters for manipiv
 */
    if (piv_post_par.operator_manipiv_logic)
        fprintf(fp,"%s%s  %d\n", GPIV_POST_PAR_KEY, "Operator_manipiv:", 
                piv_post_par.operator_manipiv);
    
    if (piv_post_par.set_logic)
        fprintf(fp,"%s%s  %d\n", GPIV_POST_PAR_KEY, "Set:", 
                piv_post_par.set);
    
    if (piv_post_par.set_dx_logic)
        fprintf(fp,"%s%s  %f\n", GPIV_POST_PAR_KEY, "Set_dx:", 
                piv_post_par.set_dx);
    
    if (piv_post_par.set_dy_logic)
        fprintf(fp,"%s%s  %f\n", GPIV_POST_PAR_KEY, "Set_dy:", 
                piv_post_par.set_dy);
    
/*
 * Parameters of s-avg
 */
    if (piv_post_par.subtract_logic) 
        fprintf(fp,"%s%s %d\n",GPIV_POST_PAR_KEY, "Subtract:", 
                piv_post_par.subtract);
    
    if (piv_post_par.z_off_dx_logic) 
        fprintf(fp,"%s%s %f\n",GPIV_POST_PAR_KEY, "Zoff_dx:", 
                piv_post_par.z_off_dx);
    
    if (piv_post_par.z_off_dy_logic) 
        fprintf(fp,"%s%s %f\n",GPIV_POST_PAR_KEY, "Zoff_dy:", 
                piv_post_par.z_off_dy);

/*
 * Parameters for vorstra
 */
    if (piv_post_par.operator_vorstra_logic)
	fprintf(fp, "%s%s %d\n", GPIV_POST_PAR_KEY, "Operator_vorstra:", 
		piv_post_par.operator_vorstra);
    
    if (piv_post_par.diff_type_logic)
	fprintf(fp, "%s%s %d\n", GPIV_POST_PAR_KEY, "Differential_Type:",
		piv_post_par.diff_type);

/*
 * Parameter for scale
 */
    if (piv_post_par.scale_type_logic)
	fprintf(fp, "%s%s %d\n", GPIV_POST_PAR_KEY, "Scale_Type:", 
                piv_post_par.scale_type);
    
    
    fprintf(fp, "\n");
    
}



char *
gpiv_post_fread_hdf5_parameters(char *fname, 
                                GpivPostPar * piv_post_par 
                                )
/* --------------------------------------------------------------------------- 
 * DESCRIPTION:
 *     Reads post parameters from hdf5 data file
 *
 * PROTOTYPE LOCATATION:
 *     post.h
 *
 * INPUTS:
 *     group_id:       hdf group identity
 *
 * OUTPUTS:
 *     piv_post_par:   parameter structure
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *-------------------------------------------------------------------------- */
{
    char *err_msg = NULL;
    int i;
/*
 * HDF declarations
 */
    hid_t       file_id, group_id, attribute_id; 
    herr_t      status;

    if ((i = H5Fis_hdf5(fname)) == 0)  {
        err_msg = "GPIV_POST_FREAD_HDF5_PARAMETERS: not an hdf5 file";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }
    file_id = H5Fopen(fname, H5F_ACC_RDONLY, H5P_DEFAULT);

    group_id = H5Gopen (file_id, "PIV");
    H5Aiterate(group_id, NULL, attr_info, piv_post_par);
    status = H5Gclose (group_id);

    group_id = H5Gopen (file_id, "SCALARS");
    H5Aiterate(group_id, NULL, attr_info, piv_post_par);
    status = H5Gclose (group_id);

    status = H5Fclose(file_id); 
    return err_msg;
}



char * 
gpiv_post_fwrite_hdf5_parameters(char *fname, 
                                GpivPostPar * piv_post_par 
                                )
/*---------------------------------------------------------------------------- 
 * DESCRIPTION:
 *     Writes post parameters to an existing hdf5 data file
 *
 * PROTOTYPE LOCATATION:
 *     post.h
 *
 * INPUTS:
 *     group_id:       hdf group identity
 *     piv_eval_par:   parameter structure
 *
 * OUTPUTS:
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *-------------------------------------------------------------------------- */
{
    char *err_msg = NULL;
    int i;
/*
 * HDF declarations
 */
    hid_t       file_id, dataspace_id, group_id, attribute_id; 
    hsize_t     dims[1];
    herr_t      status;

    if ((i = H5Fis_hdf5(fname)) == 0)  {
        err_msg = "GPIV_POST_FWRITE_HDF5_PARAMETERS: not an hdf5 file";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }
    file_id = H5Fopen(fname, H5F_ACC_RDWR, H5P_DEFAULT);
    group_id = H5Gopen (file_id, "PIV");



    dims[0] = 1;
    dataspace_id = H5Screate_simple(1, dims, NULL);

/*
 * Parameters of manipiv
 */
    if (piv_post_par->operator_manipiv_logic) {
        attribute_id = H5Acreate(group_id, "operator_manipiv", H5T_NATIVE_INT, 
                                 dataspace_id, H5P_DEFAULT);
        status = H5Awrite(attribute_id, H5T_NATIVE_INT, 
                          &piv_post_par->operator_manipiv); 
        status = H5Aclose(attribute_id); 
    }

/*
 * parameters of s-avg
 */
    if (piv_post_par->subtract_logic) {
        attribute_id = H5Acreate(group_id, "subtract", H5T_NATIVE_INT, 
                                 dataspace_id, H5P_DEFAULT);
        status = H5Awrite(attribute_id, H5T_NATIVE_INT, 
                          &piv_post_par->subtract); 
        status = H5Aclose(attribute_id); 
    }


    if (piv_post_par->z_off_dx_logic) {
        attribute_id = H5Acreate(group_id, "z_off_dx", H5T_NATIVE_FLOAT, 
                                 dataspace_id, H5P_DEFAULT);
        status = H5Awrite(attribute_id, H5T_NATIVE_FLOAT, 
                          &piv_post_par->z_off_dx); 
        status = H5Aclose(attribute_id); 
    }

    if (piv_post_par->z_off_dy_logic) {
        attribute_id = H5Acreate(group_id, "z_off_dy", H5T_NATIVE_FLOAT, 
                                 dataspace_id, H5P_DEFAULT);
        status = H5Awrite(attribute_id, H5T_NATIVE_FLOAT, 
                          &piv_post_par->z_off_dy); 
        status = H5Aclose(attribute_id); 
    }

    status = H5Gclose (group_id);

/*
 * Parameters of vorstra
 */
    group_id = H5Gopen (file_id, "SCALARS");

    if (piv_post_par->diff_type_logic) {
        attribute_id = H5Acreate(group_id, "diff_type", H5T_NATIVE_INT, 
                                 dataspace_id, H5P_DEFAULT);
        status = H5Awrite(attribute_id, H5T_NATIVE_INT, 
                          &piv_post_par->diff_type); 
        status = H5Aclose(attribute_id); 
    }

    if (piv_post_par->operator_vorstra_logic) {
        attribute_id = H5Acreate(group_id, "operator_vorstra", H5T_NATIVE_INT, 
                                 dataspace_id, H5P_DEFAULT);
        status = H5Awrite(attribute_id, H5T_NATIVE_INT, 
                          &piv_post_par->operator_vorstra); 
        status = H5Aclose(attribute_id); 
    }


/*
 * Parameters of scale
 */
    if (piv_post_par->scale_type_logic) {
        attribute_id = H5Acreate(group_id, "scale_type", H5T_NATIVE_INT, 
                                 dataspace_id, H5P_DEFAULT);
        status = H5Awrite(attribute_id, H5T_NATIVE_INT, 
                          &piv_post_par->scale_type); 
        status = H5Aclose(attribute_id); 
    }

    status = H5Sclose(dataspace_id);
    status = H5Gclose (group_id);
    status = H5Fclose(file_id); 
    return err_msg;
}
