/*******************************************************************************************************************************************
 cnotebook.c
*******************************************************************************************************************************************/

#include "cnotebook.h"

//------------------------------------------------------------------------------------------------------------------------------------------
// metaclass code resolution
//------------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_GENERIC_METACLASS (CNoteBookListener);

//------------------------------------------------------------------------------------------------------------------------------------------
// constructor
//------------------------------------------------------------------------------------------------------------------------------------------
CNoteBookListener::CNoteBookListener ()
{ }

//------------------------------------------------------------------------------------------------------------------------------------------
// destructor
//------------------------------------------------------------------------------------------------------------------------------------------
CNoteBookListener::~CNoteBookListener ()
{ }

//------------------------------------------------------------------------------------------------------------------------------------------
// CNoteBook::OnSwitchPage
//------------------------------------------------------------------------------------------------------------------------------------------
void CNoteBook::OnSwitchPage (GtkNotebook *, GtkNotebookPage *, guint inIndex, gpointer inData)
{
	// retreive the gtkol instance
	CNoteBook *inNoteBook = reinterpret_cast <CNoteBook *> (inData);

	// check pointer
	if (inNoteBook == NULL) return;

	// send the notification to listener if any
	if (inNoteBook -> GetListener() != NULL)
	{
		// get the note book children
		CWidgets inChildren (inNoteBook -> GetGtkChildren());

		// index check
		if (inIndex >= inChildren.GetLength()) return;

		// send the notification
		static_cast <CNoteBookListener *> (inNoteBook -> GetListener()) -> OnSwitch (inNoteBook, *inChildren[inIndex], inIndex);
	}
}

//------------------------------------------------------------------------------------------------------------------------------------------
// metaclass code resolution
//------------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_DYNAMIC_METACLASS (CNoteBook);

//------------------------------------------------------------------------------------------------------------------------------------------
// constructor
//------------------------------------------------------------------------------------------------------------------------------------------
CNoteBook::CNoteBook  (CContainer *inOwner, const CNoteBookListener *inListener)
	  :CContainer (inOwner, inListener)
{
	// widget instanciation handling requested
	if (inOwner != NULL) CWidget::CreateWidget (this);
}

//------------------------------------------------------------------------------------------------------------------------------------------
// destructor
//------------------------------------------------------------------------------------------------------------------------------------------
CNoteBook::~CNoteBook ()
{
	// deletion coherence requested
	CWidget::DestroyWidget (this);
}

//------------------------------------------------------------------------------------------------------------------------------------------
// gtk instanciation
//------------------------------------------------------------------------------------------------------------------------------------------
GtkWidget * CNoteBook::PerformWidgetInstanciate ()
{
	// ok
	return ::gtk_notebook_new();
}

//------------------------------------------------------------------------------------------------------------------------------------------
// gtk initialization
//------------------------------------------------------------------------------------------------------------------------------------------
void CNoteBook::PerformWidgetInitialize ()
{
	// get our gtk widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// check pointer
	if (inGtkWidget == NULL) return;

	// signal connection
	::g_signal_connect (G_OBJECT(inGtkWidget), "switch-page", G_CALLBACK(CNoteBook::OnSwitchPage), this);

	// show the job
	Show();
}

//------------------------------------------------------------------------------------------------------------------------------------------
// perform container add
//------------------------------------------------------------------------------------------------------------------------------------------
void CNoteBook::PerformContainerAdd (CWidget *inChild)
{
	// get our widget
	GtkWidget *inGtkWidget (GetGtkWidget());	

	// check pointers
	if (inGtkWidget == NULL || inChild == NULL || inChild -> GetGtkWidget() == NULL) return;

	// get the specified child in owner gtkol index
	SInt16 inIndex = CContainer::GetGtkInOwnerIndex (this, inChild);

	// insert into as new page
	if (inIndex != -1) ::gtk_notebook_insert_page (GTK_NOTEBOOK(inGtkWidget), inChild -> GetGtkWidget(), NULL, inIndex);
}

//------------------------------------------------------------------------------------------------------------------------------------------
// perform container remove
//------------------------------------------------------------------------------------------------------------------------------------------
void CNoteBook::PerformContainerRemove (CWidget *inChild)
{
	// get our widget
	GtkWidget *inGtkWidget (GetGtkWidget());	

	// check pointers
	if (inGtkWidget == NULL || inChild == NULL || inChild -> GetGtkWidget() == NULL) return;

	// get the specified child in owner gtkol index
	SInt16 inIndex = CContainer::GetGtkInOwnerIndex (this, inChild);

	// remove the page
	if (inIndex != -1) ::gtk_notebook_remove_page (GTK_NOTEBOOK(inGtkWidget), inIndex);
}

//------------------------------------------------------------------------------------------------------------------------------------------
// listener affectation
//------------------------------------------------------------------------------------------------------------------------------------------
const CMetaClass * CNoteBook::ListenerMustBe () const
{
	return __metaclass(CNoteBookListener);
}

//------------------------------------------------------------------------------------------------------------------------------------------
// expected owner type
//------------------------------------------------------------------------------------------------------------------------------------------
CMetaClasses CNoteBook::OwnerMustBe () const
{
	return __metaclasses(CContainer);
}

//------------------------------------------------------------------------------------------------------------------------------------------
// children number left
//------------------------------------------------------------------------------------------------------------------------------------------
SInt16 CNoteBook::GetGtkChildrenNumberLeft () const
{
	// unlimited handling specification
	return -1;
}

//------------------------------------------------------------------------------------------------------------------------------------------
// set show tabs
//------------------------------------------------------------------------------------------------------------------------------------------
void CNoteBook::SetShowTabs (const Bool inShow)
{
	// get our gtk widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// ok
	if (inGtkWidget != NULL) ::gtk_notebook_set_show_tabs (GTK_NOTEBOOK(inGtkWidget), inShow);
}

//------------------------------------------------------------------------------------------------------------------------------------------
// get show tabs
//------------------------------------------------------------------------------------------------------------------------------------------
Bool CNoteBook::GetShowTabs () const
{
	// get our gtk widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// ok
	return inGtkWidget != NULL ? ::gtk_notebook_get_show_tabs (GTK_NOTEBOOK(inGtkWidget)) : true;
}

//------------------------------------------------------------------------------------------------------------------------------------------
// set show border
//------------------------------------------------------------------------------------------------------------------------------------------
void CNoteBook::SetShowBorder (const Bool inShow)
{
	// get our gtk widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// ok
	if (inGtkWidget != NULL) ::gtk_notebook_set_show_border (GTK_NOTEBOOK(inGtkWidget), inShow);
}

//------------------------------------------------------------------------------------------------------------------------------------------
// get show border
//------------------------------------------------------------------------------------------------------------------------------------------
Bool CNoteBook::GetShowBorder () const
{
	// get our gtk widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// ok
	return inGtkWidget != NULL ? ::gtk_notebook_get_show_border (GTK_NOTEBOOK(inGtkWidget)) : true;
}

//------------------------------------------------------------------------------------------------------------------------------------------
// set scrollable
//------------------------------------------------------------------------------------------------------------------------------------------
void CNoteBook::SetTabScrollable (const Bool inScrollable)
{
	// get our gtk widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// ok
	if (inGtkWidget != NULL) ::gtk_notebook_set_scrollable (GTK_NOTEBOOK(inGtkWidget), inScrollable);
}

//------------------------------------------------------------------------------------------------------------------------------------------
// get scrollable
//------------------------------------------------------------------------------------------------------------------------------------------
Bool CNoteBook::GetTabScrollable () const
{
	// get our gtk widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// ok
	return inGtkWidget != NULL ? ::gtk_notebook_get_scrollable (GTK_NOTEBOOK(inGtkWidget)) : false;
}

//------------------------------------------------------------------------------------------------------------------------------------------
// tab caption affectation
//------------------------------------------------------------------------------------------------------------------------------------------
void CNoteBook::SetTabCaption (const CString &inCaption, const CWidget *inChild)
{
	// get our gtk widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// pointers check
	if (inGtkWidget == NULL || inChild == NULL || inChild -> GetGtkWidget() == NULL) return;

	// affectation
	::gtk_notebook_set_tab_label_text (GTK_NOTEBOOK(inGtkWidget), inChild -> GetGtkWidget(), inCaption.Get());
}

//------------------------------------------------------------------------------------------------------------------------------------------
// tab caption affectation
//------------------------------------------------------------------------------------------------------------------------------------------
void CNoteBook::SetTabCaption (const CString &inCaption, const UInt16 inIndex)
{
	// get our gtk widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// get the children
	CWidgets inChildren (GetGtkChildren());

	// index check
	if (inIndex >= inChildren.GetLength()) return;

	// get the child we are interested in
	CWidget *inChild = *inChildren[inIndex];

	// pointers check
	if (inGtkWidget == NULL || inChild == NULL || inChild -> GetGtkWidget() == NULL) return;

	// affectation
	::gtk_notebook_set_tab_label_text (GTK_NOTEBOOK(inGtkWidget), inChild -> GetGtkWidget(), inCaption.Get());
}

//------------------------------------------------------------------------------------------------------------------------------------------
// tab caption access
//------------------------------------------------------------------------------------------------------------------------------------------
CString CNoteBook::GetTabCaption (const CWidget *inChild) const
{
	// get our gtk widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// pointers check
	if (inGtkWidget == NULL || inChild == NULL || inChild -> GetGtkWidget() == NULL) return CString();

	// request it
	return CString (::gtk_notebook_get_tab_label_text (GTK_NOTEBOOK(inGtkWidget), inChild -> GetGtkWidget()));
}

//------------------------------------------------------------------------------------------------------------------------------------------
// tab caption access
//------------------------------------------------------------------------------------------------------------------------------------------
CString CNoteBook::GetTabCaption (const UInt16 inIndex) const
{
	// get our gtk widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// get the children
	CWidgets inChildren (GetGtkChildren());

	// index check
	if (inIndex >= inChildren.GetLength()) return CString();

	// get the child we are interested in
	CWidget *inChild = *inChildren[inIndex];

	// pointers check
	if (inGtkWidget == NULL || inChild == NULL || inChild -> GetGtkWidget() == NULL) return CString();

	// request it
	return CString (::gtk_notebook_get_tab_label_text (GTK_NOTEBOOK(inGtkWidget), inChild -> GetGtkWidget()));
}

//------------------------------------------------------------------------------------------------------------------------------------------
// current page affectation
//------------------------------------------------------------------------------------------------------------------------------------------
void CNoteBook::SetCurrentPage (const SInt16 inIndex)
{
	// get our gtk widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// affectation
	if (inGtkWidget != NULL) ::gtk_notebook_set_current_page (GTK_NOTEBOOK(inGtkWidget), inIndex);
}

//------------------------------------------------------------------------------------------------------------------------------------------
// current page access
//------------------------------------------------------------------------------------------------------------------------------------------
SInt16 CNoteBook::GetCurrentPage () const
{
	// get our gtk widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// ok
	return inGtkWidget != NULL ? ::gtk_notebook_get_current_page (GTK_NOTEBOOK(inGtkWidget)) : -1;
}

//------------------------------------------------------------------------------------------------------------------------------------------
// tab position affectation
//------------------------------------------------------------------------------------------------------------------------------------------
void CNoteBook::SetTabPosition (const GtkPositionType inGtkPositionType)
{
	// get our gtk widget
	GtkWidget *inGtkWidget (GetGtkWidget());
	
	// affectation
	if (inGtkWidget != NULL) ::gtk_notebook_set_tab_pos (GTK_NOTEBOOK(inGtkWidget), inGtkPositionType);
}

//------------------------------------------------------------------------------------------------------------------------------------------
// tab position access
//------------------------------------------------------------------------------------------------------------------------------------------
GtkPositionType CNoteBook::GetTabPosition () const
{
	// get our gtk widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// ok
	return inGtkWidget != NULL ? ::gtk_notebook_get_tab_pos (GTK_NOTEBOOK(inGtkWidget)) : GTK_POS_LEFT;
}

//------------------------------------------------------------------------------------------------------------------------------------------
// serialization
//------------------------------------------------------------------------------------------------------------------------------------------
void CNoteBook::Serialize (CXMLElementNode *&ioXMLElementNode, const int inMode) THROWABLE
{
	// generic call
	CContainer::Serialize (ioXMLElementNode, inMode);

	// request analyse
	switch (inMode)
	{
		// xml dump
		case XML_WRITE :
		{
			// create a new xml element node for this serialization level
			CXMLElement *newXMLElement = new CXMLElement (ioXMLElementNode, XML_NOTEBOOK_ELEMENT);

			// write the attributes
			newXMLElement -> AddAttribute (XML_NOTEBOOK_ATTR_SHOW_TABS, 	 CString(GetShowTabs()?	    "true":"false"));
			newXMLElement -> AddAttribute (XML_NOTEBOOK_ATTR_SHOW_BORDER, 	 CString(GetShowBorder()?   "true":"false"));
			newXMLElement -> AddAttribute (XML_NOTEBOOK_ATTR_TAB_SCROLLABLE, CString(GetTabScrollable()?"true":"false"));
			newXMLElement -> AddAttribute (XML_NOTEBOOK_ATTR_CURRENT_PAGE,   CString((SInt32)GetCurrentPage()));
			switch (GetTabPosition())
			{
				case GTK_POS_LEFT :
					newXMLElement -> AddAttribute (XML_NOTEBOOK_ATTR_TAB_POSITION, CString("left"));
					break;
				case GTK_POS_RIGHT :
					newXMLElement -> AddAttribute (XML_NOTEBOOK_ATTR_TAB_POSITION, CString("right"));
					break;
				case GTK_POS_TOP :
					newXMLElement -> AddAttribute (XML_NOTEBOOK_ATTR_TAB_POSITION, CString("top"));
					break;
				case GTK_POS_BOTTOM :
					newXMLElement -> AddAttribute (XML_NOTEBOOK_ATTR_TAB_POSITION, CString("bottom"));
					break;
			}

			// modify the io xml element node so that the potential overloaded definitions will continue under the current node
			ioXMLElementNode = newXMLElement -> GetXMLElementNode ();

			// instanciate the tab element
			newXMLElement = new CXMLElement (ioXMLElementNode, XML_NOTEBOOK_ELEMENT_TABS);
			
			// get the children of the notebook that is the tabs
			CWidgets inChildren (GetGtkChildren());
	
			// foreach of the children, keep the tab associated caption
			for (size_t i=0; i<inChildren.GetLength(); i++)
				new CXMLElement (newXMLElement->GetXMLElementNode(), XML_NOTEBOOK_ELEMENT_TAB_CAPTION, 
						 GetTabCaption(*inChildren[i]));
		}
		break;

		// xml load
		case XML_READ :
		{
			// get the child node this serialization process is interested in
			CXMLElementNode *inXMLNode = ::xml_node_get_child (ioXMLElementNode, XML_NOTEBOOK_ELEMENT);

			// check we got an expected cbutton node
			if (inXMLNode == NULL)
				throw new CException (CString("CNoteBook::Serialize, specified xml node is not a \"") + 
							      XML_NOTEBOOK_ELEMENT + CString("\" element one."), __exception(XMLPARSE));

			// modify the in/out element node pointer so that the serialization process will continue under the current node
			ioXMLElementNode = inXMLNode;

			// retreive and set the notebook attributes
			SetShowTabs  	 (::xml_node_get_attribute (inXMLNode, XML_NOTEBOOK_ATTR_SHOW_TABS).	 GetValue().ToBool());
			SetShowBorder 	 (::xml_node_get_attribute (inXMLNode, XML_NOTEBOOK_ATTR_SHOW_BORDER).	 GetValue().ToBool());
			SetTabScrollable (::xml_node_get_attribute (inXMLNode, XML_NOTEBOOK_ATTR_TAB_SCROLLABLE).GetValue().ToBool());
			SetCurrentPage	 (::xml_node_get_attribute (inXMLNode, XML_NOTEBOOK_ATTR_CURRENT_PAGE).	 GetValue().ToLong());
			CString inPosition (::xml_node_get_attribute (inXMLNode, XML_NOTEBOOK_ATTR_TAB_POSITION).GetValue());
			     if (inPosition == CString("left"))   SetTabPosition (GTK_POS_LEFT);
			else if (inPosition == CString("right"))  SetTabPosition (GTK_POS_RIGHT);
			else if (inPosition == CString("bottom")) SetTabPosition (GTK_POS_BOTTOM);
			else    				  SetTabPosition (GTK_POS_TOP);

			// get the tabs element node of the current one
			inXMLNode = ::xml_node_get_child (inXMLNode, XML_NOTEBOOK_ELEMENT_TABS);

			// check we got it 
			if (inXMLNode != NULL)
			{
				// foreach of the children
				for (size_t i=0, j=0; i<xml_node_get_children_number(inXMLNode); i++)
				{
					// get the child we are interested in
					CXMLElementNode *inChild = ::xml_node_get_child (inXMLNode, i);

					// check the node name
					if (::xml_node_get_name (inChild) == XML_NOTEBOOK_ELEMENT_TAB_CAPTION)
					{
						// retreive and set the tab caption
						if (::xml_node_get_value (inChild) != CString())
							SetTabCaption (::xml_node_get_value (inChild), (UInt16)j); j++;
					}
				}
			}
		}
		break;
	}
}


