# INDI Alignment Subsystem

## Introduction
The INDI alignment subsystem is a collection of classes that together provide support for telescope alignment using a database of stored sync points. Support is also provided for "Math Plugin Modules". One of these runtime loadable modules is active at any one time. The currently loaded module uses the sync point database to provide conversion functions to and from coordinates in the celestial reference frame and the telescope mount's local reference frame.

During observing runs the sync point database is held in memory within the INDI device driver. It can also be loaded and saved to and from a file on the system the driver is running. The database can be edited via INDI properties (for details of the properties see the class MapPropertiesToInMemoryDatabase), by an API class for use in INDI drivers(InMemoryDatabase), or an API class for use in INDI clients(ClientAPIForAlignmentDatabase).

The current math plugin module can be selected and initialised via INDI properties (for details of the properties see the class MathPluginManagement), by and API class for use in INDI drivers(MathPluginManagement), or by an API class for use in INDI clients(ClientAPIForMathPluginManagement).

## Math Plugins
The following math plugins are included in the first release.

### Built in math plugin
This is the default plugin which is used if no other plugin has been loaded. The plugin is normally initialised or re-initialised when the database has been loaded or when a new sync point has been added. The initialisation process scans the current database and builds a number of transformation matrices depending on how many sync points are present. Before the matrices are computed all celestial reference frame Right Ascension and Declination coordinates are transformed to horizontal Altitude Azimuth coordinates using the julian date stored in the sync point entry. This means that all transformations using the computed transformation matrices will be to and from a zenith aligned celestial reference frame and the telescope mounts local reference frame. This has the advantage of incorporating in the transformation any systematic alignment errors which occur due to the direction the mount is pointing relative to the zenith. Examples of such errors include those due to atmospheric refraction, and those due the effect of gravity on the telescope and mount. All transformation matrices are computed using the simple method proposed by [Toshimi Taki](http://www.geocities.jp/toshimi_taki/matrix/matrix_method_rev_e.pdf). This is quick and dirty but can result in matrices that are not true transforms. These will be reported as errors and an identity matrix will be substituted.

#### No sync points present

No action is taken. 

#### One sync point present

A transformation matrix is computed using a hint to mounts approximate alignment supplied by the driver, this can either be ZENITH, NORTH_CELESTIAL_POLE or SOUTH_CELESTIAL_POLE. The hint is used to make a dummy second sync point entry. A dummy third entry is computed from the cross product of the first two. A single transformation matrix and its inverse is computed from these three points. 

#### Two sync points present

A transformation matrix is computed using the two sync points and a dummy third sync point computed from the cross product of the first two. A single transformation matrix and its inverse is computed from these three points.

#### Three sync points present

A single transformation matrix and its inverse is computed from the three sync points.

#### Four or more sync points present

Two convex hulls are computed. One from the zenith aligned celestial reference frame sync point coordinates plus a dummy nadir, and the other from the mounts local reference frame sync point coordinates plus a dummy nadir. These convex hulls are made up of triangular facets. Forward and inverse transformation matrices are then computed for each corresponding pair of facets and stored alongside the facet in the relevant convex hull.

#### Coordinate conversion

If when the plugin is asked to translate a coordinate it only has a single conversion matrix (the one, two and three sync points case) this will be used. Otherwise (the four or more sync points case) a ray will shot from the origin of the requested source reference frame in the requested direction into the relevant convex hull and the transformation matrix from the facet it intersects will be used for the conversion.

### SVD math plugin
This plugin works in an identical manner to the built in math plugin. The only difference being that [Markley's Singular Value Decomposition algorithm](http://www.control.auc.dk/~tb/best/aug23-Bak-svdalg.pdf) is used to calculate the transformation matrices. This is a highly robust method and forms the basis of the pointing system used in many professional telescope installations.

## Using the Alignment Subsystem from KStars
The easiest way to use a telescope driver that supports the Alignment Subsystem is via an INDI aware client such as KStars. The following example uses the indi_SkywatcherAPIMount driver and a Synscan 114GT mount. If you are using a different driver then the name of that driver will appear in KStars not "skywatcherAPIMount".

1. Firstly connect the mount to the computer that is to run the driver. I use a readily available PL2303 chip based serial to USB converter cable.
2. From the handset utility menu select PC direct mode. As it is the computer that will be driving the mount not the handset, you can enter whatever values you want to get through the handset initialisation process.
3. Start indiserver and the indi_SkyWatcherAPIMount driver. Using the following command in a terminal:

        indiserver indi_skywatcherAPIMount

4. Start KStars and from the tools menu select "Devices" and then "Device Manager".
![Tools menu](toolsmenu.png)

5. In the device manager window select the "Client" tab, and in the client tab select the host that indiserver is running on. Click on connect.
![Device Manager](devicemanager.png)

6. An INDI Control Panel window should open with a skywatcherAPIMount tab. Select the "Options" subtab (I think I have invented this word!). Ensure that the port property is set to the correct serial device. My PL2303 usb cable always appears as /dev/ttyUSB0.
![INDI Control Panel](controlpanel1.png)

7. Select the "Main Control" tab and click on connect.
![INDI Control Panel](controlpanel2.png)

8. After a few seconds pause (whilst the driver determines what type of motor board is in use) a number of extra tabs should appear. One of these should be the "Site Management" tab. Select this and ensure that you have correct values entered for "Scope Location", you can safely ignore elevation at this time.
![INDI Control Panel](controlpanel3.png)

9. At this point it is probably wise to save the configuration. Return to the "Options" tab and click on "Configuration" "Save".
![INDI Control Panel](controlpanel4.png)

10. Check that the "Alignment" tab is present and select it. Using the controls on this tab you can view and manipulate the entries in the alignment database, and select which math plugin you want to use. It probably best to ignore this tab for the time being and use KStars to create sync points to align your mount.
![INDI Control Panel](controlpanel5.png)

11. To create a sync point using KStars. First ensure your target star is visible in the KStars display. I usually do this using the "Pointing...Find Object" tool.
![Find Object Tool](findobject.png)

12. Once you have the target in the KStars window right click on it and then hover your mouse over the "Sync" option in the "skywatcherAPIMount" sub-menu. Do not left click on the "Sync" option yet. N.B. The "Centre and Track" item in the main popup menu is nothing to do with your mount. It merely tells KStars to keep this object centered in the display window.
![Object popup menu](objectpopup.png)

13. Go back to your scope and centre the target in the eyepiece. Quickly get back to your computer and left click the mouse (be careful not to move it off the Sync menu item or you will have to right click to bring it up again). If you have been successful you should see the KStars telescope crosshairs displayed around the target.
![Crosshair Display](crosshair.png).

14. The Alignment Subsystem is now in "one star" alignment mode. You can try this out by right clicking on your target star or a nearby star and selecting "Track" from the "skywatcherAPIMount" sub-menu. The further away the object you track is from the sync point star the less accurate the initial slew will be and the more quickly the tracked star will drift off centre. To correct this you need to add more sync points.

15. To add another sync point you can select a new target star in KStars and use the slew command from the "skywatcherAPIMount" sub-menu to approximately slew your scope onto the target. The procedure for adding the sync point is the same as before. With the default math plugin one achieves maximum accuracy for a particular triangular patch of sky when it is surrounded by three sync points. If more than three sync points are defined then more triangular patches will be added to the mesh.

16. If would be very useful if you could collect information on how well the alignment mechanism holds a star centred, measured in degrees of drift per second. Please share these on the indi-devel list.

## Adding Alignment Subsystem support to an INDI driver
The Alignment Subsystem provides two API classes and a support function class for use in drivers. These are MapPropertiesToInMemoryDatabase, MathPluginManagement, and TelescopeDirectionVectorSupportFunctions. Driver developers can use these classes individually, however, the easiest way to use them is via the AlignmentSubsystemForDrivers class. To use this class simply ensure that is a parent of your driver class.

    class ScopeSim : public INDI::Telescope, public INDI::GuiderInterface, public INDI::AlignmentSubsystem::AlignmentSubsystemForDrivers

Somewhere in your drivers initProperties function add a call to AlignmentSubsystemForDrivers::InitProperties.

    bool ScopeSim::initProperties()
    {
        /* Make sure to init parent properties first */
        INDI::Telescope::initProperties();

        ...

        /* Add debug controls so we may debug driver if necessary */
        addDebugControl();

        // Add alignment properties
        InitProperties(this);

        return true;
    }

Hook the alignment subsystem into your drivers processing of properties by putting calls to AlignmentSubsystemForDrivers::ProcessNumberProperties,
AlignmentSubsystemForDrivers::ProcessSwitchProperties, AlignmentSubsystemForDrivers::ProcessBLOBProperties AlignmentSubsystemForDrivers::ProcessTextProperties, in the relevant routines.

    bool ScopeSim::ISNewNumber (const char *dev, const char *name, double values[], char *names[], int n)
    {
        //  first check if it's for our device

        if(strcmp(dev,getDeviceName())==0)
        {
            ...

            // Process alignment properties
            ProcessNumberProperties(this, name, values, names, n);

        }

        //  if we didn't process it, continue up the chain, let somebody else
        //  give it a shot
        return INDI::Telescope::ISNewNumber(dev,name,values,names,n);
    }

    bool ScopeSim::ISNewSwitch (const char *dev, const char *name, ISState *states, char *names[], int n)
    {
        if(strcmp(dev,getDeviceName())==0)
        {
            ...
            // Process alignment properties
            ProcessSwitchProperties(this, name, states, names, n);
        }

        //  Nobody has claimed this, so, ignore it
        return INDI::Telescope::ISNewSwitch(dev,name,states,names,n);
    }

    bool ScopeSim::ISNewBLOB (const char *dev, const char *name, int sizes[], int blobsizes[], char *blobs[], char *formats[], char *names[], int n)
    {
        if(strcmp(dev,getDeviceName())==0)
        {
            // Process alignment properties
            ProcessBlobProperties(this, name, sizes, blobsizes, blobs, formats, names, n);
        }
        // Pass it up the chain
        return INDI::Telescope::ISNewBLOB(dev, name, sizes, blobsizes, blobs, formats, names, n);
    }

    bool ScopeSim::ISNewText (const char *dev, const char *name, char *texts[], char *names[], int n)
    {
        if(strcmp(dev,getDeviceName())==0)
        {
            // Process alignment properties
            ProcessTextProperties(this, name, texts, names, n);
        }
        // Pass it up the chain
        return INDI::Telescope::ISNewText(dev, name, texts, names, n);
    }

The next step is to add the handling of sync points into your drivers Sync function.

TBD - add sample code. I will do this  for the scope simulator when I can work out what it is doing with snooped properties!

    bool SkywatcherAPIMount::Sync(double ra, double dec)
    {
        DEBUG(INDI::AlignmentSubsystem::DBG_ALIGNMENT, "SkywatcherAPIMount::Sync");

        // Compute a telescope direction vector from the current encoders
        if (!GetEncoder(AXIS1))
            return false;
        if (!GetEncoder(AXIS2))
            return false;

        // Might as well do this
        UpdateDetailedMountInformation(true);

        struct ln_hrz_posn AltAz;
        AltAz.alt = MicrostepsToDegrees(AXIS2, CurrentEncoders[AXIS2] - ZeroPositionEncoders[AXIS2]);
        DEBUGF(INDI::AlignmentSubsystem::DBG_ALIGNMENT, "Axis2 encoder %ld initial %ld alt(degrees) %lf", CurrentEncoders[AXIS2], ZeroPositionEncoders[AXIS2], AltAz.alt);
        AltAz.az = MicrostepsToDegrees(AXIS1, CurrentEncoders[AXIS1] - ZeroPositionEncoders[AXIS1]);
        DEBUGF(INDI::AlignmentSubsystem::DBG_ALIGNMENT, "Axis1 encoder %ld initial %ld az(degrees) %lf", CurrentEncoders[AXIS1], ZeroPositionEncoders[AXIS1], AltAz.az);

        AlignmentDatabaseEntry NewEntry;
    #ifdef USE_INITIAL_JULIAN_DATE
        NewEntry.ObservationJulianDate = InitialJulianDate;
    #else
        NewEntry.ObservationJulianDate = ln_get_julian_from_sys();
    #endif
        NewEntry.RightAscension = ra;
        NewEntry.Declination = dec;
        NewEntry.TelescopeDirection = TelescopeDirectionVectorFromAltitudeAzimuth(AltAz);
        NewEntry.PrivateDataSize = 0;

        DEBUGF(INDI::AlignmentSubsystem::DBG_ALIGNMENT, "New sync point Date %lf RA %lf DEC %lf TDV(x %lf y %lf z %lf)",
                        NewEntry.ObservationJulianDate, NewEntry.RightAscension, NewEntry.Declination,
                        NewEntry.TelescopeDirection.x, NewEntry.TelescopeDirection.y, NewEntry.TelescopeDirection.z);

        if (!CheckForDuplicateSyncPoint(NewEntry))
        {

            GetAlignmentDatabase().push_back(NewEntry);

            // Tell the client about size change
            UpdateSize();

            // Tell the math plugin to reinitialise
            Initialise(this);

            return true;
        }
        return false;
    }


The final step is to add coordinate conversion to ReadScopeStatus, TimerHit (for tracking), and Goto.

TBD - add sample code. I will do this  for the scope simulator when I can work out what it is doing with snooped properties!

    bool SkywatcherAPIMount::ReadScopeStatus()
    {
        DEBUG(INDI::AlignmentSubsystem::DBG_ALIGNMENT, "SkywatcherAPIMount::ReadScopeStatus");

        // Horrible hack to get over the fact that the base class calls ReadScopeStatus from inside Connect
        // before I have a chance to set up the serial port
        SetSerialPort(PortFD);

        // leave the following stuff in for the time being it is mostly harmless

        // Quick check of the mount
        if (!GetMotorBoardVersion(AXIS1))
            return false;

        if (!GetStatus(AXIS1))
            return false;

        if (!GetStatus(AXIS2))
            return false;

        // Update Axis Position
        if (!GetEncoder(AXIS1))
            return false;
        if (!GetEncoder(AXIS2))
            return false;

        UpdateDetailedMountInformation(true);

        // Calculate new RA DEC
        struct ln_hrz_posn AltAz;
        AltAz.alt = MicrostepsToDegrees(AXIS2, CurrentEncoders[AXIS2] - ZeroPositionEncoders[AXIS2]);
        DEBUGF(INDI::AlignmentSubsystem::DBG_ALIGNMENT, "Axis2 encoder %ld initial %ld alt(degrees) %lf", CurrentEncoders[AXIS2], ZeroPositionEncoders[AXIS2], AltAz.alt);
        AltAz.az = MicrostepsToDegrees(AXIS1, CurrentEncoders[AXIS1] - ZeroPositionEncoders[AXIS1]);
        DEBUGF(INDI::AlignmentSubsystem::DBG_ALIGNMENT, "Axis1 encoder %ld initial %ld az(degrees) %lf", CurrentEncoders[AXIS1], ZeroPositionEncoders[AXIS1], AltAz.az);
        TelescopeDirectionVector TDV = TelescopeDirectionVectorFromAltitudeAzimuth(AltAz);
        DEBUGF(INDI::AlignmentSubsystem::DBG_ALIGNMENT, "TDV x %lf y %lf z %lf", TDV.x, TDV.y, TDV.z);

        double RightAscension, Declination;
        if (TransformTelescopeToCelestial( TDV, RightAscension, Declination))
            DEBUG(INDI::AlignmentSubsystem::DBG_ALIGNMENT, "Conversion OK");
        else
        {
            bool HavePosition = false;
            ln_lnlat_posn Position;
            if ((NULL != IUFindNumber(&LocationNP, "LAT")) && ( 0 != IUFindNumber(&LocationNP, "LAT")->value)
                && (NULL != IUFindNumber(&LocationNP, "LONG")) && ( 0 != IUFindNumber(&LocationNP, "LONG")->value))
            {
                // I assume that being on the equator and exactly on the prime meridian is unlikely
                Position.lat = IUFindNumber(&LocationNP, "LAT")->value;
                Position.lng = IUFindNumber(&LocationNP, "LONG")->value;
                HavePosition = true;
            }
            struct ln_equ_posn EquatorialCoordinates;
            if (HavePosition)
            {
                TelescopeDirectionVector RotatedTDV(TDV);
                switch (GetApproximateMountAlignment())
                {
                    case ZENITH:
                        break;

                    case NORTH_CELESTIAL_POLE:
                        // Rotate the TDV coordinate system anticlockwise (positive) around the y axis by 90 minus
                        // the (positive)observatory latitude. The vector itself is rotated clockwise
                        RotatedTDV.RotateAroundY(90.0 - Position.lat);
                        AltitudeAzimuthFromTelescopeDirectionVector(RotatedTDV, AltAz);
                        break;

                    case SOUTH_CELESTIAL_POLE:
                        // Rotate the TDV coordinate system clockwise (negative) around the y axis by 90 plus
                        // the (negative)observatory latitude. The vector itself is rotated anticlockwise
                        RotatedTDV.RotateAroundY(-90.0 - Position.lat);
                        AltitudeAzimuthFromTelescopeDirectionVector(RotatedTDV, AltAz);
                        break;
                }
    #ifdef USE_INITIAL_JULIAN_DATE
                ln_get_equ_from_hrz(&AltAz, &Position, InitialJulianDate, &EquatorialCoordinates);
    #else
                ln_get_equ_from_hrz(&AltAz, &Position, ln_get_julian_from_sys(), &EquatorialCoordinates);
    #endif
            }
            else
                // The best I can do is just do a direct conversion to RA/DEC
                EquatorialCoordinatesFromTelescopeDirectionVector(TDV, EquatorialCoordinates);
            // libnova works in decimal degrees
            RightAscension = EquatorialCoordinates.ra * 24.0 / 360.0;
            Declination = EquatorialCoordinates.dec;
            DEBUGF(INDI::AlignmentSubsystem::DBG_ALIGNMENT, "Conversion Failed - HavePosition %d RA (degrees) %lf DEC (degrees) %lf", HavePosition, EquatorialCoordinates.ra, EquatorialCoordinates.dec);
        }

        DEBUGF(INDI::AlignmentSubsystem::DBG_ALIGNMENT, "New RA %lf (hours) DEC %lf (degrees)", RightAscension, Declination);
        NewRaDec(RightAscension, Declination);

        return true;
    }

    void SkywatcherAPIMount::TimerHit()
    {
        // By default this method is called every POLLMS milliseconds

        // Call the base class handler
        // This normally just calls ReadScopeStatus
        INDI::Telescope::TimerHit();

        // Do my own timer stuff assuming ReadScopeStatus has just been called

        switch(TrackState)
        {
            case SCOPE_SLEWING:
                if ((AxesStatus[AXIS1].FullStop) && (AxesStatus[AXIS2].FullStop))
                {
                    if (ISS_ON == IUFindSwitch(&CoordSP,"TRACK")->s)
                    {
                        // Goto has finished start tracking
                        TrackState = SCOPE_TRACKING;
                        // Fall through to tracking case
                    }
                    else
                    {
                        TrackState = SCOPE_IDLE;
                        break;
                    }
                }
                else
                    break;

            case SCOPE_TRACKING:
            {
                // Continue or start tracking
                // Calculate where the mount needs to be in POLLMS time
                // POLLMS is hardcoded to be one second
                double JulianOffset = 1.0 / (24.0 * 60 * 60); // TODO may need to make this longer to get a meaningful result
                TelescopeDirectionVector TDV;
                ln_hrz_posn AltAz;
                if (TransformCelestialToTelescope(CurrentTrackingTarget.ra, CurrentTrackingTarget.dec,
    #ifdef USE_INITIAL_JULIAN_DATE
                                                    0, TDV))
    #else
                                                    JulianOffset, TDV))
    #endif
                    AltitudeAzimuthFromTelescopeDirectionVector(TDV, AltAz);
                else
                {
                    // Try a conversion with the stored observatory position if any
                    bool HavePosition = false;
                    ln_lnlat_posn Position;
                    if ((NULL != IUFindNumber(&LocationNP, "LAT")) && ( 0 != IUFindNumber(&LocationNP, "LAT")->value)
                        && (NULL != IUFindNumber(&LocationNP, "LONG")) && ( 0 != IUFindNumber(&LocationNP, "LONG")->value))
                    {
                        // I assume that being on the equator and exactly on the prime meridian is unlikely
                        Position.lat = IUFindNumber(&LocationNP, "LAT")->value;
                        Position.lng = IUFindNumber(&LocationNP, "LONG")->value;
                        HavePosition = true;
                    }
                    struct ln_equ_posn EquatorialCoordinates;
                    // libnova works in decimal degrees
                    EquatorialCoordinates.ra = CurrentTrackingTarget.ra * 360.0 / 24.0;
                    EquatorialCoordinates.dec = CurrentTrackingTarget.dec;
                    if (HavePosition)
                        ln_get_hrz_from_equ(&EquatorialCoordinates, &Position,
    #ifdef USE_INITIAL_JULIAN_DATE
                                                InitialJulianDate, &AltAz);
    #else
                                                ln_get_julian_from_sys() + JulianOffset, &AltAz);
    #endif
                    else
                    {
                        // No sense in tracking in this case
                        TrackState = SCOPE_IDLE;
                        break;
                    }
                }
                DEBUGF(INDI::Logger::DBG_SESSION, "Tracking AXIS1 CurrentEncoder %ld OldTrackingTarget %ld AXIS2 CurrentEncoder %ld OldTrackingTarget %ld",
                                                CurrentEncoders[AXIS1], OldTrackingTarget[AXIS1], CurrentEncoders[AXIS2], OldTrackingTarget[AXIS2]);
                DEBUGF(INDI::Logger::DBG_SESSION, "New Tracking Target Altitude %lf degrees %ld microsteps Azimuth %lf degrees %ld microsteps",
                                    AltAz.alt, DegreesToMicrosteps(AXIS2, AltAz.alt), AltAz.az, DegreesToMicrosteps(AXIS1, AltAz.az));

                long AltitudeOffsetMicrosteps = DegreesToMicrosteps(AXIS2, AltAz.alt) + ZeroPositionEncoders[AXIS2] - CurrentEncoders[AXIS2];
                long AzimuthOffsetMicrosteps = DegreesToMicrosteps(AXIS1, AltAz.az) + ZeroPositionEncoders[AXIS1] - CurrentEncoders[AXIS1];

                DEBUGF(INDI::Logger::DBG_SESSION, "New Tracking Target AltitudeOffset %ld microsteps AzimuthOffset %ld microsteps",
                                    AltitudeOffsetMicrosteps, AzimuthOffsetMicrosteps);

                if (AzimuthOffsetMicrosteps > MicrostepsPerRevolution[AXIS1] / 2)
                {
                    DEBUG(INDI::Logger::DBG_SESSION, "Tracking AXIS1 going long way round");
                    // Going the long way round - send it the other way
                    AzimuthOffsetMicrosteps -= MicrostepsPerRevolution[AXIS1];
                }
                if (0 != AzimuthOffsetMicrosteps)
                {
                    // Calculate the slewing rates needed to reach that position
                    // at the correct time.
                    long AzimuthRate = StepperClockFrequency[AXIS1] / AzimuthOffsetMicrosteps;
                    if (!AxesStatus[AXIS1].FullStop &&
                        ((AxesStatus[AXIS1].SlewingForward && (AzimuthRate < 0)) || (!AxesStatus[AXIS1].SlewingForward && (AzimuthRate > 0))))
                    {
                        // Direction change whilst axis running
                        // Abandon tracking for this clock tick
                        DEBUG(INDI::Logger::DBG_SESSION, "Tracking - AXIS1 direction change");
                        SlowStop(AXIS1);
                    }
                    else
                    {
                        char Direction = AzimuthRate > 0 ? '0' : '1';
                        AzimuthRate = std::abs(AzimuthRate);
                        SetClockTicksPerMicrostep(AXIS1, AzimuthRate < 1 ? 1 : AzimuthRate);
                        if (AxesStatus[AXIS1].FullStop)
                        {
                            DEBUG(INDI::Logger::DBG_SESSION, "Tracking - AXIS1 restart");
                            SetMotionMode(AXIS1, '1', Direction);
                            StartMotion(AXIS1);
                        }
                        DEBUGF(INDI::Logger::DBG_SESSION, "Tracking - AXIS1 offset %ld microsteps rate %ld direction %c",
                                                                    AzimuthOffsetMicrosteps, AzimuthRate, Direction);
                    }
                }
                else
                {
                    // Nothing to do - stop the axis
                    DEBUG(INDI::Logger::DBG_SESSION, "Tracking - AXIS1 zero offset");
                    SlowStop(AXIS1);
                }

                // Do I need to take out any complete revolutions before I do this test?
                if (AltitudeOffsetMicrosteps > MicrostepsPerRevolution[AXIS2] / 2)
                {
                    DEBUG(INDI::Logger::DBG_SESSION, "Tracking AXIS2 going long way round");
                    // Going the long way round - send it the other way
                    AltitudeOffsetMicrosteps -= MicrostepsPerRevolution[AXIS2];
                }
                if (0 != AltitudeOffsetMicrosteps)
                {
                     // Calculate the slewing rates needed to reach that position
                    // at the correct time.
                    long AltitudeRate = StepperClockFrequency[AXIS2] / AltitudeOffsetMicrosteps;

                    if (!AxesStatus[AXIS2].FullStop &&
                        ((AxesStatus[AXIS2].SlewingForward && (AltitudeRate < 0)) || (!AxesStatus[AXIS2].SlewingForward && (AltitudeRate > 0))))
                    {
                        // Direction change whilst axis running
                        // Abandon tracking for this clock tick
                        DEBUG(INDI::Logger::DBG_SESSION, "Tracking - AXIS2 direction change");
                        SlowStop(AXIS2);
                    }
                    else
                    {
                        char Direction = AltitudeRate > 0 ? '0' : '1';
                        AltitudeRate = std::abs(AltitudeRate);
                        SetClockTicksPerMicrostep(AXIS2, AltitudeRate < 1 ? 1 : AltitudeRate);
                        if (AxesStatus[AXIS2].FullStop)
                        {
                            DEBUG(INDI::Logger::DBG_SESSION, "Tracking - AXIS2 restart");
                            SetMotionMode(AXIS2, '1', Direction);
                            StartMotion(AXIS2);
                        }
                        DEBUGF(INDI::Logger::DBG_SESSION, "Tracking - AXIS2 offset %ld microsteps rate %ld direction %c",
                                                                        AltitudeOffsetMicrosteps, AltitudeRate, Direction);
                    }
                }
                else
                {
                    // Nothing to do - stop the axis
                    DEBUG(INDI::Logger::DBG_SESSION, "Tracking - AXIS2 zero offset");
                    SlowStop(AXIS2);
                }


                DEBUGF(INDI::Logger::DBG_SESSION, "Tracking - AXIS1 error %d AXIS2 error %d",
                                                                    OldTrackingTarget[AXIS1] - CurrentEncoders[AXIS1],
                                                                    OldTrackingTarget[AXIS2] - CurrentEncoders[AXIS2]);

                OldTrackingTarget[AXIS1] = AzimuthOffsetMicrosteps + CurrentEncoders[AXIS1];
                OldTrackingTarget[AXIS2] = AltitudeOffsetMicrosteps + CurrentEncoders[AXIS2];
                break;
            }

            default:
                break;
        }
    }

    bool SkywatcherAPIMount::Goto(double ra,double dec)
    {
        DEBUG(INDI::AlignmentSubsystem::DBG_ALIGNMENT, "SkywatcherAPIMount::Goto");

        DEBUGF(INDI::AlignmentSubsystem::DBG_ALIGNMENT, "RA %lf DEC %lf", ra, dec);

        if (ISS_ON == IUFindSwitch(&CoordSP,"TRACK")->s)
        {
            char RAStr[32], DecStr[32];
            fs_sexa(RAStr, ra, 2, 3600);
            fs_sexa(DecStr, dec, 2, 3600);
            CurrentTrackingTarget.ra = ra;
            CurrentTrackingTarget.dec = dec;
            DEBUGF(INDI::Logger::DBG_SESSION, "New Tracking target RA %s DEC %s", RAStr, DecStr);
        }

        TelescopeDirectionVector TDV;
        ln_hrz_posn AltAz;
        if (TransformCelestialToTelescope(ra, dec, 0.0, TDV))
        {
            AltitudeAzimuthFromTelescopeDirectionVector(TDV, AltAz);
            DEBUG(INDI::AlignmentSubsystem::DBG_ALIGNMENT, "Conversion OK");
        }
        else
        {
            // Try a conversion with the stored observatory position if any
            bool HavePosition = false;
            ln_lnlat_posn Position;
            if ((NULL != IUFindNumber(&LocationNP, "LAT")) && ( 0 != IUFindNumber(&LocationNP, "LAT")->value)
                && (NULL != IUFindNumber(&LocationNP, "LONG")) && ( 0 != IUFindNumber(&LocationNP, "LONG")->value))
            {
                // I assume that being on the equator and exactly on the prime meridian is unlikely
                Position.lat = IUFindNumber(&LocationNP, "LAT")->value;
                Position.lng = IUFindNumber(&LocationNP, "LONG")->value;
                HavePosition = true;
            }
            struct ln_equ_posn EquatorialCoordinates;
            // libnova works in decimal degrees
            EquatorialCoordinates.ra = ra * 360.0 / 24.0;
            EquatorialCoordinates.dec = dec;
            if (HavePosition)
            {
    #ifdef USE_INITIAL_JULIAN_DATE
                ln_get_hrz_from_equ(&EquatorialCoordinates, &Position, InitialJulianDate, &AltAz);
    #else
                ln_get_hrz_from_equ(&EquatorialCoordinates, &Position, ln_get_julian_from_sys(), &AltAz);
    #endif
                TDV = TelescopeDirectionVectorFromAltitudeAzimuth(AltAz);
                switch (GetApproximateMountAlignment())
                {
                    case ZENITH:
                        break;

                    case NORTH_CELESTIAL_POLE:
                        // Rotate the TDV coordinate system clockwise (negative) around the y axis by 90 minus
                        // the (positive)observatory latitude. The vector itself is rotated anticlockwise
                        TDV.RotateAroundY(Position.lat - 90.0);
                        break;

                    case SOUTH_CELESTIAL_POLE:
                        // Rotate the TDV coordinate system anticlockwise (positive) around the y axis by 90 plus
                        // the (negative)observatory latitude. The vector itself is rotated clockwise
                        TDV.RotateAroundY(Position.lat + 90.0);
                        break;
                }
                AltitudeAzimuthFromTelescopeDirectionVector(TDV, AltAz);
            }
            else
            {
                // The best I can do is just do a direct conversion to Alt/Az
                TDV = TelescopeDirectionVectorFromEquatorialCoordinates(EquatorialCoordinates);
                AltitudeAzimuthFromTelescopeDirectionVector(TDV, AltAz);
            }
            DEBUGF(INDI::AlignmentSubsystem::DBG_ALIGNMENT, "Conversion Failed - HavePosition %d", HavePosition);
        }
        DEBUGF(INDI::AlignmentSubsystem::DBG_ALIGNMENT, "New Altitude %lf degrees %ld microsteps Azimuth %lf degrees %ld microsteps",
                                        AltAz.alt, DegreesToMicrosteps(AXIS2, AltAz.alt), AltAz.az, DegreesToMicrosteps(AXIS1, AltAz.az));

        // Update the current encoder positions
        GetEncoder(AXIS1);
        GetEncoder(AXIS2);

        long AltitudeOffsetMicrosteps = DegreesToMicrosteps(AXIS2, AltAz.alt) + ZeroPositionEncoders[AXIS2] - CurrentEncoders[AXIS2];
        long AzimuthOffsetMicrosteps = DegreesToMicrosteps(AXIS1, AltAz.az) + ZeroPositionEncoders[AXIS1] - CurrentEncoders[AXIS1];

        // Do I need to take out any complete revolutions before I do this test?
        if (AltitudeOffsetMicrosteps > MicrostepsPerRevolution[AXIS2] / 2)
        {
            // Going the long way round - send it the other way
            AltitudeOffsetMicrosteps -= MicrostepsPerRevolution[AXIS2];
        }

        if (AzimuthOffsetMicrosteps > MicrostepsPerRevolution[AXIS1] / 2)
        {
            // Going the long way round - send it the other way
            AzimuthOffsetMicrosteps -= MicrostepsPerRevolution[AXIS1];
        }
        DEBUGF(INDI::AlignmentSubsystem::DBG_ALIGNMENT, "Initial Axis2 %ld microsteps Axis1 %ld microsteps",
                                                        ZeroPositionEncoders[AXIS2], ZeroPositionEncoders[AXIS1]);
        DEBUGF(INDI::AlignmentSubsystem::DBG_ALIGNMENT, "Current Axis2 %ld microsteps Axis1 %ld microsteps",
                                                        CurrentEncoders[AXIS2], CurrentEncoders[AXIS1]);
        DEBUGF(INDI::AlignmentSubsystem::DBG_ALIGNMENT, "Altitude offset %ld microsteps Azimuth offset %ld microsteps",
                                                        AltitudeOffsetMicrosteps, AzimuthOffsetMicrosteps);

        SlewTo(AXIS1, AzimuthOffsetMicrosteps);
        SlewTo(AXIS2, AltitudeOffsetMicrosteps);

        TrackState = SCOPE_SLEWING;

        EqNP.s    = IPS_BUSY;

        return true;
    }



## Developing Alignment Subsystem clients
The Alignment Subsystem provides two API classes for use in clients. These are ClientAPIForAlignmentDatabase and ClientAPIForMathPluginManagement. Client developers can use these classes individually, however, the easiest way to use them is via the AlignmentSubsystemForClients class. To use this class simply ensure that is a parent of your client class.

	class LoaderClient : public INDI::BaseClient, INDI::AlignmentSubsystem::AlignmentSubsystemForClients

Somewhere in the initialisation of your client make a call to the Initalise method of the AlignmentSubsystemForClients class for example:

    void LoaderClient::Initialise(int argc, char* argv[])
    {
        std::string HostName("localhost");
        int Port = 7624;

        if (argc > 1)
            DeviceName = argv[1];
        if (argc > 2)
            HostName = argv[2];
        if (argc > 3)
        {
            std::istringstream Parameter(argv[3]);
            Parameter >> Port;
        }

        AlignmentSubsystemForClients::Initialise(DeviceName.c_str(), this);

        setServer(HostName.c_str(), Port);

        watchDevice(DeviceName.c_str());

        connectServer();

        setBLOBMode(B_ALSO, DeviceName.c_str(), NULL);
    }

To hook the Alignment Subsystem into the clients property handling you must ensure that the following virtual functions are overriden.

    virtual void newBLOB(IBLOB *bp);
    virtual void newDevice(INDI::BaseDevice *dp);
    virtual void newNumber(INumberVectorProperty *nvp);
    virtual void newProperty(INDI::Property *property);
    virtual void newSwitch(ISwitchVectorProperty *svp);

A call to the Alignment Subsystems property handling functions must then be placed in the body of these functions.

    void LoaderClient::newBLOB(IBLOB *bp)
    {
        ProcessNewBLOB(bp);
    }

    void LoaderClient::newDevice(INDI::BaseDevice *dp)
    {
        ProcessNewDevice(dp);
    }

    void LoaderClient::newNumber(INumberVectorProperty *nvp)
    {
        ProcessNewNumber(nvp);
    }

    void LoaderClient::newProperty(INDI::Property *property)
    {
        ProcessNewProperty(property);
    }

    void LoaderClient::newSwitch(ISwitchVectorProperty *svp)
    {
        ProcessNewSwitch(svp);
    }

See the documentation for the ClientAPIForAlignmentDatabase and ClientAPIForMathPluginManagement to see what other functionality is available.