/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002 The Inti Development Team.
 *
 *  iochannel.cc - A GIOChannel C++ wrapper implementation
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "iochannel.h"
#include "error.h"

using namespace Inti;

/*  G::IOChannel
 */

G::IOChannel::IOChannel(GIOChannel *channel, bool reference)
: channel_(channel)
{
	set_referenced(reference);
}

G::IOChannel::~IOChannel()
{
	if (channel_)
	{
		g_io_channel_unref(channel_);
		channel_ = 0;
	}
}

Pointer<G::IOChannel>
G::IOChannel::create(const String& filename, const char *mode, G::Error *error)
{
	GIOChannel *io = g_io_channel_new_file(filename.c_str(), mode, *error);
	return io ? new IOChannel(io) : 0;
}

Pointer<G::IOChannel>
G::IOChannel::create(int fd)
{
	GIOChannel *io = g_io_channel_unix_new(fd);
	return io ? new IOChannel(io) : 0;
}

G::IOChannelError
G::IOChannel::error_from_errno(int errno_number)
{
	return (IOChannelError)g_io_channel_error_from_errno(errno_number);
}

size_t
G::IOChannel::get_buffer_size() const
{
	return g_io_channel_get_buffer_size(channel_);
}

G::IOConditionField
G::IOChannel::get_buffer_condition() const
{
	return (IOConditionField)g_io_channel_get_buffer_condition(channel_);
}

G::IOFlagsField
G::IOChannel::get_flags() const
{
	return (IOFlagsField)g_io_channel_get_flags(channel_);
}

String
G::IOChannel::get_line_term() const
{
	int length;
	const char *line_term = g_io_channel_get_line_term(channel_, &length);
	String s(line_term, length);
	return s;
}

bool
G::IOChannel::get_buffered() const
{
	return g_io_channel_get_buffered(channel_);
}

String
G::IOChannel::get_encoding() const
{
	return g_io_channel_get_encoding(channel_);
}

bool
G::IOChannel::get_close_on_unref() const
{
	return g_io_channel_get_close_on_unref(channel_);
}

int
G::IOChannel::get_fd() const
{
	return g_io_channel_unix_get_fd(channel_);
}

void
G::IOChannel::ref()
{
	g_io_channel_ref(channel_);
}

void
G::IOChannel::unref()
{
	bool destroy = channel_->ref_count == 1;
	g_io_channel_unref(channel_);
	if (destroy)
	{
		channel_ = 0;
		delete this;
	}
}

G::IOStatus
G::IOChannel::close(bool flush, G::Error *error)
{
	return (IOStatus)g_io_channel_shutdown(channel_, flush, *error);
}

namespace { // io_slot_callack

gboolean io_slot_callack(GIOChannel *source, GIOCondition condition, gpointer data)
{
	G::IOChannel::IOSlot *slot = static_cast<G::IOChannel::IOSlot*>(data);
	G::IOChannel tmp_channel(source, false);
	bool result = slot->call(tmp_channel, (G::IOConditionField)condition);
	tmp_channel.unref();
	return result;
}

} // namespace

unsigned int
G::IOChannel::add_watch(IOConditionField condition, const IOSlot *slot, int priority, GDestroyNotify notify)
{
	return g_io_add_watch_full(channel_, priority, (GIOCondition)condition, &io_slot_callack, (void*)slot, notify);
}

void
G::IOChannel::set_buffer_size(size_t size)
{
	g_io_channel_set_buffer_size(channel_, size);
}

G::IOStatus
G::IOChannel::set_flags(IOFlagsField flags, G::Error *error)
{
	return (IOStatus)g_io_channel_set_flags(channel_, (GIOFlags)flags, *error);
}

void
G::IOChannel::set_line_term(const String& line_term)
{
	g_io_channel_set_line_term(channel_, line_term.c_str(), line_term.size());
}

void
G::IOChannel::set_buffered(bool buffered)
{
	g_io_channel_set_buffered(channel_, buffered);
}

G::IOStatus
G::IOChannel::set_encoding(const char *encoding, G::Error *error)
{
	return (IOStatus)g_io_channel_set_encoding(channel_, encoding, *error);
}

void
G::IOChannel::set_close_on_unref(bool do_close)
{
	g_io_channel_set_close_on_unref(channel_, do_close);
}

G::IOStatus
G::IOChannel::flush(G::Error *error)
{
	return (IOStatus)g_io_channel_flush(channel_, *error);
}

G::IOStatus
G::IOChannel::read_line(String& str, G::Error *error)
{
	char *tmp_str = 0;
	gsize bytes;
	GIOStatus status = g_io_channel_read_line(channel_, &tmp_str, &bytes, 0, *error);
	if (tmp_str)
	{
		str.assign(tmp_str, tmp_str + bytes);
		g_free(tmp_str);
	}
	return (IOStatus)status;
}

G::IOStatus
G::IOChannel::read_to_end(String& str, G::Error *error)
{
	char *tmp_str = 0;
	gsize bytes;
	GIOStatus status = g_io_channel_read_to_end(channel_, &tmp_str, &bytes, *error);
	if (tmp_str)
	{
		str.assign(tmp_str, tmp_str + bytes);
		g_free(tmp_str);
	}
	return (IOStatus)status;
}

G::IOStatus
G::IOChannel::read(char *buffer, size_t count, size_t *bytes_read, G::Error *error)
{
	return (IOStatus)g_io_channel_read_chars(channel_, buffer, count, bytes_read, *error);
}

G::IOStatus
G::IOChannel::read(String& str, size_t count, G::Error *error)
{
	char *tmp_str = new char[count];
	gsize bytes;
	GIOStatus status = g_io_channel_read_chars(channel_, tmp_str, count, &bytes, *error);
	if (tmp_str)
	{
		str.assign(tmp_str, tmp_str + bytes);
		g_free(tmp_str);
	}
	return (IOStatus)status;
}

G::IOStatus
G::IOChannel::read(G::Unichar& unichar, G::Error *error)
{
	gunichar tmp_unichar;
	GIOStatus status = g_io_channel_read_unichar(channel_, &tmp_unichar, *error);
	unichar = tmp_unichar;
	return (IOStatus)status;
}

G::IOStatus
G::IOChannel::write(const char *buffer, size_t count, size_t *bytes_written, G::Error *error)
{
	return (IOStatus)g_io_channel_write_chars(channel_, buffer, (gssize)count, bytes_written, *error);
}

G::IOStatus
G::IOChannel::write(const String& str, G::Error *error)
{
	gsize bytes_written;
	return (IOStatus)g_io_channel_write_chars(channel_, str.data(), (gssize)str.size(), &bytes_written, *error);
}

G::IOStatus
G::IOChannel::write(G::Unichar unichar, G::Error *error)
{
	return (IOStatus)g_io_channel_write_unichar(channel_, unichar, *error);
}

G::IOStatus
G::IOChannel::seek(gint64 offset, SeekType type, G::Error *error)
{
	return (IOStatus)g_io_channel_seek_position(channel_, offset, (GSeekType)type, *error);
}

