/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002-2003 The Inti Development Team.
 *  Copyright (C) 2000 Red Hat, Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
//! @file inti/gtk/paned.h
//! @brief A GtkPaned, GtkHPaned and GtkVPaned C++ wrapper interface.
//!
//! Provides Paned, the base class for widgets with two panes, arranged either
//! horizontally (HPaned) or vertically (VPaned).

#ifndef INTI_GTK_PANED_H
#define INTI_GTK_PANED_H

#ifndef INTI_GTK_CONTAINER_H
#include <inti/gtk/container.h>
#endif

#ifndef __GTK_HPANED_H__
#include <gtk/gtkhpaned.h>
#endif

#ifndef __GTK_VPANED_H__
#include <gtk/gtkvpaned.h>
#endif

namespace Inti {
	
namespace Gtk {

class PanedClass;
class HPanedClass;
class VPanedClass;

//! @class Paned paned.h inti/gtk/paned.h
//! @brief A GtkPaned C++ wrapper class.
//!
//! Paned is the base class for widgets with two panes, arranged either horizontally (HPaned)
//! or vertically (VPaned). Child widgets are added to the panes of the widget with pack1()
//! and pack2(). The division beween the two children is set by default from the size requests
//! of the children, but it can be adjusted by the user. A paned widget draws a separator
//! between the two child widgets and a small handle that the user can drag to adjust the
//! division. It does not draw any relief around the children or around the separator. (The
//! space in which the separator is drawn is called the gutter.) Often, it is useful to put
//! each child inside a Frame with the shadow type set to Gtk::SHADOW_IN so that the gutter
//! appears as a ridge. The application can set the position of the separator as if it were
//! set by the user, by calling set_position().
//!
//! Each child has two options that can be set, resize and shrink. If resize is true, then
//! when the Paned is resized, that child will expand or shrink along with the paned widget.
//! If shrink is true, then when that child can be made smaller than it's requisition by the
//! user. Setting shrink to false allows the application to set a minimum size. If resize is
//! false for both children, then this is treated as if resize is true for both children.
//!
//! <B>Example:</B> Creating a paned widget with minimum sizes.
//! @code
//! Gtk::HPaned *hpaned = new Gtk::HPaned;
//! hpaned->set_size_request(200, -1);
//!
//! Gtk::Frame *frame1 = new Gtk::Frame;
//! frame1->set_shadow_type(Gtk::SHADOW_IN);
//! frame1->set_size_request(50, -1);
//!
//! Gtk::Frame *frame2 = new Gtk::Frame;
//! frame2->set_shadow_type(Gtk::SHADOW_IN);
//! frame2->set_size_request(50, -1);
//!
//! hpaned->pack1(*frame1, true, false);
//! hpaned->pack2(*frame2, false, false);
//! @endcode

class Paned : public Container
{
	friend class G::Object;
	friend class PanedClass;

	Paned(const Paned&);
	Paned& operator=(const Paned&);
	
protected:
//! @name Constructors
//! @{

	Paned();
	//!< Construct a new Paned.

	explicit Paned(GtkPaned *paned, bool reference = false);
	//!< Construct a new Paned from an existing GtkPaned.
	//!< @param paned A pointer to a GtkPaned.
	//!< @param reference Set false if the initial reference count is floating, set true if it's not.
	//!<
	//!< <BR>The <EM>paned</EM> can be a newly created GtkPaned or an existing
	//!< GtkPaned (see G::Object::Object).

	virtual ~Paned() = 0;
	//!< Destructor.
	
//! @}
//  Properties

	typedef G::Property<int> PositionPropertyType;
	typedef G::PropertyProxy<G::Object, PositionPropertyType> PositionPropertyProxy;
	static const PositionPropertyType position_property;

	typedef G::Property<bool> PositionSetPropertyType;
	typedef G::PropertyProxy<G::Object, PositionSetPropertyType> PositionSetPropertyProxy;
	static const PositionSetPropertyType position_set_property;

public:
//! @name Accessors
//! @{

	GtkPaned* gtk_paned() const { return (GtkPaned*)instance; }
	//!< Get a pointer to the GtkPaned structure.

	GtkPanedClass* gtk_paned_class() const;
	//!< Get a pointer to the GtkPanedClass structure.

	operator GtkPaned* () const;
	//!< Conversion operator; safely converts a Paned to a GtkPaned pointer.

	Widget* child1() const;
	//!< Gets the child widget in the top or left pane.

	Widget* child2() const;
	//!< Gets the child widget in the bottom or right pane.

	Gdk::Window* handle() const;
	//!< Returns the handle window in the divider between the two panes.

	int get_position() const;
	//!< Obtains the position of the divider between the two panes.
	
//! @}
//! @name Methods
//! @{

	void add1(Widget& child);
	//!< Adds a child to the top or left pane with default parameters. 
	//!< @param child The child to add.
	//!<
	//!< <BR>This is equivalent to pack1(child, false, true).

	void add2(Widget& child);
	//!< Adds a child to the bottom or right pane with default parameters. 
	//!< @param child The child to add.
	//!<
	//!< <BR>This is equivalent to pack2(child, true, true).
	
	void pack1(Widget& child, bool resize = false, bool shrink = true);
	//!< Adds a child to the top or left pane.
	//!< @param child The child to add.
	//!< @param resize Should this child expand when the paned widget is resized.
	//!< @param shrink Can this child be made smaller than its requsition.

	void pack2(Widget& child, bool resize = true, bool shrink = true);
	//!< Adds a child to the bottom or right pane.
	//!< @param child The child to add.
	//!< @param resize Should this child expand when the paned widget is resized.
	//!< @param shrink Can this child be made smaller than its requsition.

	void set_position(int position);
	//!< Sets the position of the divider between the two panes.
	//!< @param position The pixel position of divider, a negative value means that the position is unset.

//! @}
//! @name Property Proxies
//! @{

	const PositionPropertyProxy prop_position()
	{
		return PositionPropertyProxy(this, &position_property);
	}
	//!< Position of the paned separator in pixels (0 means all the way to the left/top) (int : Read / Write).

	const PositionSetPropertyProxy prop_position_set()
	{
		return PositionSetPropertyProxy(this, &position_set_property);
	}
	//!< Set true if the position property should be used (bool : Read / Write).

//! @}
};

//! @class HPaned paned.h inti/gtk/paned.h
//! @brief A GtkHPaned C++ wrapper class.
//!
//! The HPaned widget is a container widget with two children arranged horizontally. 
//! The division between the two panes is adjustable by the user by dragging a handle.
//! See Paned for details.

class HPaned : public Paned
{
	friend class G::Object;
	friend class HPanedClass;

	HPaned(const HPaned&);
	HPaned& operator=(const HPaned&);
	
protected:
//! @name Constructor
//! @{

	explicit HPaned(GtkHPaned *hpaned, bool reference = false);
	//!< Construct a new HPaned from an existing GtkHPaned.
	//!< @param hpaned A pointer to a GtkHPaned.
	//!< @param reference Set false if the initial reference count is floating, set true if it's not.
	//!<
	//!< <BR>The <EM>hpaned</EM> can be a newly created GtkHPaned or an existing
	//!< GtkHPaned (see G::Object::Object).
	
//! @}

public:
//! @name Constructors
//! @{

	HPaned();
	//!< Construct a new HPaned.

	virtual ~HPaned();
	//!< Destructor.

//! @}
//! @name Accessors
//! @{

	GtkHPaned* gtk_hpaned() const { return (GtkHPaned*)instance; }
	//!< Get a pointer to the GtkHPaned structure.

	GtkHPanedClass* gtk_hpaned_class() const;
	//!< Get a pointer to the GtkHPanedClass structure.

	operator GtkHPaned* () const;
	//!< Conversion operator; safely converts a HPaned to a GtkHPaned pointer.

//! @}
};

//! @class VPaned paned.h inti/gtk/paned.h
//! @brief A GtkVPaned C++ wrapper class.
//!
//! The VPaned widget is a container widget with two children arranged vertically.
//! The division between the two panes is adjustable by the user by dragging a
//! handle. See Paned for details.

class VPaned : public Paned
{
	friend class G::Object;
	friend class VPanedClass;

	VPaned(const VPaned&);
	VPaned& operator=(const VPaned&);
	
protected:
//! @name Constructors
//! @{

	explicit VPaned(GtkVPaned *vpaned, bool reference = false);
	//!< Construct a new VPaned from an existing GtkVPaned.
	//!< @param vpaned A pointer to a GtkVPaned.
	//!< @param reference Set false if the initial reference count is floating, set true if it's not.
	//!<
	//!< <BR>The <EM>vpaned</EM> can be a newly created GtkVPaned or an existing
	//!< GtkVPaned (see G::Object::Object).

//! @}

public:
//! @name Constructors
//! @{

	VPaned();
	//!< Construct a new VPaned.

	virtual ~VPaned();
	//!< Destructor.

//! @}
//! @name Accessors
//! @{

	GtkVPaned* gtk_vpaned() const { return (GtkVPaned*)instance; }
	//!< Get a pointer to the GtkVPaned structure.

	GtkVPanedClass* gtk_vpaned_class() const;
	//!< Get a pointer to the GtkVPanedClass structure.

	operator GtkVPaned* () const;
	//!< Conversion operator; safely converts a VPaned to a GtkVPaned pointer.

//! @}
};

} // namespace Gtk

} // namespace Inti

#endif // INTI_GTK_PANED_H

