/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002-2003 The Inti Development Team.
 *
 *  treemodel.cc - GtkTreePath, GtkTreeModel, GtkTreeRowReference and GtkTreeIter C++ wrapper implementation
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
#include "treemodel.h"
#include "private/treemodel_p.h"
#include "selection.h"
#include "../glib/object.h"
#include "../glib/value.h"

using namespace Inti;

/*  Gtk::TreePath
 */
 
Gtk::TreePath::TreePath()
: G::Boxed(GTK_TYPE_TREE_PATH, gtk_tree_path_new(), false)
{
}

Gtk::TreePath::TreePath(const char *path)
: G::Boxed(GTK_TYPE_TREE_PATH, path ? gtk_tree_path_new_from_string(path) : gtk_tree_path_new_first(), false)
{
	if (!path)
		append_index(0);
}

Gtk::TreePath::TreePath(int first_index, ...)
: G::Boxed(GTK_TYPE_TREE_PATH, gtk_tree_path_new(), false)
{
	va_list args;
	va_start(args, first_index);
	int arg = first_index;

	while (arg != -1)
	{
		append_index(arg);
		arg = va_arg(args, int);
	}

	va_end(args);
}

Gtk::TreePath::TreePath(GtkTreePath *path)
: G::Boxed(GTK_TYPE_TREE_PATH, path)
{
}

Gtk::TreePath::TreePath(GtkTreePath *path, bool copy)
: G::Boxed(GTK_TYPE_TREE_PATH, path, copy)
{
}

Gtk::TreePath::TreePath(const TreePath& src)
: G::Boxed(GTK_TYPE_TREE_PATH, src.g_boxed(), true)
{
}

Gtk::TreePath::~TreePath()
{
}

Gtk::TreePath&
Gtk::TreePath::operator=(const TreePath& src)
{
	if (src.boxed_ != boxed_)
		copy(src);
	return *this;
}

Gtk::TreePath::operator GtkTreePath* () const
{
	return this ? gtk_tree_path() : 0;
}

String
Gtk::TreePath::to_string() const
{
	char *tmp_path = gtk_tree_path_to_string(gtk_tree_path());
	String path(tmp_path);
	g_free(tmp_path);
	return path;
}

int
Gtk::TreePath::get_depth() const
{
	return gtk_tree_path_get_depth(gtk_tree_path());
}

std::vector<int>
Gtk::TreePath::get_indices() const
{
	std::vector<int> indices;
	int *tmp_indices = gtk_tree_path_get_indices(gtk_tree_path());
	int count = gtk_tree_path_get_depth(gtk_tree_path());

	int i = 0;
	while (i < count)
	{
		indices.push_back(tmp_indices[i]);
		++i;
	}
	return indices;
}
	
int
Gtk::TreePath::compare(const TreePath& other) const
{
	return gtk_tree_path_compare(gtk_tree_path(), other.gtk_tree_path());
}

bool 
Gtk::TreePath::is_ancestor(const TreePath& descendant) const
{
	return gtk_tree_path_is_ancestor(gtk_tree_path(), descendant.gtk_tree_path());
}

bool 
Gtk::TreePath::is_descendant(const TreePath& ancestor) const
{
 	return gtk_tree_path_is_descendant(gtk_tree_path(), ancestor.gtk_tree_path());
}

bool 
Gtk::TreePath::operator==(const TreePath& other) const
{
	return compare(other) == 0;
}

bool 
Gtk::TreePath::operator!=(const TreePath& other) const
{
	return compare(other);
}
	
bool 
Gtk::TreePath::operator<(const TreePath& other) const
{
	return compare(other) > 0;
}
	
bool 
Gtk::TreePath::operator>(const TreePath& other) const
{
	return compare(other) < 0;
}
	
void
Gtk::TreePath::append_index(int index)
{
	gtk_tree_path_append_index(gtk_tree_path(), index);
}

void
Gtk::TreePath::prepend_index(int index)
{
	gtk_tree_path_prepend_index(gtk_tree_path(), index);
}

void 
Gtk::TreePath::next()
{
	gtk_tree_path_next(gtk_tree_path());
}

bool 
Gtk::TreePath::prev()
{
	return gtk_tree_path_prev(gtk_tree_path());
}

bool 
Gtk::TreePath::up()
{
	return gtk_tree_path_up(gtk_tree_path());
}

void 
Gtk::TreePath::down()
{
	gtk_tree_path_down(gtk_tree_path());
}

/*  Gtk::TreeIter
 */
 
Gtk::TreeIter::TreeIter()
: G::Boxed(GTK_TYPE_TREE_ITER)
{
	GtkTreeIter iter = { 0, };
	copy(&iter);
}

Gtk::TreeIter::TreeIter(GtkTreeIter *iter)
: G::Boxed(GTK_TYPE_TREE_ITER, iter)
{
}

Gtk::TreeIter::TreeIter(GtkTreeIter *iter, bool copy)
: G::Boxed(GTK_TYPE_TREE_ITER, iter, copy)
{
}

Gtk::TreeIter::TreeIter(const TreeIter& src)
: G::Boxed(GTK_TYPE_TREE_ITER, src.g_boxed(), true)
{
}

Gtk::TreeIter::~TreeIter()
{
}

Gtk::TreeIter&
Gtk::TreeIter::operator=(const TreeIter& src)
{
	if (src.boxed_ != boxed_)
		copy(src);
	return *this;
}

Gtk::TreeIter::operator GtkTreeIter* () const
{
	return this ? gtk_tree_iter() : 0;
}

/*  Gtk::TreeModel
 */

Gtk::TreeModel::~TreeModel()
{
}

GtkTreeModelIface*
Gtk::TreeModel::gtk_tree_model_iface() const
{
	return peek<GtkTreeModelIface>(GTK_TYPE_TREE_MODEL);
}

Gtk::TreeModel::operator GtkTreeModel* () const
{
	return this ? gtk_tree_model() : 0;
}

Gtk::TreeModelFlagsField
Gtk::TreeModel::get_flags() const
{
	return (TreeModelFlagsField)gtk_tree_model_get_flags(gtk_tree_model());
}

int
Gtk::TreeModel::get_n_columns() const
{
	return gtk_tree_model_get_n_columns(gtk_tree_model());
}

GType
Gtk::TreeModel::get_column_type(int index) const
{
	return gtk_tree_model_get_column_type(gtk_tree_model(), index);
}

bool
Gtk::TreeModel::get_iter(TreeIter& iter, const TreePath& path) const
{
	return gtk_tree_model_get_iter(gtk_tree_model(), iter.gtk_tree_iter(), path.gtk_tree_path());
}

bool
Gtk::TreeModel::get_iter(TreeIter& iter, const String& path) const
{
	return gtk_tree_model_get_iter_from_string(gtk_tree_model(), iter.gtk_tree_iter(), path.c_str());
}

String 
Gtk::TreeModel::get_string_from_iter(const TreeIter& iter) const
{
	char *tmp =  gtk_tree_model_get_string_from_iter(gtk_tree_model(), iter.gtk_tree_iter());
	String s(tmp);
	g_free(tmp);
	return s;
}

bool
Gtk::TreeModel::get_iter_first(TreeIter& iter) const
{
	return gtk_tree_model_get_iter_first(gtk_tree_model(), iter.gtk_tree_iter());
}

Pointer<Gtk::TreePath>
Gtk::TreeModel::get_path(const TreeIter& iter) const
{
	return G::Boxed::wrap<TreePath>(GTK_TYPE_TREE_PATH, gtk_tree_model_get_path(gtk_tree_model(), iter.gtk_tree_iter()), false);
}

void
Gtk::TreeModel::get_value(const TreeIter& iter, int column, G::Value& value) const
{
	gtk_tree_model_get_value(gtk_tree_model(), iter.gtk_tree_iter(), column, value.g_value());
}

String
Gtk::TreeModel::get_value(const TreeIter& iter, int column) const
{
	String s;
	get_value(iter, column, s);
	return s;
}

bool
Gtk::TreeModel::get_value(const TreeIter& iter, int column, String& str) const
{
	G::Value value;
	gtk_tree_model_get_value(gtk_tree_model(), iter, column, value.g_value());
	return value.get(str);
}

bool
Gtk::TreeModel::iter_has_child(const TreeIter& iter) const
{
	return gtk_tree_model_iter_has_child(gtk_tree_model(), iter.gtk_tree_iter());
}

bool
Gtk::TreeModel::iter_next(TreeIter& iter)
{
	return gtk_tree_model_iter_next(gtk_tree_model(), iter.gtk_tree_iter());
}

bool
Gtk::TreeModel::iter_children(TreeIter& iter, const TreeIter *parent)
{
	return gtk_tree_model_iter_children(gtk_tree_model(), iter.gtk_tree_iter(), *parent);
}

int
Gtk::TreeModel::iter_n_children(const TreeIter *iter)
{
	return gtk_tree_model_iter_n_children(gtk_tree_model(), *iter);
}

bool
Gtk::TreeModel::iter_nth_child(TreeIter& iter, const TreeIter *parent, int n)
{
	return gtk_tree_model_iter_nth_child(gtk_tree_model(), iter.gtk_tree_iter(), *parent, n);
}

bool
Gtk::TreeModel::iter_parent(TreeIter& iter, const TreeIter& child)
{
	return gtk_tree_model_iter_parent(gtk_tree_model(), iter.gtk_tree_iter(), child.gtk_tree_iter());
}

void
Gtk::TreeModel::ref_node(TreeIter *iter)
{
	gtk_tree_model_ref_node(gtk_tree_model(), *iter);
}

void
Gtk::TreeModel::unref_node(TreeIter *iter)
{
	gtk_tree_model_unref_node(gtk_tree_model(), *iter);
}

namespace { // ForeachSlot callback

gboolean foreach_slot_callback(GtkTreeModel*, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
	Gtk::TreeModel::ForeachSlot *slot = static_cast<Gtk::TreeModel::ForeachSlot*>(data);
	Gtk::TreePath tmp_path(path);
	Gtk::TreeIter tmp_iter(iter);
	return slot->call(tmp_path, tmp_iter);
}

} // ForeachSlot callback

void
Gtk::TreeModel::foreach(const ForeachSlot *each)
{
	gtk_tree_model_foreach(gtk_tree_model(), &foreach_slot_callback, (void*)each);
}

void
Gtk::TreeModel::row_changed(const TreePath& path, const TreeIter& iter)
{
	gtk_tree_model_row_changed(gtk_tree_model(), path.gtk_tree_path(), iter.gtk_tree_iter());
}

void
Gtk::TreeModel::row_inserted(const TreePath& path, const TreeIter& iter)
{
	gtk_tree_model_row_inserted(gtk_tree_model(), path.gtk_tree_path(), iter.gtk_tree_iter());
}

void
Gtk::TreeModel::row_has_child_toggled(const TreePath& path, const TreeIter& iter)
{
	gtk_tree_model_row_has_child_toggled(gtk_tree_model(), path.gtk_tree_path(), iter.gtk_tree_iter());
}

void
Gtk::TreeModel::row_deleted(const TreePath& path)
{
	gtk_tree_model_row_deleted(gtk_tree_model(), path.gtk_tree_path());
}

void
Gtk::TreeModel::rows_reordered(const TreePath& path, const TreeIter& iter, int *new_order)
{
	gtk_tree_model_rows_reordered(gtk_tree_model(), path.gtk_tree_path(), iter.gtk_tree_iter(), new_order);
}

/*  Gtk::TreeModelIface
 */

void
Gtk::TreeModelIface::init(GtkTreeModelIface *g_iface)
{
	g_iface->row_changed = &row_changed_proxy;
	g_iface->row_inserted = &row_inserted_proxy;
	g_iface->row_has_child_toggled = &row_has_child_toggled_proxy;
	g_iface->row_deleted = &row_deleted_proxy;
	g_iface->rows_reordered = &rows_reordered_proxy;
	g_iface->get_flags = &get_flags_proxy;
	g_iface->get_n_columns = &get_n_columns_proxy;
	g_iface->get_column_type = &get_column_type_proxy;
	g_iface->get_iter = &get_iter_proxy;
	g_iface->get_path = &get_path_proxy;
	g_iface->get_value = &get_value_proxy;
	g_iface->iter_next = &iter_next_proxy;
	g_iface->iter_children = &iter_children_proxy;
	g_iface->iter_has_child = &iter_has_child_proxy;
	g_iface->iter_n_children = &iter_n_children_proxy;
	g_iface->iter_nth_child = &iter_nth_child_proxy;
	g_iface->iter_parent = &iter_parent_proxy;
	g_iface->ref_node = &ref_node_proxy;
	g_iface->unref_node = &unref_node_proxy;
}

GtkTreeModelFlags
Gtk::TreeModelIface::get_flags_proxy(GtkTreeModel *tree_model)
{
	GtkTreeModelFlags result = (GtkTreeModelFlags)0;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(tree_model));
	Gtk::TreeModel *tmp_tree_model = dynamic_cast<Gtk::TreeModel*>(object);
	if (tmp_tree_model)
		result = tmp_tree_model->do_get_flags();
	else
	{
		GtkTreeModelIface *tmp_iface = GTK_TREE_MODEL_GET_IFACE(tree_model);
		GtkTreeModelIface *g_iface = static_cast<GtkTreeModelIface*>(g_type_interface_peek_parent(tmp_iface));
		if (g_iface->get_flags)
			result = g_iface->get_flags(tree_model);
	}
	return result;
}

gint
Gtk::TreeModelIface::get_n_columns_proxy(GtkTreeModel *tree_model)
{
	gint result = 0;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(tree_model));
	Gtk::TreeModel *tmp_tree_model = dynamic_cast<Gtk::TreeModel*>(object);
	if (tmp_tree_model)
		result = tmp_tree_model->do_get_n_columns();
	else
	{
		GtkTreeModelIface *tmp_iface = GTK_TREE_MODEL_GET_IFACE(tree_model);
		GtkTreeModelIface *g_iface = static_cast<GtkTreeModelIface*>(g_type_interface_peek_parent(tmp_iface));
		if (g_iface->get_n_columns)
			result = g_iface->get_n_columns(tree_model);
	}
	return result;
}

GType
Gtk::TreeModelIface::get_column_type_proxy(GtkTreeModel *tree_model, gint index)
{
	GType result = G_TYPE_INVALID;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(tree_model));
	Gtk::TreeModel *tmp_tree_model = dynamic_cast<Gtk::TreeModel*>(object);
	if (tmp_tree_model)
		result = tmp_tree_model->do_get_column_type(index);
	else
	{
		GtkTreeModelIface *tmp_iface = GTK_TREE_MODEL_GET_IFACE(tree_model);
		GtkTreeModelIface *g_iface = static_cast<GtkTreeModelIface*>(g_type_interface_peek_parent(tmp_iface));
		if (g_iface->get_column_type)
			result = g_iface->get_column_type(tree_model, index);
	}
	return result;
}

gboolean
Gtk::TreeModelIface::get_iter_proxy(GtkTreeModel *tree_model, GtkTreeIter *iter, GtkTreePath *path)
{
	gboolean result = FALSE;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(tree_model));
	Gtk::TreeModel *tmp_tree_model = dynamic_cast<Gtk::TreeModel*>(object);
	if (tmp_tree_model)
		result = tmp_tree_model->do_get_iter(iter, path);
	else
	{
		GtkTreeModelIface *tmp_iface = GTK_TREE_MODEL_GET_IFACE(tree_model);
		GtkTreeModelIface *g_iface = static_cast<GtkTreeModelIface*>(g_type_interface_peek_parent(tmp_iface));
		if (g_iface->get_iter)
			result = g_iface->get_iter(tree_model, iter, path);
	}
	return result;
}

GtkTreePath*
Gtk::TreeModelIface::get_path_proxy(GtkTreeModel *tree_model, GtkTreeIter *iter)
{
	GtkTreePath *result = 0;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(tree_model));
	Gtk::TreeModel *tmp_tree_model = dynamic_cast<Gtk::TreeModel*>(object);
	if (tmp_tree_model)
		result = tmp_tree_model->do_get_path(iter);
	else
	{
		GtkTreeModelIface *tmp_iface = GTK_TREE_MODEL_GET_IFACE(tree_model);
		GtkTreeModelIface *g_iface = static_cast<GtkTreeModelIface*>(g_type_interface_peek_parent(tmp_iface));
		if (g_iface->get_path)
			result = g_iface->get_path(tree_model, iter);
	}
	return result;
}

void
Gtk::TreeModelIface::get_value_proxy(GtkTreeModel *tree_model, GtkTreeIter *iter, gint column, GValue *value)
{
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(tree_model));
	Gtk::TreeModel *tmp_tree_model = dynamic_cast<Gtk::TreeModel*>(object);
	if (tmp_tree_model)
		tmp_tree_model->do_get_value(iter, column, value);
	else
	{
		GtkTreeModelIface *tmp_iface = GTK_TREE_MODEL_GET_IFACE(tree_model);
		GtkTreeModelIface *g_iface = static_cast<GtkTreeModelIface*>(g_type_interface_peek_parent(tmp_iface));
		if (g_iface->get_value)
			g_iface->get_value(tree_model, iter, column, value);
	}
}

gboolean
Gtk::TreeModelIface::iter_next_proxy(GtkTreeModel *tree_model, GtkTreeIter *iter)
{
	gboolean result = FALSE;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(tree_model));
	Gtk::TreeModel *tmp_tree_model = dynamic_cast<Gtk::TreeModel*>(object);
	if (tmp_tree_model)
		result = tmp_tree_model->do_iter_next(iter);
	else
	{
		GtkTreeModelIface *tmp_iface = GTK_TREE_MODEL_GET_IFACE(tree_model);
		GtkTreeModelIface *g_iface = static_cast<GtkTreeModelIface*>(g_type_interface_peek_parent(tmp_iface));
		if (g_iface->iter_next)
			result = g_iface->iter_next(tree_model, iter);
	}
	return result;
}

gboolean
Gtk::TreeModelIface::iter_children_proxy(GtkTreeModel *tree_model, GtkTreeIter *iter, GtkTreeIter *parent)
{
	gboolean result = FALSE;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(tree_model));
	Gtk::TreeModel *tmp_tree_model = dynamic_cast<Gtk::TreeModel*>(object);
	if (tmp_tree_model)
		result = tmp_tree_model->do_iter_children(iter, parent);
	else
	{
		GtkTreeModelIface *tmp_iface = GTK_TREE_MODEL_GET_IFACE(tree_model);
		GtkTreeModelIface *g_iface = static_cast<GtkTreeModelIface*>(g_type_interface_peek_parent(tmp_iface));
		if (g_iface->iter_children)
			result = g_iface->iter_children(tree_model, iter, parent);
	}
	return result;
}

gboolean
Gtk::TreeModelIface::iter_has_child_proxy(GtkTreeModel *tree_model, GtkTreeIter *iter)
{
	gboolean result = FALSE;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(tree_model));
	Gtk::TreeModel *tmp_tree_model = dynamic_cast<Gtk::TreeModel*>(object);
	if (tmp_tree_model)
		result = tmp_tree_model->do_iter_has_child(iter);
	else
	{
		GtkTreeModelIface *tmp_iface = GTK_TREE_MODEL_GET_IFACE(tree_model);
		GtkTreeModelIface *g_iface = static_cast<GtkTreeModelIface*>(g_type_interface_peek_parent(tmp_iface));
		if (g_iface->iter_has_child)
			result = g_iface->iter_has_child(tree_model, iter);
	}
	return result;
}

gint
Gtk::TreeModelIface::iter_n_children_proxy(GtkTreeModel *tree_model, GtkTreeIter *iter)
{
	gint result = 0;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(tree_model));
	Gtk::TreeModel *tmp_tree_model = dynamic_cast<Gtk::TreeModel*>(object);
	if (tmp_tree_model)
		result = tmp_tree_model->do_iter_n_children(iter);
	else
	{
		GtkTreeModelIface *tmp_iface = GTK_TREE_MODEL_GET_IFACE(tree_model);
		GtkTreeModelIface *g_iface = static_cast<GtkTreeModelIface*>(g_type_interface_peek_parent(tmp_iface));
		if (g_iface->iter_n_children)
			result = g_iface->iter_n_children(tree_model, iter);
	}
	return result;
}

gboolean
Gtk::TreeModelIface::iter_nth_child_proxy(GtkTreeModel *tree_model, GtkTreeIter *iter, GtkTreeIter *parent, gint n)
{
	gboolean result = FALSE;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(tree_model));
	Gtk::TreeModel *tmp_tree_model = dynamic_cast<Gtk::TreeModel*>(object);
	if (tmp_tree_model)
		result = tmp_tree_model->do_iter_nth_child(iter, parent, n);
	else
	{
		GtkTreeModelIface *tmp_iface = GTK_TREE_MODEL_GET_IFACE(tree_model);
		GtkTreeModelIface *g_iface = static_cast<GtkTreeModelIface*>(g_type_interface_peek_parent(tmp_iface));
		if (g_iface->iter_nth_child)
			result = g_iface->iter_nth_child(tree_model, iter, parent, n);
	}
	return result;
}

gboolean
Gtk::TreeModelIface::iter_parent_proxy(GtkTreeModel *tree_model, GtkTreeIter *iter, GtkTreeIter *child)
{
	gboolean result = FALSE;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(tree_model));
	Gtk::TreeModel *tmp_tree_model = dynamic_cast<Gtk::TreeModel*>(object);
	if (tmp_tree_model)
		result = tmp_tree_model->do_iter_parent(iter, child);
	else
	{
		GtkTreeModelIface *tmp_iface = GTK_TREE_MODEL_GET_IFACE(tree_model);
		GtkTreeModelIface *g_iface = static_cast<GtkTreeModelIface*>(g_type_interface_peek_parent(tmp_iface));
		if (g_iface->iter_parent)
			result = g_iface->iter_parent(tree_model, iter, child);
	}
	return result;
}

void
Gtk::TreeModelIface::ref_node_proxy(GtkTreeModel *tree_model, GtkTreeIter *iter)
{
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(tree_model));
	Gtk::TreeModel *tmp_tree_model = dynamic_cast<Gtk::TreeModel*>(object);
	if (tmp_tree_model)
		tmp_tree_model->do_ref_node(iter);
	else
	{
		GtkTreeModelIface *tmp_iface = GTK_TREE_MODEL_GET_IFACE(tree_model);
		GtkTreeModelIface *g_iface = static_cast<GtkTreeModelIface*>(g_type_interface_peek_parent(tmp_iface));
		if (g_iface->ref_node)
			g_iface->ref_node(tree_model, iter);
	}
}

void
Gtk::TreeModelIface::unref_node_proxy(GtkTreeModel *tree_model, GtkTreeIter *iter)
{
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(tree_model));
	Gtk::TreeModel *tmp_tree_model = dynamic_cast<Gtk::TreeModel*>(object);
	if (tmp_tree_model)
		tmp_tree_model->do_unref_node(iter);
	else
	{
		GtkTreeModelIface *tmp_iface = GTK_TREE_MODEL_GET_IFACE(tree_model);
		GtkTreeModelIface *g_iface = static_cast<GtkTreeModelIface*>(g_type_interface_peek_parent(tmp_iface));
		if (g_iface->unref_node)
			g_iface->unref_node(tree_model, iter);
	}
}

void
Gtk::TreeModelIface::row_changed_proxy(GtkTreeModel *tree_model, GtkTreePath *path, GtkTreeIter *iter)
{
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(tree_model));
	Gtk::TreeModel *tmp_tree_model = dynamic_cast<Gtk::TreeModel*>(object);
	if (tmp_tree_model)
	{
		TreePath tmp_path(path);
		TreeIter tmp_iter(iter);
		tmp_tree_model->on_row_changed(tmp_path, tmp_iter);
	}
	else
	{
		GtkTreeModelIface *tmp_iface = GTK_TREE_MODEL_GET_IFACE(tree_model);
		GtkTreeModelIface *g_iface = static_cast<GtkTreeModelIface*>(g_type_interface_peek_parent(tmp_iface));
		if (g_iface->row_changed)
			g_iface->row_changed(tree_model, path, iter);
	}
}

void
Gtk::TreeModelIface::row_inserted_proxy(GtkTreeModel *tree_model, GtkTreePath *path, GtkTreeIter *iter)
{
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(tree_model));
	Gtk::TreeModel *tmp_tree_model = dynamic_cast<Gtk::TreeModel*>(object);
	if (tmp_tree_model)
	{
		TreePath tmp_path(path);
		TreeIter tmp_iter(iter);
		tmp_tree_model->on_row_inserted(tmp_path, tmp_iter);
	}
	else
	{
		GtkTreeModelIface *tmp_iface = GTK_TREE_MODEL_GET_IFACE(tree_model);
		GtkTreeModelIface *g_iface = static_cast<GtkTreeModelIface*>(g_type_interface_peek_parent(tmp_iface));
		if (g_iface->row_inserted)
			g_iface->row_inserted(tree_model, path, iter);
	}
}

void
Gtk::TreeModelIface::row_has_child_toggled_proxy(GtkTreeModel *tree_model, GtkTreePath *path, GtkTreeIter *iter)
{
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(tree_model));
	Gtk::TreeModel *tmp_tree_model = dynamic_cast<Gtk::TreeModel*>(object);
	if (tmp_tree_model)
	{
		TreePath tmp_path(path);
		TreeIter tmp_iter(iter);
		tmp_tree_model->on_row_has_child_toggled(tmp_path, tmp_iter);
	}
	else
	{
		GtkTreeModelIface *tmp_iface = GTK_TREE_MODEL_GET_IFACE(tree_model);
		GtkTreeModelIface *g_iface = static_cast<GtkTreeModelIface*>(g_type_interface_peek_parent(tmp_iface));
		if (g_iface->row_has_child_toggled)
			g_iface->row_has_child_toggled(tree_model, path, iter);
	}
}

void
Gtk::TreeModelIface::row_deleted_proxy(GtkTreeModel *tree_model, GtkTreePath *path)
{
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(tree_model));
	Gtk::TreeModel *tmp_tree_model = dynamic_cast<Gtk::TreeModel*>(object);
	if (tmp_tree_model)
	{
		TreePath tmp_path(path);
		tmp_tree_model->on_row_deleted(tmp_path);
	}
	else
	{
		GtkTreeModelIface *tmp_iface = GTK_TREE_MODEL_GET_IFACE(tree_model);
		GtkTreeModelIface *g_iface = static_cast<GtkTreeModelIface*>(g_type_interface_peek_parent(tmp_iface));
		if (g_iface->row_deleted)
			g_iface->row_deleted(tree_model, path);
	}
}

void
Gtk::TreeModelIface::rows_reordered_proxy(GtkTreeModel *tree_model, GtkTreePath *path, GtkTreeIter *iter, gint *new_order)
{
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(tree_model));
	Gtk::TreeModel *tmp_tree_model = dynamic_cast<Gtk::TreeModel*>(object);
	if (tmp_tree_model)
	{
		TreePath tmp_path(path);
		TreeIter tmp_iter(iter);
		tmp_tree_model->on_rows_reordered(tmp_path, tmp_iter, new_order);
	}
	else
	{
		GtkTreeModelIface *tmp_iface = GTK_TREE_MODEL_GET_IFACE(tree_model);
		GtkTreeModelIface *g_iface = static_cast<GtkTreeModelIface*>(g_type_interface_peek_parent(tmp_iface));
		if (g_iface->rows_reordered)
			g_iface->rows_reordered(tree_model, path, iter, new_order);
	}
}

/*  Overridable methods
 */

GtkTreeModelFlags
Gtk::TreeModel::do_get_flags()
{
	GtkTreeModelFlags result = (GtkTreeModelFlags)0;
	GtkTreeModelIface *g_iface = peek_parent<GtkTreeModelIface>(gtk_tree_model_iface());
	if (g_iface->get_flags)
		result = g_iface->get_flags(gtk_tree_model());
	return result;
}

gint
Gtk::TreeModel::do_get_n_columns()
{
	gint result = 0;
	GtkTreeModelIface *g_iface = peek_parent<GtkTreeModelIface>(gtk_tree_model_iface());
	if (g_iface->get_n_columns)
		result = g_iface->get_n_columns(gtk_tree_model());
	return result;
}

GType
Gtk::TreeModel::do_get_column_type(int index)
{
	GType result = G_TYPE_INVALID;
	GtkTreeModelIface *g_iface = peek_parent<GtkTreeModelIface>(gtk_tree_model_iface());
	if (g_iface->get_column_type)
		result = g_iface->get_column_type(gtk_tree_model(), index);
	return result;
}

gboolean
Gtk::TreeModel::do_get_iter(GtkTreeIter *iter, GtkTreePath *path)
{
	gboolean result = FALSE;
	GtkTreeModelIface *g_iface = peek_parent<GtkTreeModelIface>(gtk_tree_model_iface());
	if (g_iface->get_iter)
		result = g_iface->get_iter(gtk_tree_model(), iter, path);
	return result;
}

GtkTreePath*
Gtk::TreeModel::do_get_path(GtkTreeIter *iter)
{
	GtkTreePath *result = 0;
	GtkTreeModelIface *g_iface = peek_parent<GtkTreeModelIface>(gtk_tree_model_iface());
	if (g_iface->get_path)
		result = g_iface->get_path(gtk_tree_model(), iter);
	return result;
}

void
Gtk::TreeModel::do_get_value(GtkTreeIter *iter, int column, GValue *value)
{
	GtkTreeModelIface *g_iface = peek_parent<GtkTreeModelIface>(gtk_tree_model_iface());
	if (g_iface->get_value)
		g_iface->get_value(gtk_tree_model(), iter, column, value);
}

gboolean
Gtk::TreeModel::do_iter_next(GtkTreeIter *iter)
{
	gboolean result = FALSE;
	GtkTreeModelIface *g_iface = peek_parent<GtkTreeModelIface>(gtk_tree_model_iface());
	if (g_iface->iter_next)
		result = g_iface->iter_next(gtk_tree_model(), iter);
	return result;
}

gboolean
Gtk::TreeModel::do_iter_children(GtkTreeIter *iter, GtkTreeIter *parent)
{
	gboolean result = FALSE;
	GtkTreeModelIface *g_iface = peek_parent<GtkTreeModelIface>(gtk_tree_model_iface());
	if (g_iface->iter_children)
		result = g_iface->iter_children(gtk_tree_model(), iter, parent);
	return result;
}

gboolean
Gtk::TreeModel::do_iter_has_child(GtkTreeIter *iter)
{
	gboolean result = FALSE;
	GtkTreeModelIface *g_iface = peek_parent<GtkTreeModelIface>(gtk_tree_model_iface());
	if (g_iface->iter_has_child)
		result = g_iface->iter_has_child(gtk_tree_model(), iter);
	return result;
}

gint
Gtk::TreeModel::do_iter_n_children(GtkTreeIter *iter)
{
	gint result = 0;
	GtkTreeModelIface *g_iface = peek_parent<GtkTreeModelIface>(gtk_tree_model_iface());
	if (g_iface->iter_n_children)
		result = g_iface->iter_n_children(gtk_tree_model(), iter);
	return result;
}

gboolean
Gtk::TreeModel::do_iter_nth_child(GtkTreeIter *iter, GtkTreeIter *parent, int n)
{
	gboolean result = FALSE;
	GtkTreeModelIface *g_iface = peek_parent<GtkTreeModelIface>(gtk_tree_model_iface());
	if (g_iface->iter_nth_child)
		result = g_iface->iter_nth_child(gtk_tree_model(), iter, parent, n);
	return result;
}

gboolean
Gtk::TreeModel::do_iter_parent(GtkTreeIter *iter, GtkTreeIter *child)
{
	gboolean result = FALSE;
	GtkTreeModelIface *g_iface = peek_parent<GtkTreeModelIface>(gtk_tree_model_iface());
	if (g_iface->iter_parent)
		result = g_iface->iter_parent(gtk_tree_model(), iter, child);
	return result;
}

void
Gtk::TreeModel::do_ref_node(GtkTreeIter *iter)
{
	GtkTreeModelIface *g_iface = peek_parent<GtkTreeModelIface>(gtk_tree_model_iface());
	if (g_iface->ref_node)
		g_iface->ref_node(gtk_tree_model(), iter);
}

void
Gtk::TreeModel::do_unref_node(GtkTreeIter *iter)
{
	GtkTreeModelIface *g_iface = peek_parent<GtkTreeModelIface>(gtk_tree_model_iface());
	if (g_iface->unref_node)
		g_iface->unref_node(gtk_tree_model(), iter);
}

/*  Gtk::TreeModel signal handlers
 */

void
Gtk::TreeModel::on_row_changed(const TreePath& path, const TreeIter& iter)
{
	GtkTreeModelIface *g_iface = peek_parent<GtkTreeModelIface>(gtk_tree_model_iface());
	if (g_iface->row_changed)
		g_iface->row_changed(gtk_tree_model(), path.gtk_tree_path(), iter.gtk_tree_iter());
}

void
Gtk::TreeModel::on_row_inserted(const TreePath& path, const TreeIter& iter)
{
	GtkTreeModelIface *g_iface = peek_parent<GtkTreeModelIface>(gtk_tree_model_iface());
	if (g_iface->row_inserted)
		g_iface->row_inserted(gtk_tree_model(), path.gtk_tree_path(), iter.gtk_tree_iter());
}

void
Gtk::TreeModel::on_row_has_child_toggled(const TreePath& path, const TreeIter& iter)
{
	GtkTreeModelIface *g_iface = peek_parent<GtkTreeModelIface>(gtk_tree_model_iface());
	if (g_iface->row_has_child_toggled)
		g_iface->row_has_child_toggled(gtk_tree_model(), path.gtk_tree_path(), iter.gtk_tree_iter());
}

void
Gtk::TreeModel::on_row_deleted(const TreePath& path)
{
	GtkTreeModelIface *g_iface = peek_parent<GtkTreeModelIface>(gtk_tree_model_iface());
	if (g_iface->row_deleted)
		g_iface->row_deleted(gtk_tree_model(), path.gtk_tree_path());
}

void
Gtk::TreeModel::on_rows_reordered(const TreePath& path, const TreeIter& iter, int *new_order)
{
	GtkTreeModelIface *g_iface = peek_parent<GtkTreeModelIface>(gtk_tree_model_iface());
	if (g_iface->rows_reordered)
		g_iface->rows_reordered(gtk_tree_model(), path.gtk_tree_path(), iter.gtk_tree_iter(), new_order);
}

/*  Gtk::TreeModel signals
 */

const Gtk::TreeModel::RowChangedSignalType Gtk::TreeModel::row_changed_signal("row_changed");

const Gtk::TreeModel::RowInsertedSignalType Gtk::TreeModel::row_inserted_signal("row_inserted");

const Gtk::TreeModel::RowHasChildToggledSignalType Gtk::TreeModel::row_has_child_toggled_signal("row_has_child_toggled");

const Gtk::TreeModel::RowDeletedSignalType Gtk::TreeModel::row_deleted_signal("row_deleted");

const Gtk::TreeModel::RowsReorderedSignalType Gtk::TreeModel::rows_reordered_signal("rows_reordered");

/*  Gtk::TreeRowReference
 */

Gtk::TreeRowReference::TreeRowReference(TreeModel& model, const TreePath& path)
: G::Boxed(GTK_TYPE_TREE_ROW_REFERENCE, gtk_tree_row_reference_new(model.gtk_tree_model(), path.gtk_tree_path()), false)
{
}

Gtk::TreeRowReference::TreeRowReference(G::Object& proxy, TreeModel& model, const TreePath& path)
: G::Boxed(GTK_TYPE_TREE_ROW_REFERENCE, gtk_tree_row_reference_new_proxy(proxy.g_object(), model.gtk_tree_model(), path.gtk_tree_path()), false)
{
}

Gtk::TreeRowReference::TreeRowReference(GtkTreeRowReference *reference)
: G::Boxed(GTK_TYPE_TREE_ROW_REFERENCE, reference)
{
}

Gtk::TreeRowReference::TreeRowReference(GtkTreeRowReference *reference, bool copy)
: G::Boxed(GTK_TYPE_TREE_ROW_REFERENCE, reference, copy)
{
}

Gtk::TreeRowReference::TreeRowReference(const TreeRowReference& src)
: G::Boxed(GTK_TYPE_TREE_ROW_REFERENCE, src.g_boxed(), true)
{
}

Gtk::TreeRowReference::~TreeRowReference()
{
}

Gtk::TreeRowReference&
Gtk::TreeRowReference::operator=(const TreeRowReference& src)
{
	if (src.boxed_ != boxed_)
		copy(src);
	return *this;
}

Gtk::TreeRowReference::operator GtkTreeRowReference* () const
{
	return this ? gtk_tree_row_reference() : 0;
}

Pointer<Gtk::TreePath>
Gtk::TreeRowReference::get_path() const
{
	return G::Boxed::wrap<TreePath>(GTK_TYPE_TREE_PATH, gtk_tree_row_reference_get_path(gtk_tree_row_reference()), false);
}

bool
Gtk::TreeRowReference::valid() const
{
	return gtk_tree_row_reference_valid(gtk_tree_row_reference());
}

void
Gtk::TreeRowReference::inserted(const G::Object& proxy, const TreePath& path)
{
	gtk_tree_row_reference_inserted(proxy.g_object(), path.gtk_tree_path());
}

void
Gtk::TreeRowReference::deleted(const G::Object& proxy, const TreePath& path)
{
	gtk_tree_row_reference_deleted(proxy.g_object(), path.gtk_tree_path());
}

void
Gtk::TreeRowReference::reordered(const G::Object& proxy, const TreePath& path, const TreeIter& iter, int *new_order)
{
	gtk_tree_row_reference_reordered(proxy.g_object(), path.gtk_tree_path(), iter.gtk_tree_iter(), new_order);
}

