/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.test.remoting.transport.bisocket;

import java.lang.reflect.Field;
import java.net.InetAddress;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Map;
import java.util.Set;

import javax.management.MBeanServer;

import junit.framework.TestCase;

import org.apache.log4j.ConsoleAppender;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.apache.log4j.PatternLayout;
import org.jboss.remoting.Client;
import org.jboss.remoting.InvocationRequest;
import org.jboss.remoting.InvokerLocator;
import org.jboss.remoting.ServerInvocationHandler;
import org.jboss.remoting.ServerInvoker;
import org.jboss.remoting.callback.Callback;
import org.jboss.remoting.callback.HandleCallbackException;
import org.jboss.remoting.callback.InvokerCallbackHandler;
import org.jboss.remoting.callback.ServerInvokerCallbackHandler;
import org.jboss.remoting.transport.Connector;
import org.jboss.remoting.transport.PortUtil;
import org.jboss.remoting.transport.bisocket.BisocketServerInvoker;
import org.jboss.remoting.transport.socket.MicroSocketClientInvoker;


/**
 * Unit test for JBREM-767.
 * 
 * @author <a href="ron.sigal@jboss.com">Ron Sigal</a>
 * @version $Revision: 2732 $
 * <p>
 * Copyright Aug 4, 2007
 * </p>
 */
public class CallbackDisconnectTestCase extends TestCase
{
   public static int port;
   private static Logger log = Logger.getLogger(CallbackDisconnectTestCase.class);
   private static final String CALLBACK_TEST = "callbackTest";
   
   private static boolean firstTime = true;
   
   // remoting server connector
   private Connector connector;
   private InvokerLocator serverLocator;
   private TestInvocationHandler invocationHandler;

   
   /**
    * Sets up target remoting server.
    */
   public void setUp() throws Exception
   {
      if (firstTime)
      {
         firstTime = false;
         Logger.getLogger("org.jboss.remoting").setLevel(Level.INFO);
         Logger.getLogger("org.jboss.test.remoting").setLevel(Level.INFO);
         String pattern = "[%d{ABSOLUTE}] [%t] %5p (%F:%L) - %m%n";
         PatternLayout layout = new PatternLayout(pattern);
         ConsoleAppender consoleAppender = new ConsoleAppender(layout);
         Logger.getRootLogger().addAppender(consoleAppender);  
      }
   }

   
   public void tearDown()
   {
   }
   
   
   /**
    * Verifies that BisocketClientInvoker.handleDisconnect() wakes up any threads
    * blocked in BisocketClientInvoker.createSocket().
    */
   public void testWakeupsInHandleDisconnect() throws Throwable
   {
      log.info("entering " + getName());
      
      // Start server.
      String host = InetAddress.getLocalHost().getHostAddress();
      port = PortUtil.findFreePort(host);
      String locatorURI = getTransport() + "://" + host + ":" + port; 
      serverLocator = new InvokerLocator(locatorURI);
      log.info("Starting remoting server with locator uri of: " + locatorURI);
      HashMap config = new HashMap();
      config.put(InvokerLocator.FORCE_REMOTE, "true");
      config.put("timeout", "0");
      addExtraServerConfig(config);
      connector = new Connector(serverLocator, config);
      connector.create();
      invocationHandler = new TestInvocationHandler();
      connector.addInvocationHandler("sample", invocationHandler);
      connector.start();
      
      // Create client.
      HashMap clientConfig = new HashMap();
      clientConfig.put(InvokerLocator.FORCE_REMOTE, "true");
      addExtraClientConfig(clientConfig);
      final Client client = new Client(serverLocator, clientConfig);
      client.connect();
      assertEquals("abc", client.invoke("abc"));
      log.info("client is connected");
      
      // Add callback handler.
      TestInvokerCallbackHandler callbackHandler = new TestInvokerCallbackHandler();
      client.addListener(callbackHandler, new HashMap());
      client.invoke(CALLBACK_TEST);
      assertEquals(1, callbackHandler.counter);
      log.info("callback handler is installed");
      
      // Remove existing connections for callback clients.
      ServerInvoker serverInvoker = connector.getServerInvoker();
      Field field = ServerInvoker.class.getDeclaredField("callbackHandlers");
      field.setAccessible(true);
      Map callbackHandlers = (Map) field.get(serverInvoker);
      assertEquals(1, callbackHandlers.size());
      ServerInvokerCallbackHandler serverInvokerCallbackHandler;
      serverInvokerCallbackHandler = (ServerInvokerCallbackHandler) callbackHandlers.values().iterator().next();
      Client callbackClient = serverInvokerCallbackHandler.getCallbackClient();
      assertTrue(callbackClient.getInvoker() instanceof MicroSocketClientInvoker);
      MicroSocketClientInvoker callbackClientInvoker = (MicroSocketClientInvoker) callbackClient.getInvoker();
      field = MicroSocketClientInvoker.class.getDeclaredField("pool");
      field.setAccessible(true);
      LinkedList pool = (LinkedList) field.get(callbackClientInvoker);
      assertEquals(1, pool.size());
      pool.clear();
      
      // Prevent creation of new connections for callback clients.
      field = Client.class.getDeclaredField("callbackConnectors");
      field.setAccessible(true);
      Map callbackConnectors = (Map) field.get(client);
      assertEquals(1, callbackConnectors.size());
      Set callbackConnectorSet = (Set) callbackConnectors.values().iterator().next();
      assertEquals(1, callbackConnectorSet.size());
      Connector callbackConnector = (Connector) callbackConnectorSet.iterator().next();
      assertTrue(callbackConnector.getServerInvoker() instanceof BisocketServerInvoker);
      BisocketServerInvoker callbackServerInvoker = (BisocketServerInvoker) callbackConnector.getServerInvoker();
      field = BisocketServerInvoker.class.getDeclaredField("controlConnectionThreadMap");
      field.setAccessible(true);
      Map controlConnectionThreadMap = (Map) field.get(callbackServerInvoker);
      assertEquals(1, controlConnectionThreadMap.size());
      Thread controlConnectionThread = (Thread) controlConnectionThreadMap.values().iterator().next();
      controlConnectionThread.stop();
      
      // Verify that the call to sockets.wait() in BisocketClientInvoker will
      // wake up, even though the timeout value is set to 0.
      class TestThread extends Thread
      {
         public boolean done;
         public Throwable throwable;
         public void run()
         {
            try
            {
               log.info("calling client.invoke(CALLBACK_TEST)");
               client.invoke(CALLBACK_TEST);
               log.info("called client.invoke(CALLBACK_TEST)");
            }
            catch (Throwable t)
            {
               log.info("TestThread got expected exception: " + t.getMessage());
               throwable = t;
            }
            done = true;
         }
      }
      
      TestThread testThread = new TestThread();
      testThread.start();
      Thread.sleep(2000);
      callbackClient.disconnect();
      Thread.sleep(2000);
      assertTrue(testThread.done);
      assertNotNull(testThread.throwable);
      assertEquals("Error handling callback", testThread.throwable.getMessage());
      
      client.removeListener(callbackHandler);
      client.disconnect();
      connector.stop();
   }
   
   
   protected String getTransport()
   {
      return "bisocket";
   }
   
   
   protected void addExtraClientConfig(Map config) {}
   protected void addExtraServerConfig(Map config) {}
   

   static class TestInvocationHandler implements ServerInvocationHandler
   {
      public Set listeners = new HashSet();
      
      public void addListener(InvokerCallbackHandler callbackHandler)
      {
         listeners.add(callbackHandler);
      }
      public Object invoke(final InvocationRequest invocation) throws Throwable
      {
         if (CALLBACK_TEST.equals(invocation.getParameter()))
         {
            Iterator it = listeners.iterator();
            while (it.hasNext())
            {
               InvokerCallbackHandler handler = (InvokerCallbackHandler) it.next();
               handler.handleCallback(new Callback("test"));
            }
         }
         return invocation.getParameter();
      }
      public void removeListener(InvokerCallbackHandler callbackHandler) {}
      public void setMBeanServer(MBeanServer server) {}
      public void setInvoker(ServerInvoker invoker) {}
   }
   
   static class TestInvokerCallbackHandler implements InvokerCallbackHandler
   {
      public int counter;
      
      public void handleCallback(Callback callback) throws HandleCallbackException
      {
         counter++;
      }
   }
}