/*
 * libjingle
 * Copyright 2004--2005, Google Inc.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 *
 *  1. Redistributions of source code must retain the above copyright notice, 
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright notice,
 *     this list of conditions and the following disclaimer in the documentation
 *     and/or other materials provided with the distribution.
 *  3. The name of the author may not be used to endorse or promote products 
 *     derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF 
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR 
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF 
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "talk/base/common.h"
#include "talk/base/httpcommon.h"
#include "talk/base/stringutils.h"

namespace cricket {

//////////////////////////////////////////////////////////////////////
// Enum - TODO: expose globally later?
//////////////////////////////////////////////////////////////////////

bool find_string(size_t& index, const char* needle, const char* const haystack[], size_t max_index) {
  for (index=0; index<=max_index; ++index) {
	if (strcmp(needle, haystack[index]) == 0) {
	  return true;
	}
  }
  return false;
}

template<class E>
struct Enum {
  static const char** Names;
  static size_t Size;

  static inline const char* Name(E val) { return Names[val]; }
  static inline bool Parse(E& val, const char* name) { 
	size_t index;
	if (!find_string(index, name, Names, Size))
	  return false;
	val = static_cast<E>(index);
	return true;
  }

  E val;

  inline operator E&() { return val; }
  inline Enum& operator=(E rhs) { val = rhs; return *this; }

  inline const char* name() const { return Name(val); }
  inline bool assign(const char* name) { return Parse(val, name); }
  inline Enum& operator=(const char* rhs) { assign(rhs); return *this; }
};

#define ENUM(e,n) \
  template<> const char** Enum<e>::Names = n; \
  template<> size_t Enum<e>::Size = sizeof(n)/sizeof(n[0])

//////////////////////////////////////////////////////////////////////
// HttpCommon
//////////////////////////////////////////////////////////////////////

static const char* HVER_NAMES[HVER_LAST+1] = { "1.0", "1.1" };
ENUM(HttpVersion, HVER_NAMES);

const char* ToString(HttpVersion version) {
  return Enum<HttpVersion>::Name(version);
}

bool FromString(HttpVersion& version, const char* str) {
  return Enum<HttpVersion>::Parse(version, str);
}

static const char* HV_NAMES[HV_LAST+1] = { "GET", "POST", "PUT", "DELETE", "CONNECT", "HEAD" };
ENUM(HttpVerb, HV_NAMES);

const char* ToString(HttpVerb verb) {
  return Enum<HttpVerb>::Name(verb);
}

bool FromString(HttpVerb& verb, const char* str) {
  return Enum<HttpVerb>::Parse(verb, str);
}

//////////////////////////////////////////////////////////////////////
// HttpData
//////////////////////////////////////////////////////////////////////

void
HttpData::clear() {
  document.reset();
  m_headers.clear();
}

void
HttpData::changeHeader(const std::string& name, const std::string& value, HeaderCombine combine) {
  if (combine == HC_AUTO) {
    combine = HC_YES;
    const char* NO_AUTO_COMBINE[] = { "Set-Cookie", "WWW-Authenticate", "Proxy-Authenticate", 0 };
    for (size_t i=0; NO_AUTO_COMBINE[i]; ++i) {
      if (_stricmp(name.c_str(), NO_AUTO_COMBINE[i]) != 0)
        continue;
      combine = HC_NO;
      break;
    }
  } else if (combine == HC_REPLACE) {
    m_headers.erase(name);
	combine = HC_NO;
  }
  // At this point, combine in [YES, NO, NEW]
  if (combine != HC_NO) {
    HeaderMap::iterator it = m_headers.find(name);
    if (it != m_headers.end()) {
      if (combine == HC_YES) {
        it->second.append(",");
        it->second.append(value);
	  }
      return;
	}
  }
  m_headers.insert(HeaderMap::value_type(name, value));
}

void
HttpData::clearHeader(const std::string& name) {
  m_headers.erase(name);
}

bool
HttpData::hasHeader(const std::string& name, std::string& value) const {
  HeaderMap::const_iterator it = m_headers.find(name);
  if (it == m_headers.end()) {
    return false;
  } else {
    value = it->second;
    return true;
  }
}

void
HttpData::setContent(const std::string& content_type, StreamInterface* document) {
  ASSERT(document != NULL);
  this->document.reset(document);
  setHeader("Content-Type", content_type);
  size_t content_length = 0;
  if (this->document->GetSize(&content_length)) {
    char buffer[32];
    sprintfn(buffer, sizeof(buffer), "%d", content_length);
    setHeader("Content-Length", buffer);
  } else {
    setHeader("Transfer-Encoding", "chunked");
  }
}

//
// HttpRequestData
//

void
HttpRequestData::clear() {
  HttpData::clear();
  verb = HV_GET;
  path.clear();
}

size_t
HttpRequestData::formatLeader(char* buffer, size_t size) {
  ASSERT(path.find(' ') == std::string::npos);
  return sprintfn(buffer, size, "%s %.*s HTTP/%s", ToString(verb), path.size(), path.data(), ToString(version));
}

HttpError
HttpRequestData::parseLeader(const char* line, size_t len) {
  UNUSED(len);
  uint32 vmajor, vminor;
  int vend, dstart, dend;
  if ((sscanf(line, "%*s%n %n%*s%n HTTP/%lu.%lu", &vend, &dstart, &dend, &vmajor, &vminor) != 2) || (vmajor != 1)) {
    return HE_PROTOCOL;
  }
  if (vminor == 0) {
    version = HVER_1_0;
  } else if (vminor == 1) {
    version = HVER_1_1;
  } else {
    return HE_PROTOCOL;
  }
  std::string sverb(line, vend);
  if (!FromString(verb, sverb.c_str())) {
    return HE_PROTOCOL; // !?! HC_METHOD_NOT_SUPPORTED?
  }
  path.assign(line + dstart, line + dend);
  return HE_NONE;
}

//
// HttpResponseData
//

void
HttpResponseData::clear() {
  HttpData::clear();
  scode = HC_INTERNAL_SERVER_ERROR;
  message.clear();
}
 
void
HttpResponseData::success(uint32 scode) {
  this->scode = scode;
  message.clear();
  setHeader("Content-Length", "0");
}

void
HttpResponseData::success(const char* content_type, StreamInterface* document, uint32 scode) {
  this->scode = scode;
  message.erase(message.begin(), message.end());
  setContent(content_type, document);
}

void
HttpResponseData::redirect(const char* location, uint32 scode) {
  this->scode = scode;
  message.clear();
  setHeader("Location", location);
  setHeader("Content-Length", "0");
}

void
HttpResponseData::error(uint32 scode) {
  this->scode = scode;
  message.clear();
  setHeader("Content-Length", "0");
}

size_t
HttpResponseData::formatLeader(char* buffer, size_t size) {
  size_t len = sprintfn(buffer, size, "HTTP/%s %lu", ToString(version), scode);
  if (!message.empty()) {
    len += sprintfn(buffer + len, size - len, " %.*s", message.size(), message.data());
  }
  return len;
}

HttpError
HttpResponseData::parseLeader(const char* line, size_t len) {
  size_t pos = 0;
  uint32 vmajor, vminor;
  if ((sscanf(line, "HTTP/%lu.%lu %lu%n", &vmajor, &vminor, &scode, &pos) != 3) || (vmajor != 1)) {
    return HE_PROTOCOL;
  }
  if (vminor == 0) {
    version = HVER_1_0;
  } else if (vminor == 1) {
    version = HVER_1_1;
  } else {
    return HE_PROTOCOL;
  }
  while ((pos < len) && isspace(line[pos])) ++pos;
  message.assign(line + pos, len - pos);
  return HE_NONE;
}

} // namespace cricket
