/*
 * libjingle
 * Copyright 2004--2005, Google Inc.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 *
 *  1. Redistributions of source code must retain the above copyright notice, 
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright notice,
 *     this list of conditions and the following disclaimer in the documentation
 *     and/or other materials provided with the distribution.
 *  3. The name of the author may not be used to endorse or promote products 
 *     derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF 
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR 
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF 
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef TALK_BASE_HTTPCOMMON_H__
#define TALK_BASE_HTTPCOMMON_H__

#include <map>
#include <string>
#include "talk/base/basictypes.h"
#include "talk/base/scoped_ptr.h"
#include "talk/base/stringutils.h"
#include "talk/base/stream.h"

namespace cricket {

//////////////////////////////////////////////////////////////////////
// Constants
//////////////////////////////////////////////////////////////////////

enum HttpCode { 
  HC_OK = 200, HC_NO_CONTENT = 204,

  HC_MOVED_PERMANENTLY = 301, HC_FOUND = 302, HC_SEE_OTHER = 303,
  HC_NOT_MODIFIED = 304, HC_MOVED_TEMPORARILY = 307,

  HC_BAD_REQUEST = 400, HC_UNAUTHORIZED = 401, HC_FORBIDDEN = 403,
  HC_NOT_FOUND = 404, HC_PROXY_AUTHENTICATION_REQUIRED = 407,

  HC_INTERNAL_SERVER_ERROR = 500 
};

enum HttpVersion { HVER_1_0, HVER_1_1, HVER_LAST = HVER_1_1 };

enum HttpVerb { HV_GET, HV_POST, HV_PUT, HV_DELETE, HV_CONNECT, HV_HEAD, HV_LAST = HV_HEAD };

enum HttpError {
  HE_NONE,
  HE_PROTOCOL, HE_DISCONNECTED, HE_OVERFLOW,
  HE_SOCKET, HE_SHUTDOWN, HE_OPERATION_CANCELLED,
  HE_AUTH,                // Proxy Authentication Required
  HE_CERTIFICATE_EXPIRED, // During SSL negotiation
  HE_STREAM,              // Problem reading or writing to the document
  HE_DEFAULT
};
inline HttpError mkerr(HttpError err, HttpError def_err = HE_DEFAULT) { return (err != HE_NONE) ? err : def_err; }

const char* ToString(HttpVersion version);
bool FromString(HttpVersion& version, const char* str);

const char* ToString(HttpVerb verb);
bool FromString(HttpVerb& verb, const char* str);

inline bool is_redirect(uint32 scode) { return ((scode / 100) == 3); }

// functional for insensitive std::string compare
struct iless {
  bool operator()(const std::string& lhs, const std::string& rhs) const {
    return (::_stricmp(lhs.c_str(), rhs.c_str()) < 0);
  }
};

//////////////////////////////////////////////////////////////////////
// Url
//////////////////////////////////////////////////////////////////////

const uint16 HTTP_DEFAULT_PORT = 80;
const uint16 HTTP_SECURE_PORT = 443;

template<class CTYPE>
class Url {
public:
  typedef typename Traits<CTYPE>::string string;

  // TODO(bpm): Implement Encode/Decode
  static int Encode(const CTYPE* source, CTYPE* destination, size_t len);
  static int Encode(const string& source, string& destination);
  static int Decode(const CTYPE* source, CTYPE* destination, size_t len);
  static int Decode(const string& source, string& destination);

  Url(const CTYPE* url);
  Url(const CTYPE* path, const CTYPE* server, uint16 port = HTTP_DEFAULT_PORT)
    : m_server(server), m_path(path), m_port(port),
      m_secure(HTTP_SECURE_PORT == port) {
    ASSERT(m_path.empty() || (m_path[0] == static_cast<CTYPE>('/')));
  }
  
  bool valid() const { return !m_server.empty(); }
  const CTYPE* url(CTYPE* buffer, size_t len);
  const CTYPE* server() const { return m_server.c_str(); }
  const CTYPE* path() const { return m_path.c_str(); }
  uint16 port() const { return m_port; }
  bool secure() const { return m_secure; }

  void server(const CTYPE* val) { m_server = val; }
  void path(const CTYPE* val) {
    m_path = val;
    ASSERT(m_path.empty() || (m_path[0] == static_cast<CTYPE>('/')));
  }
  void port(uint16 val) { m_port = val; }
  void secure(bool val) { m_secure = val; }

private:
  string m_server, m_path;
  uint16 m_port;
  bool m_secure;
};

//////////////////////////////////////////////////////////////////////
// HttpData
//////////////////////////////////////////////////////////////////////

struct HttpData {
  typedef std::multimap<std::string, std::string, iless> HeaderMap;
  typedef HeaderMap::const_iterator const_iterator;
  
  HttpVersion version;
  scoped_ptr<StreamInterface> document;

  HttpData() : version(HVER_1_1) { }

  enum HeaderCombine { HC_YES, HC_NO, HC_AUTO, HC_REPLACE, HC_NEW };
  void changeHeader(const std::string& name, const std::string& value, HeaderCombine combine);
  inline void addHeader(const std::string& name, const std::string& value, bool append = true) { changeHeader(name, value, append ? HC_AUTO : HC_NO); }
  inline void setHeader(const std::string& name, const std::string& value, bool overwrite = true) { changeHeader(name, value, overwrite ? HC_REPLACE : HC_NEW); }
  void clearHeader(const std::string& name);

  // keep in mind, this may not do what you want in the face of multiple headers
  bool hasHeader(const std::string& name, std::string& value) const;

  inline const_iterator begin() const { return m_headers.begin(); }
  inline const_iterator end() const { return m_headers.end(); }
  inline const_iterator begin(const std::string& name) const { return m_headers.lower_bound(name); }
  inline const_iterator end(const std::string& name) const { return m_headers.upper_bound(name); }
  
  void setContent(const std::string& content_type, StreamInterface* document);

  virtual size_t formatLeader(char* buffer, size_t size) = 0;
  virtual HttpError parseLeader(const char* line, size_t len) = 0;
  
protected:  
  virtual ~HttpData() { }
  void clear();

private:
  HeaderMap m_headers;
};

struct HttpRequestData : public HttpData {
  HttpVerb verb;
  std::string path;

  HttpRequestData() : verb(HV_GET) { }

  void clear();

  virtual size_t formatLeader(char* buffer, size_t size);
  virtual HttpError parseLeader(const char* line, size_t len);
};

struct HttpResponseData : public HttpData {
  uint32 scode;
  std::string message;

  HttpResponseData() : scode(HC_INTERNAL_SERVER_ERROR) { }
  void clear();

  // Convenience methods
  // TODO(bpm): change these to set_success, set_error
  void success(uint32 scode = HC_OK);
  void success(const char* content_type, StreamInterface* document, uint32 scode = HC_OK);
  void redirect(const char* location, uint32 scode = HC_MOVED_TEMPORARILY);
  void error(uint32 scode);

  virtual size_t formatLeader(char* buffer, size_t size);
  virtual HttpError parseLeader(const char* line, size_t len);
};

//////////////////////////////////////////////////////////////////////

} // namespace cricket

#endif // TALK_BASE_HTTPCOMMON_H__
