#!/usr/bin/env python
#------------------------------------------------------------------------------
#   This file is part of the "Mathematical Expression Interpreter" library.
#
#   Copyright (C) 2008-2009  EDF
#
#   This library is free software; you can redistribute it and/or
#   modify it under the terms of the GNU Lesser General Public
#   License as published by the Free Software Foundation; either
#   version 2.1 of the License, or (at your option) any later version.
#
#   This library is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#   Lesser General Public License for more details.
#
#   You should have received a copy of the GNU Lesser General Public
#   License along with this library; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#------------------------------------------------------------------------------

import unittest
import sys, math
import mei

class MeiTestCase(unittest.TestCase):
    """
    """

    def check_mei_type(self):
        """Check whether type could be create"""
        print "\ncheck_mei_type\n"
        #print dir(mei)
        #print "type IDENTIFIANT : ", mei.IDENTIFIANT

        mei.data_t.func = math.cos
        #print "cos(pi) = ", mei.data_t.func(3.14)

        mei.item.char = "toto"
        #print "toto = ", mei.item.char

        mei.HashTable.record = 13
        #print "treize = ", mei.HashTable.record

        h = mei.HashTable()


    def check_mei_tree_new(self):
        """Check whether a new mei interpreter could be create"""
        print "\ncheck_mei_tree_new\n"
        expr = "x=(+2.5); y = 8.7;"
        intr = mei.mei_tree_new(expr)

        msg = 'Could not create a new mei interpreter'

        assert intr.string  == expr, msg
        assert intr.lines   == [],   msg
        assert intr.columns == [],   msg
        assert intr.labels  == [],   msg
        assert intr.errors  == 0,    msg

        intr = mei.mei_tree_destroy(intr)


    def check_mei_tree_builder(self):
        """Check whether a mei interpreter could be build"""
        print "\ncheck_mei_tree_builder\n"
        expr = "y = cos(-pi) ; tt = y*sqrt(2.2) ; abs(tt);"
        intr = mei.mei_tree_new(expr)
        iok  = mei.mei_tree_builder(intr)

        msg = 'Could not build a mei interpreter'

        assert iok == 0, msg
        assert intr.string  == expr, msg
        assert intr.lines   == [],   msg
        assert intr.columns == [],   msg
        assert intr.labels  == [],   msg
        assert intr.errors  == 0,    msg

        intr = mei.mei_tree_destroy(intr)


    def check_mei_tree_insert(self):
        """Check whether a symbol could be insert in an interpreter"""
        print "\ncheck_mei_tree_insert\n"
        expr = "y = cos(-pi) ; tt = K+y ; abs(tt);"
        intr = mei.mei_tree_new(expr)

        iok = mei.mei_tree_builder(intr)
        assert iok == 1, 'Could not build a mei interpreter'

        # the symbol table is completed with a value fo K

        mei.mei_tree_insert(intr, "K", -5.3e2)
        assert mei.mei_tree_lookup(intr, "K") == -530, 'Could not insert a symbol'

        intr = mei.mei_tree_destroy(intr)


    def check_mei_tree_find_symbol(self):
        """Check whether a symbol could be find in an interpreter"""
        print "\ncheck_mei_tree_find_symbol\n"
        expr = "y = cos(-pi) ; tt = K+y ; abs(tt);"
        intr = mei.mei_tree_new(expr)

        # the interpreter is not build,
        # the symbol table is initialized with functions and constant
        # but contains no variable

        iok = mei.mei_tree_find_symbol(intr, "K")
        assert iok == 1, 'Could not find a symbol'
        assert intr.errors == 1, 'Could not find a symbol'
        assert intr.labels[0] == "K", 'Could not find a symbol'

        # the interpreter can not build because K is unknown
        # by the symbol table

        iok = mei.mei_tree_builder(intr)
        assert iok == 1, 'Could not build an interpreter'

        # the symbol table is completed with a value fo K

        mei.mei_tree_insert(intr, "K", -5.3e2)
        assert mei.mei_tree_lookup(intr, "K") == -530, 'Could not find a symbol'

        # the interpreter can be build

        iok = mei.mei_tree_builder(intr)
        assert iok == 0, 'Could not build an interpreter'

        # one can find the symbol K in the symbol table

        iok = mei.mei_tree_find_symbol(intr, "K")
        assert iok == 0, 'Could not find a symbol'

        # one can find a list of symbol

        l = ["y", "tt", "A", "B"]
        iok = mei.mei_tree_find_symbols(intr, len(l), l)
        assert iok == 2, 'Could not find a list of symbols'
        assert intr.errors == 2, 'Could not find a list of symbols'
        assert intr.labels[0] == "A", 'Could not find a list of symbols'
        assert intr.labels[1] == "B", 'Could not find a list of symbols'

        intr = mei.mei_tree_destroy(intr)


#    def check_mei_tree_check_required_symbols(self):
#        """Check whether a symbol could be find in an interpreter"""
#        expr = "y = cos(-pi) ; x = 2e2 ; z = x+y ;"
#        intr = mei.mei_tree_new(expr)
#
#        iok = mei.mei_tree_check_required_symbol(intr, "z")
#        assert iok == 0, 'Could not find a symbol'
#
#        l = ["x", "y"]
#        iok = mei.mei_tree_check_required_symbols(intr, len(l), l)
#        assert iok == 0, 'Could not evaluate x or y'
#
#        l = ["A", "y", "x", "B"]
#        iok = mei.mei_tree_check_required_symbols(intr, len(l), l)
#        assert iok == 2, 'Could not evaluate A, y, x or B'
#
#        intr = mei.mei_tree_destroy(intr)


    def check_mei_evaluate(self):
        """Check whether an interpreter could be evaluate"""
        print "\ncheck_mei_evaluate\n"
        expr = "x=(+2.5); y=8.7; zz=y+x; yy = zz+0.8; yy ;\n"
        intr = mei.mei_tree_new(expr)

        msg = "Could not evaluate an expression"

        iok =  mei.mei_tree_builder(intr)

        assert iok == 0, msg
        assert mei.mei_evaluate(intr) == 12, msg

        intr = mei.mei_tree_destroy(intr)


    def check_mei_tree_lookup(self):
        """Check whether a symbol value could be get from the symbol table"""
        print "\ncheck_mei_tree_lookup\n"
        expr = "x=(+2.5); y=8.7; zz=y+x; yy = zz+0.8; \n"
        intr = mei.mei_tree_new(expr)

        msg = "Could not a value from the symbol table"

        iok = mei.mei_tree_builder(intr)
        mei.mei_evaluate(intr)

        assert iok == 0, 'Could not build a mei interpreter'
        assert mei.mei_tree_lookup(intr, "zz") == 11.2, msg
        assert mei.mei_tree_lookup(intr, "yy") == 12,   msg

        intr = mei.mei_tree_destroy(intr)


        expr = "Temp.C = 27.5; \n"
        intr = mei.mei_tree_new(expr)

        iok = mei.mei_tree_builder(intr)
        mei.mei_evaluate(intr)

        assert iok == 0, 'Could not build a mei interpreter'
        assert mei.mei_tree_lookup(intr, "Temp.C") == 27.5, msg

        intr = mei.mei_tree_destroy(intr)


        expr = "Temp_C = 27.5; \n"
        intr = mei.mei_tree_new(expr)

        iok = mei.mei_tree_builder(intr)
        mei.mei_evaluate(intr)

        assert iok == 0, 'Could not build a mei interpreter'
        assert mei.mei_tree_lookup(intr, "Temp_C") == 27.5, msg

        intr = mei.mei_tree_destroy(intr)


        expr = "Temp-C = -27.5; \n"
        intr = mei.mei_tree_new(expr)

        iok = mei.mei_tree_builder(intr)
        mei.mei_evaluate(intr)

        assert iok == 0, 'Could not build a mei interpreter'
        assert mei.mei_tree_lookup(intr, "Temp-C") == -27.5, msg

        intr = mei.mei_tree_destroy(intr)


    def check_mei_error(self):
        """Check whether an error could be detected"""
        print "\ncheck_mei_error\n"
        expr = "x=(+2.5); y=8.7; zz=y+x?; yy = zz+%0.8; \n"

        err1 = [1, 1]
        err2 = [24, 34]
        err3 = ['Error: unknown character: ? \n',
                'Error: unknown character: % \n']

        intr = mei.mei_tree_new(expr)

        iok = mei.mei_tree_builder(intr)
        msg = 'Could not detect interpreter errors'

        assert iok == 2, msg

        assert intr.string  == expr, msg
        assert intr.errors  == iok,  msg
        assert intr.lines   == err1, msg
        assert intr.columns == err2, msg
        assert intr.labels  == err3, msg

        intr = mei.mei_tree_destroy(intr)


    def check_mei_tree_destroy(self):
        """Check whether a mei interpreter could be destroy"""
        print "\ncheck_mei_tree_destroy\n"
        expr = "x=(+2.5); y = 8.7;"
        intr = mei.mei_tree_new(expr)
        intr = mei.mei_tree_destroy(intr)

        assert intr == None, 'Could not destroy a mei interpreter'


def suite():
    testSuite = unittest.makeSuite(MeiTestCase, "check")
    return testSuite


def runTest():
    runner = unittest.TextTestRunner()
    runner.run(suite())


if __name__ == '__main__':
    runTest()
