(**
   Implements a load-like percentage display.
**)

MODULE VO:Load;

(*
    Implements a load-like percentage display.
    Copyright (C) 1997  Tim Teulings (rael@edge.ping.de)



    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)

IMPORT D   := VO:Base:Display,
       F   := VO:Base:Frame,
       O   := VO:Base:Object,
       U   := VO:Base:Util,

       V   := VO:Model:Value,

       G   := VO:Object,
       S   := VO:Segment,

       co  := IntStr,
       str := Strings;

TYPE
  Prefs*     = POINTER TO PrefsDesc;

  (**
    In this class all preferences stuff of the button is stored.
  **)

  PrefsDesc* = RECORD (G.PrefsDesc)
               END;

  Load*     = POINTER TO LoadDesc;
  LoadDesc* = RECORD (G.ImageDesc)
                 labelTxt : S.Segment;
                 labelSfx : U.Text;
                 top,
                 bottom   : LONGINT;
                 onColor,
                 maxColor,
                 mediumColor,
                 offColor : LONGINT;
                 current  : V.ValueModel;
                 framed,
                 vertical,
                 labeled  : BOOLEAN;
               END;

VAR
  prefs* : Prefs;

  PROCEDURE (p : Prefs) Init*;

  BEGIN
    p.Init^;

    p.frame:=F.double3DIn;
  END Init;

  PROCEDURE (l : Load) Init*;

  BEGIN
    l.Init^;

    l.SetBackground(D.blackColor);

    l.SetPrefs(prefs);

    l.top:=100;
    l.bottom:=0;

    l.current:=NIL;

    l.labelTxt:=NIL;
    l.labelSfx:=NIL;

    l.vertical:=TRUE;
    l.framed:=TRUE;
    l.labeled:=FALSE;

    l.onColor:=D.shineColor;
    l.maxColor:=l.onColor;
    l.mediumColor:=l.onColor;
    l.offColor:=D.shadowColor;
  END Init;

  (**
    Tell, if the load should be display horizontally or
    vertically.

    Default is vertical.
  **)

  PROCEDURE (l : Load) SetVertical*(vertical : BOOLEAN);

  BEGIN
    l.vertical:=vertical;
    IF l.visible THEN
      l.Redraw;
    END;
  END SetVertical;

  (**
    Tell, if the object should display its label.
  **)

  PROCEDURE (l : Load) ShowLabel*(useLabel : BOOLEAN);

  BEGIN
    l.labeled:=useLabel;
    IF l.visible THEN
      l.Redraw;
    END;
  END ShowLabel;

  (**
    Tell, if the object should be framed. Defaults to TRUE.
  **)

  PROCEDURE (l : Load) ShowFrame*(framed : BOOLEAN);

  BEGIN
    (* We cannot switch back to using no frame if we already generated one *)
    ASSERT(l.objectFrame=NIL);

    l.framed:=framed;
  END ShowFrame;

  (**
    Set the integer model which represents the current value.
  **)

  PROCEDURE (l : Load) SetModel*(model : O.Model);

  BEGIN
    IF l.current#NIL THEN
      l.UnattachModel(l.current);
    END;
    IF (model#NIL) & (model IS V.ValueModel) THEN
      l.current:=model(V.ValueModel);
      l.AttachModel(l.current);
      IF l.visible THEN
        l.Redraw;
      END;
    ELSE
      l.current:=NIL;
    END;
  END SetModel;

  (**
    This function is used to check if an argument to SetModel
    was successfully accepted.
   **)

  PROCEDURE (l : Load) ModelAccepted * (m : O.Model):BOOLEAN;

  BEGIN
    RETURN m=l.current
  END ModelAccepted;

  PROCEDURE (l : Load) CalcSize*;

  VAR
    window : D.Window;

  BEGIN
    IF ~(G.inited IN l.flags) THEN
      IF D.display.colorMode=D.colorMode THEN
        l.onColor:=D.display.AllocateNamedColor("green",l.onColor);
        l.maxColor:=D.display.AllocateNamedColor("red",l.maxColor);
        l.mediumColor:=D.display.AllocateNamedColor("yellow",l.mediumColor);
      ELSIF D.display.colorMode=D.monochromeMode THEN
        l.onColor:=D.blackColor;
        l.maxColor:=l.onColor;
        l.mediumColor:=l.onColor;
        l.offColor:=D.whiteColor;
        l.SetBackground(D.whiteColor);
      END;
    END;

    IF l.framed THEN
      l.SetObjectFrame(l.prefs.frame);
    ELSE
      l.SetObjectFrame(F.none);
    END;

    l.width:=2*D.display.spaceWidth;
    l.height:=2*D.display.spaceHeight;

    IF l.labeled THEN
      NEW(l.labelTxt);
      l.labelTxt.Init;
      l.labelTxt.SetParent(l);
      l.labelTxt.SetStringWidth(3);
      l.CopyBackground(l.labelTxt);
      l.labelTxt.SetForeground(l.onColor);
      l.labelTxt.CalcSize;

      IF l.vertical THEN
        l.width:=U.MaxLong(l.width,l.labelTxt.oWidth);
        INC(l.height,D.display.spaceHeight DIV 2 + l.labelTxt.oHeight);
      ELSE
        INC(l.width,D.display.spaceWidth DIV 2 + l.labelTxt.oWidth);
        l.height:=U.MaxLong(l.height,l.labelTxt.oHeight);
      END;
    END;

    l.minWidth:=l.width;
    l.minHeight:=l.height;

    window:=l.GetWindow();
    window.AddFreeList(l);

    l.CalcSize^;
  END CalcSize;

  (**
    Set the bottom and top value the object should use for displaying.
  **)

  PROCEDURE (l : Load) SetRange*(bottom,top : LONGINT);

  BEGIN
    IF (bottom#l.bottom) OR (top#l.top) THEN
      l.bottom:=bottom;
      l.top:=top;

      IF l.visible THEN
        l.Redraw;
      END;
    END;
  END SetRange;

  (**
    Refresh loadmeter and text.
  **)

  PROCEDURE (l : Load) DrawLoad;

  VAR
    boxWidth,
    boxHeight,
    lines,
    current,
    top,left,
    mark,help,
    maxMark,
    mediumMark,
    value      : LONGINT;
    text       : ARRAY 4 OF CHAR;
    draw       : D.DrawInfo;

  BEGIN
    draw:=l.GetDrawInfo();


    IF l.current.IsNull() THEN
      value:=l.bottom;
    ELSE
      value:=l.current.GetLongint();
    END;

    boxHeight:=l.height;
    boxWidth:=l.width;

    IF l.labeled THEN
      IF l.vertical THEN
        DEC(boxHeight,D.display.spaceHeight DIV 2+l.labelTxt.oHeight);
      ELSE
        DEC(boxWidth,D.display.spaceWidth DIV 2+l.labelTxt.oWidth);
      END;
    END;

    IF l.vertical THEN
      lines:=boxHeight DIV 4;
    ELSE
      lines:=boxWidth DIV 4;
    END;

    help:=((value-l.bottom)*100) DIV (l.top-l.bottom+1);

    mark:=(help*lines) DIV 100;
    IF (help*lines) MOD 100>=50 THEN
      INC(mark);
    END;

    maxMark:=(75*lines) DIV 100;
    IF (75*lines) MOD 100>=50 THEN
      INC(maxMark);
    END;

    mediumMark:=(50*lines) DIV 100;
    IF (50*lines) MOD 100>=50 THEN
      INC(mediumMark);
    END;

    IF l.labeled THEN
      help:=((value-l.bottom)*100) DIV (l.top-l.bottom+1);
      IF ((value-l.bottom)*100) DIV (l.top-l.bottom+1)>=50 THEN
        INC(help);
      END;
      IF help>=75 THEN
        l.labelTxt.SetForeground(l.maxColor);
      ELSIF help>=50 THEN
        l.labelTxt.SetForeground(l.mediumColor);
      ELSE
        l.labelTxt.SetForeground(l.onColor);
      END;
      co.IntToStr(help,text);
      IF help<100 THEN
        IF help>=10 THEN
          str.Insert(" ",0,text);
        ELSE
          str.Insert("  ",0,text);
        END;
      END;
      l.labelTxt.SetString(text);
    END;

    current:=1;
    IF l.vertical THEN
      IF l.labeled THEN
        top:=l.y+(l.height-(2*lines-1)*2-D.display.spaceHeight DIV 2 -l.labelTxt.oHeight) DIV 2;
      ELSE
        top:=l.y+(l.height-(2*lines-1)*2) DIV 2;
      END;
      left:=l.x+(l.width-boxWidth) DIV 2;
      WHILE current<=lines DO
        IF current>lines-mark THEN
          IF current<=lines-maxMark THEN
            draw.PushForeground(l.maxColor);
          ELSIF current<=lines-mediumMark THEN
            draw.PushForeground(l.mediumColor);
          ELSE
            draw.PushForeground(l.onColor);
          END;
        ELSE
          draw.PushForeground(l.offColor);
        END;
        draw.DrawLine(left,top,left+boxWidth-1,top);
        draw.DrawLine(left,top+1,left+boxWidth-1,top+1);
        draw.PopForeground;
        INC(top,4);
        INC(current);
      END;
      IF l.labeled THEN
        l.labelTxt.Move(l.x+(l.width-l.labelTxt.oWidth) DIV 2,
                        l.y+l.height-1-l.labelTxt.oHeight);
        l.labelTxt.Draw(l.oX,l.oY,l.oWidth,l.oHeight);
      END;
    ELSE
      top:=l.y+(l.height-boxHeight) DIV 2;
      IF l.labeled THEN
        left:=l.x+(l.width-(2*lines-1)*2-D.display.spaceWidth DIV 2 -l.labelTxt.oWidth) DIV 2;
      ELSE
        left:=l.x+(l.width-(2*lines-1)*2) DIV 2;
      END;
      WHILE current<=lines DO
        IF current<=mark THEN
          IF current>maxMark THEN
            draw.PushForeground(l.maxColor);
          ELSIF current>mediumMark THEN
            draw.PushForeground(l.mediumColor);
          ELSE
            draw.PushForeground(l.onColor);
          END;
        ELSE
          draw.PushForeground(l.offColor);
        END;
        draw.DrawLine(left,top,left,top+boxHeight-1);
        draw.DrawLine(left+1,top,left+1,top+boxHeight-1);
        draw.PopForeground;
        INC(left,4);
        INC(current);
      END;
    END;
  END DrawLoad;

  PROCEDURE (l : Load) Draw*(x,y,w,h : LONGINT);

  VAR
    draw : D.DrawInfo;

  BEGIN
    l.Draw^(x,y,w,h);

    IF ~l.Intersect(x,y,w,h) THEN
      RETURN;
    END;

    draw:=l.GetDrawInfo();

    l.DrawBackground(l.x,l.y,l.width,l.height);

    l.DrawLoad;
  END Draw;

  PROCEDURE (l : Load) Hide*;

  BEGIN
    IF l.visible THEN
      l.DrawHide;
      l.Hide^;
    END;
  END Hide;

  PROCEDURE (l : Load) Resync*(model : O.Model; msg : O.ResyncMsg);

  BEGIN
    IF l.visible THEN
      l.DrawLoad;
    END;
  END Resync;

  PROCEDURE (l : Load) Free*;

  BEGIN
    D.display.FreeColor(l.onColor);
    D.display.FreeColor(l.mediumColor);
    D.display.FreeColor(l.maxColor);
  END Free;

  PROCEDURE CreateLoad*():Load;

  VAR
    load : Load;

  BEGIN
    NEW(load);
    load.Init;

    RETURN load;
  END CreateLoad;

BEGIN
  NEW(prefs);
  prefs.Init;
END VO:Load.