MODULE VO:EditSyntaxParser;

IMPORT D   := VO:Base:Display,

       R   := VO:EditRun,

              Ascii,
              Err;

TYPE

  (**
    Holds all information necessary for drawing a line.
    start=TRUE marks a new area of equal style.
  **)

  DrawInfo = RECORD
               aPen-,
               bPen-   : LONGINT;
               style-  : SET;
               start-,
               handled : BOOLEAN;
             END;


  (**
    An entry for each character
    in the alphabet.
  **)

  Character = RECORD
                aPen,
                bPen        : LONGINT; (* color of the character            *)
                newChar     : CHAR;    (* The substitute for this character *)
              END;

  SyntaxParser*     = POINTER TO SyntaxParserDesc;
  SyntaxParserDesc* = RECORD
                        table-      : ARRAY 256 OF Character;
                        buffer-     : POINTER TO ARRAY OF CHAR;
                        tmp-        : POINTER TO ARRAY OF CHAR;
                        info-       : POINTER TO ARRAY OF DrawInfo;
                        pos-        : POINTER TO ARRAY OF LONGINT;
                        chars,                  (* nuber of characters n buffer *)
                        endPos      : LONGINT;  (* position of last char in buffer *)

                        (* reference to block marks *)
                        markA*,
                        markB*      : R.Mark;

                        (* colors *)
                        foreground-,
                        background-,
                        cursorColor-,
                        markColor-,
                        blockColor-,
                        blockTextColor- : LONGINT;

                        (* other data *)
                        tabSize-    : LONGINT;

                        (* other flags *)
                        showEOL-,
                        showSpace-,
                        showTabs-   : BOOLEAN;
                      END;

  ParserInfo*       = POINTER TO ParserInfoDesc;
  ParserInfoDesc*   = RECORD
                        next      : ParserInfo;
                        priority- : LONGINT;
                      END;

VAR
  id       : LONGINT;
  infoList : ParserInfo;


  (**
    Initializes the parser with some default values.
  **)

  PROCEDURE (s : SyntaxParser) Init*;

  VAR
    x : INTEGER;

  BEGIN
    s.foreground:=D.blackColor;
    s.background:=D.whiteColor;
(*    s.foreground:=D.whiteColor;
    s.background:=D.blackColor;*)
    s.cursorColor:=D.warnColor;
    s.markColor:=D.backgroundColor;
    s.blockColor:=D.fillColor;
    s.blockTextColor:=D.textColor;

    FOR x:=0 TO LEN(s.table)-1 DO
      s.table[x].newChar:=0X;
      s.table[x].aPen:=s.foreground;
      s.table[x].bPen:=s.background;
    END;

    s.table[ORD(0X)].newChar:=" ";
    FOR x:=1 TO 31 DO
      IF x#ORD(Ascii.ht) THEN
        s.table[x].newChar:="";
      END;
    END;

    s.showEOL:=FALSE;
    s.showSpace:=FALSE;
    s.showTabs:=FALSE;
(*    s.showEOL:=TRUE;*)
(*    s.showSpace:=TRUE;*)

    IF s.showEOL THEN
      s.table[ORD(Ascii.lf)].newChar:="";
      s.table[ORD(Ascii.lf)].aPen:=D.shadowColor;
      s.table[ORD(Ascii.lf)].bPen:=s.background;
    ELSE
      s.table[ORD(Ascii.lf)].newChar:=" ";
      s.table[ORD(Ascii.lf)].aPen:=D.shadowColor;
      s.table[ORD(Ascii.lf)].bPen:=s.background;
    END;

    IF s.showSpace THEN
      s.table[ORD(" ")].newChar:="";
      s.table[ORD(" ")].aPen:=D.shadowColor;
      s.table[ORD(" ")].bPen:=s.background;
    END;

    IF s.showTabs THEN
      s.table[ORD(Ascii.ht)].newChar:=Ascii.ht;
      s.table[ORD(Ascii.ht)].aPen:=D.warnColor;
      s.table[ORD(Ascii.ht)].bPen:=s.background;
    END;

    s.tabSize:=4;
  END Init;

  PROCEDURE (s : SyntaxParser) SetArea(from,to : LONGINT;
                                       aPen,bPen : LONGINT; style : SET;
                                       handled : BOOLEAN);

  VAR
    x,a : LONGINT;

  BEGIN
    IF from>s.pos[s.chars-1] THEN
      RETURN;
    END;

    IF from<1 THEN
      from:=1;
    END;

    IF to>s.pos[s.chars-1] THEN
      to:=s.pos[s.chars-1];
    END;

    x:=0;
    WHILE s.pos[x]<from DO
      INC(x);
    END;
    a:=x;

    s.info[a].aPen:=aPen;
    s.info[a].bPen:=bPen;
    s.info[a].style:=style;
    s.info[a].start:=FALSE;
    IF handled THEN
      s.info[a].handled:=TRUE;
    END;

    WHILE (x<s.chars) & (s.pos[x]<=to) DO
      s.info[x]:=s.info[a];
      INC(x);
    END;

    IF (a=0)
    OR (s.info[a-1].aPen#s.info[a].aPen)
    OR (s.info[a-1].bPen#s.info[a].bPen)
    OR (s.info[a-1].style#s.info[a].style) THEN
      s.info[a].start:=TRUE;
    END;

    IF x<s.chars THEN

      IF (s.info[x].aPen#s.info[a].aPen)
      OR (s.info[x].bPen#s.info[a].bPen)
      OR (s.info[x].style#s.info[a].style) THEN
        s.info[x].start:=TRUE;
      END;
    END;
  END SetArea;

  PROCEDURE (s : SyntaxParser) FillLineBuffer(line : R.LineRun;chars : LONGINT);

  VAR
    run    : R.Run;
    x,y,z,
    pos,
    spaces : LONGINT;

  BEGIN
    s.chars:=chars;

    IF (s.buffer=NIL) OR (LEN(s.buffer^)<=chars) THEN
      NEW(s.buffer,chars+1);
      NEW(s.tmp,chars+1);
      NEW(s.info,chars);
      NEW(s.pos,chars);
    END;

    run:=line.next;
    x:=1;
    y:=0;
    WHILE (run#NIL) & ~(run IS R.LineRun) & (y<chars) DO
      WITH run : R.TextRun DO
        z:=1;
        pos:=run.pos;
        WHILE (z<=run.length) & (y<chars) DO
          IF (run.block.text[pos]=Ascii.ht) & ~s.showTabs THEN
            spaces:=(((y+1) DIV s.tabSize)+1)*s.tabSize-y-1;
            WHILE (spaces>0) & (y<chars) DO
              s.buffer[y]:=" ";
              s.pos[y]:=x;
              INC(y);
              DEC(spaces);
            END;
          ELSE
            s.buffer[y]:=run.block.text[pos];
            s.pos[y]:=x;
            INC(y);
          END;
          INC(z);
          INC(pos);
          INC(x);
        END;
      ELSE
      END;
      run:=run.next;
    END;

    IF y<chars THEN
      s.buffer[y]:=Ascii.lf;
      s.pos[y]:=x;
      INC(x);
      INC(y);
    END;

    WHILE y<chars DO
      s.buffer[y]:=0X;
      s.pos[y]:=x;
      INC(x);
      INC(y);
    END;
  END FillLineBuffer;

  PROCEDURE (s : SyntaxParser) HandleMarks(line : R.LineRun);

  VAR
    entry    : R.LineEntry;

  BEGIN

    (* drawing gerneral marks *)
    entry:=line.first;
    WHILE entry#NIL DO
      WITH entry : R.Mark DO
        IF entry.type=R.cursor THEN
          (* skipping *)
        ELSIF entry.type=R.block THEN
          s.SetArea(entry.x,entry.x,s.blockTextColor,s.blockColor,{},TRUE);
        ELSE
          s.SetArea(entry.x,entry.x,s.foreground,s.markColor,{},TRUE);
        END;
      ELSE
      END;
      entry:=entry.next;
    END;

    (* drawing cursors *)
    entry:=line.first;
    WHILE entry#NIL DO
      WITH entry : R.Mark DO
        IF entry.type=R.cursor THEN
          s.SetArea(entry.x,entry.x,s.foreground,s.cursorColor,{},TRUE);
        END;
      ELSE
      END;
      entry:=entry.next;
    END;
  END HandleMarks;

  PROCEDURE (s : SyntaxParser) HandleBlock(line : R.LineRun; y : LONGINT);

  BEGIN
    IF (s.markA#NIL) & (s.markB#NIL) & (y>=s.markA.y) & (y<=s.markB.y) THEN
      IF (s.markA.y=s.markB.y) THEN
        s.SetArea(s.markA.x,s.markB.x,s.blockTextColor,s.blockColor,{},TRUE);
      ELSIF (y=s.markA.y) THEN
        s.SetArea(s.markA.x,MAX(LONGINT),s.blockTextColor,s.blockColor,{},TRUE);
      ELSIF (y=s.markB.y) THEN
        s.SetArea(1,s.markB.x,s.blockTextColor,s.blockColor,{},TRUE);
      ELSE
        s.SetArea(1,MAX(LONGINT),s.blockTextColor,s.blockColor,{},TRUE);
      END;
    END;
  END HandleBlock;

  PROCEDURE (s : SyntaxParser) HandleChars;

  VAR
    x,h : LONGINT;

  BEGIN
    (* converting characters *)
    FOR x:=0 TO s.chars-1 DO
      h:=ORD(s.buffer[x]);
      IF s.table[h].newChar#0X THEN
        s.buffer[x]:=s.table[h].newChar;
        s.SetArea(x+1,x+1,s.table[h].aPen,s.table[h].bPen,{},FALSE)
      END;
    END;
  END HandleChars;

  PROCEDURE (s : SyntaxParser) PrintBuffer;

  VAR
    x : LONGINT;

  BEGIN
    FOR x:=0 TO s.chars-1 DO
      IF s.info[x].start THEN
        Err.Char("|");
      END;
      Err.Char(s.buffer[x]);
    END;
    Err.Ln;
    FOR x:=0 TO s.chars-1 DO
      IF s.info[x].start THEN
        Err.Char(" ");
      END;
      Err.LongInt(s.pos[x] MOD 10,0);
    END;
    Err.Ln;
  END PrintBuffer;

  PROCEDURE (s : SyntaxParser) ParseLine*(y,chars : LONGINT; line : R.LineRun);

  VAR
    x      : LONGINT;

  BEGIN
    s.FillLineBuffer(line,chars);

    (* initializing with default colors *)
    s.info[0].aPen:=s.foreground;
    s.info[0].bPen:=s.background;
    s.info[0].style:={};
    s.info[0].start:=FALSE;
    s.info[0].handled:=FALSE;

    FOR x:=1 TO chars-1 DO
      s.info[x]:=s.info[0];
    END;

    s.HandleBlock(line,y);
    s.HandleChars();
    s.HandleMarks(line);
  END ParseLine;

  PROCEDURE (p : ParserInfo) Init*;

  BEGIN
    p.priority:=0;
  END Init;

  PROCEDURE (p : ParserInfo) GetParser*(bufferName,fileName : ARRAY OF CHAR;
                                        text : R.Run):SyntaxParser;

  BEGIN
    RETURN NIL;
  END GetParser;

  PROCEDURE AddParserInfo*(info : ParserInfo);

  VAR
    help : ParserInfo;

  BEGIN
    IF infoList=NIL THEN
      infoList:=info;
      info.next:=NIL;
    ELSIF info.priority<=infoList.priority THEN
      info.next:=infoList;
      infoList:=info;
    ELSE
      help:=infoList;
      WHILE (help.next#NIL) & (info.priority>help.next.priority) DO
        help:=help.next;
      END;
      info.next:=help.next;
      help.next:=info;
    END;
  END AddParserInfo;

  PROCEDURE GetParser*(bufferName,fileName : ARRAY OF CHAR; text : R.Run):SyntaxParser;

  VAR
    entry  : ParserInfo;
    parser : SyntaxParser;

  BEGIN
    entry:=infoList;
    WHILE entry#NIL DO
      parser:=entry.GetParser(bufferName,fileName,text);
      IF parser#NIL THEN
        RETURN parser;
      END;
      entry:=entry.next;
    END;

    NEW(parser);
    parser.Init;

    RETURN parser;
  END GetParser;

BEGIN
  id:=0;
  infoList:=NIL;
END VO:EditSyntaxParser.