/***************************************************************************
 $RCSfile: balance.h,v $
 -------------------
 cvs         : $Id: balance.h,v 1.6 2003/01/29 14:44:31 cstim Exp $
 begin       : Tue Jun 12 2002
 copyright   : (C) 2002 by Martin Preuss
 email       : openhbci@aquamaniac.de


 ***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU Lesser General Public            *
 *   License as published by the Free Software Foundation; either          *
 *   version 2.1 of the License, or (at your option) any later version.    *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   Lesser General Public License for more details.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the Free Software   *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,                 *
 *   MA  02111-1307  USA                                                   *
 *                                                                         *
 ***************************************************************************/


#ifndef HBCIBALANCE
#define HBCIBALANCE

/** @file balance.h
 *
 * @short HBCI::Balance, HBCI::AccountBalance, and their C wrappers
 * @ref HBCI_Balance, @ref HBCI_AccountBalance. */

#include <openhbci/date.h>
#include <openhbci/value.h>

#ifdef __cplusplus
#include <string>

namespace HBCI {
/**
 * @short Stores all data belonging to one specific balance. 
 *
 * This class stores the data that represent one specific balance. An
 * account can have multiple balances according to different balancing
 * rules.  */
class DLLIMPORT Balance {
private:
    bool _debit;
    Value _value;
    Date _date;
    Time _time;

public:
    /**  Empty Constructor. */
    Balance();
    /**  Default Destructor. */
    ~Balance();

    /** @short Dumps the data of this object to stderr.
     *
     * Useful for debuggin purposes. */
    void dump();

    /** @name Balance
     *
     * The actual balance. */
    /*@{*/
    /** @short Returns <code>true</code> if the value is a debit
     *(i.e. positive).
     *
     * If the value of this balance is a debit, then this method
     * returns <code>true</code>. Usually this means that your account
     * has a positive amount of money in it, i.e. the bank owes you
     * money. 
     *
     * However, for some account types (which are probably
     * not-so-common for private homebanking), in accountant-speak a
     * "debit" can also mean that you owe money to the bank. Please
     * take the written statements of the bank as a reference: If the
     * bank calls your balance a "debit", then this method returns
     * true -- so eventually the interpretation is up to you. */
    bool isDebit() const { return _debit;};
    /**  Set whether this value is a debit (i.e. positive).
     *
     * @param c <code>true</true> if this value is a debit. */
    void setDebit(bool c) { _debit=c;};

    /** @short Returns the value of this balance. */
    const Value &value() const { return _value;};
    /**  Set the value of this balance.
     *
     * @param v The value that this balance should be set to. */
    void setValue(const Value &v) { _value=v;};
    /*@}*/

    /** @name Date and Time
     *
     * The point in thime when this balance was created. */
    /*@{*/
    /**  Returns the Date of this balance. 
     *
     * Returns the Date when this balance was valid/computed/fixed. */
    const Date &date() const { return _date;};
    /**  Set the Date of this balance. 
     *
     * @param d The Date at which this balance is valid. */
    void setDate(const Date &d) { _date=d;};

    /**  Returns the Time of day of this balance. */
    const Time &time() const { return _time;};
    /**  Set the Time of this balance.
     * 
     * @param t The Time of this balance. */
    void setTime(const Time &t) { _time=t;};
    /*@}*/
};

} /* namespace HBCI */
typedef struct HBCI::Balance HBCI_Balance;
extern "C" {
#else /* __cplusplus */
    typedef struct HBCI_Balance HBCI_Balance;
#endif /* __cplusplus */

    /** @name HBCI_Balance functions */
    /*@{*/
    /** @short Returns <code>true</code> if the value is a debit
     *(i.e. positive).
     *
     * If the value of this balance is a debit, then this method
     * returns <code>true</code>. Usually this means that your account
     * has a positive amount of money in it, i.e. the bank owes you
     * money. 
     *
     * However, for some account types (which are probably
     * not-so-common for private homebanking), in accountant-speak a
     * "debit" can also mean that you owe money to the bank. Please
     * take the written statements of the bank as a reference: If the
     * bank calls your balance a "debit", then this method returns
     * true -- so eventually the interpretation is up to you. */
    extern int HBCI_Balance_isDebit(const HBCI_Balance *b);
    /** @short Returns the value of this balance. */
    extern const HBCI_Value *HBCI_Balance_value(const HBCI_Balance *b);
    /**  Returns the Date of this balance. 
     *
     * Returns the Date when this balance was valid/computed/fixed. */
    extern const HBCI_Date *HBCI_Balance_date(const HBCI_Balance *b);
    /**  Returns the Time of day of this balance. */
    extern const HBCI_Time *HBCI_Balance_time(const HBCI_Balance *b);
    /*@}*/
#ifdef __cplusplus
}
#endif /* __cplusplus */




#ifdef __cplusplus
namespace HBCI {
/** 
 * @short Stores all values/balances of an Account as sent to us
 * by the Bank.
 *
 * This class stores the balances of your account. An Account usually
 * has several different balances at the same time (e.g. booked/noted
 * balance). You can query it from the institute by sending the
 * OutboxJobGetBalance (using the lowlevel job JOBGetBalance).  */
class DLLIMPORT AccountBalance {
    friend class SEGBalance;
private:
    string _currency;
    Balance _bookedBalance;
    Balance _notedBalance;
    Value _bankLine;
    Value _disposable;
    Value _disposed;
    Date _date;
    Time _time;
public:
    /**  Empty Constructor. */
    AccountBalance();
    /**  Default Destructor. */
    ~AccountBalance();
    /** @short Dumps the data of this object to stderr.
     *
     * Useful for debuggin purposes. */
    void dump();

    /** @name Balance
     *
     * The actual balance. */
    /*@{*/
    /**  Set the booked balance. 
     *
     * @param b The booked balance. */
    void setBookedBalance(const Balance &b) { _bookedBalance=b;};
    /** @short Returns the booked Balance.
     * 
     * The booked balance is the balance including all transactions
     * that are booked, i.e. that are "really" executed. */
    const Balance &bookedBalance() const { return _bookedBalance;};

    /**  Set the noted balance.
     *
     * @param b The noted balance. */
    void setNotedBalance(const Balance &b) { _notedBalance=b;};
    /** @short Returns the noted Balance (which is probably more
     * up-to-date than the booked balance).
     *
     * The noted balance includes all transactions from the booked,
     * plus those transactions which the bank has noted for execution
     * but are not yet "really" executed. */
    const Balance &notedBalance() const { return _notedBalance;};

    /**  Sets the ISO-4217 currency string of this account
     * balance.
     *
     * @param s The currency of this balance. */
    void setCurrency(const string &s) { _currency=s;};
    /** @short Returns the ISO-4217 currency string of this account
        balance. */
    const string &currency() const { return _currency;};
    /*@}*/

    /** @name Overdraft
     *
     * Information about the bank line / limit value / overdraft
     * facility / credit facility, i.e. the amount that this account
     * is allowed to be overdrawn. */
    /*@{*/
    /**  Set the limit value you have on this account. */
    void setBankLine(const Value &v) { _bankLine=v;};
    /** @short Returns the bank line (limit value) for this account.
     *
     * I.e. the money your balance is allowed to go negative. */
    const Value &bankLine() const { return _bankLine;};

    /**  Set the amount that is still at your disposal in this
     *   account. 
     * 
     * @param v The amount at your disposal. */
    void setDisposable(const Value &v) { _disposable=v;};
    /** @short Returns the amount that is still at your disposal in
     * this account.
     *
     * If the difference between your current balance (whether noted
     * or booked depends on the conditions of your bank) and the bank
     * line value results in a positive amount, then you have money
     * left at your disposal. (Which is [hopefully] the usual case.)
     * */
    const Value &disposable() const { return _disposable;};

    /**  Set the amount that is already disposed. 
     *
     * @param v New disposed value. */
    void setDisposed(const Value &v) { _disposed=v;};
    /** @short Returns the amount that is already disposed.
     *
     * FIXME: exakt meaning still unknown. hmm, we will se when first
     * data arrive, i don't know yet */
    const Value &disposed() const { return _disposed;};
    /*@}*/

    /** @name Date and Time
     *
     * The point in thime when this balance was created. */
    /*@{*/
    /**  Returns the Date when this balance was created. */
    const Date &date() const { return _date;};
    /**  Set the Date when this balance was created.
     *
     * @param d New date. */
    void setDate(const Date &d) { _date=d;};

    /**  Returns the Time when this balance was created. */
    const Time &time() const { return _time;};
    /**  Set the Time when this balance was created.
     *
     * @param t New Time. */
    void setTime(const Time &t) { _time=t;};
    /*@}*/
};
} /* namespace */
typedef struct HBCI::AccountBalance HBCI_AccountBalance;
extern "C" {
#else /* __cplusplus */
  typedef struct HBCI_AccountBalance HBCI_AccountBalance;
#endif /* __cplusplus */

    /** @name HBCI_AccountBalance functions */
    /*@{*/
    /** @short Returns the ISO-4217 currency string of this account
        balance. */
    extern const char* 
    HBCI_AccountBalance_currency(const HBCI_AccountBalance *b);
    /** @short Returns the booked Balance.
     * 
     * The booked balance is the balance including all transactions
     * that are booked, i.e. that are "really" executed. */
    extern const HBCI_Balance*
    HBCI_AccountBalance_bookedBalance(const HBCI_AccountBalance *b);
    /** @short Returns the noted Balance (which is probably more
     * up-to-date than the booked balance).
     *
     * The noted balance includes all transactions from the booked,
     * plus those transactions which the bank has noted for execution
     * but are not yet "really" executed. */
    extern const HBCI_Balance*
    HBCI_AccountBalance_notedBalance(const HBCI_AccountBalance *b);
    /** @short Returns the bank line (limit value) for this account.
     *
     * I.e. the money your balance is allowed to go negative. */
    extern const HBCI_Value*
    HBCI_AccountBalance_bankLine(const HBCI_AccountBalance *b);
    /** @short Returns the amount that is still at your disposal in
     * this account.
     *
     * If the difference between your current balance (whether noted
     * or booked depends on the conditions of your bank) and the bank
     * line value results in a positive amount, then you have money
     * left at your disposal. (Which is [hopefully] the usual case.)
     * */
    extern const HBCI_Value*
    HBCI_AccountBalance_disposable(const HBCI_AccountBalance *b);
    /** @short Returns the amount that is already disposed.
     *
     * FIXME: exakt meaning still unknown. hmm, we will se when first
     * data arrive, i don't know yet */
    extern const HBCI_Value*
    HBCI_AccountBalance_disposed(const HBCI_AccountBalance *b);
    /**  Returns the Date when this balance was created. */
    extern const HBCI_Date *
    HBCI_AccountBalance_date(const HBCI_AccountBalance *b);
    /**  Returns the Time when this balance was created. */
    extern const HBCI_Time *
    HBCI_AccountBalance_time(const HBCI_AccountBalance *b);
    /*@}*/
#ifdef __cplusplus
}
#endif /* __cplusplus */



#endif

