/***************************************************************************
 $RCSfile: loader.cpp,v $
                             -------------------
    cvs         : $Id: loader.cpp,v 1.19 2003/07/02 23:07:05 aquamaniac Exp $
    begin       : Sat Jun 08 2002
    copyright   : (C) 2001 by Martin Preuss
    email       : openhbci@aquamaniac.de

 ***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU Lesser General Public            *
 *   License as published by the Free Software Foundation; either          *
 *   version 2.1 of the License, or (at your option) any later version.    *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   Lesser General Public License for more details.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the Free Software   *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,                 *
 *   MA  02111-1307  USA                                                   *
 *                                                                         *
 ***************************************************************************/


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "loader.h"
#include <hbcistring.h>
#include <accountimpl.h>
#include <bankimpl.h>
#include <connection.h>


namespace HBCI {

Loader::Loader(API *hbci):_hbci(hbci) {
}


Loader::~Loader() {
}


Error Loader::saveTransaction(const Transaction &t,
                                      SimpleConfig &cfg,
                                      cfgPtr w){
    list<string> sl;
    list<string>::const_iterator it;

    cfg.setIntVariable("tid",t.id(),w);
    cfg.setVariable("institute",t.ourBankCode(),w);
    cfg.setVariable("id",t.ourAccountId(),w);
    cfg.setVariable("otherinstitute",t.otherBankCode(),w);
    cfg.setVariable("otherid",t.otherAccountId(),w);
    sl=t.otherName();
    for (it=sl.begin(); it!=sl.end(); it++)
        cfg.setVariable("othername",*it,w);
    cfg.setVariable("primanota",t.primanota(),w);
    cfg.setVariable("key",t.transactionKey(),w);
    cfg.setIntVariable("code",t.transactionCode(),w);
    sl=t.description();
    for (it=sl.begin(); it!=sl.end(); it++)
        cfg.setVariable("description",*it,w);
    cfg.setVariable("text",t.transactionText(),w);
    cfg.setVariable("customerreference",t.customerReference(),w);
    cfg.setVariable("bankreference",t.bankReference(),w);
    cfg.setVariable("date",t.date().toString(),w);
    cfg.setVariable("valutadate",t.valutaDate().toString(),w);
    cfg.setVariable("value",t.value().toString(),w);
    if (t.originalValue().getValue()!=0)
      cfg.setVariable("originalvalue",t.originalValue().toString(),w);
    if (t.charge().getValue()!=0)
      cfg.setVariable("charge",t.charge().toString(),w);

    return Error();
}


Error Loader::saveUpdJob(const updJob &j,
                                 SimpleConfig &cfg,
                                 cfgPtr group2){
    cfgPtr group3;
    string t;

    group3=cfg.createGroup("job",group2);
    if (!group3.isValid())
        return Error("Loader::saveUpdJob()",
                         ERROR_LEVEL_INTERNAL,
                         0,
                         ERROR_ADVISE_ABORT,
                         "Could not create group",
                         "job");
    cfg.setVariable("segmentcode",
                    j.segmentCode(),
                    group3);
    cfg.setIntVariable("minsigcount",
                       j.minSigCount(),
                       group3);
    t=j.limitType();
    if (t[0]==0)
        t[0]=32;
    cfg.setVariable("limittype",
                    t,
                    group3);
    cfg.setVariable("limitvalue",
                    j.limitValue().toString(),
                    group3);
    cfg.setIntVariable("limitdays",
                       j.limitDays(),
                       group3);
    // done
    return Error();
}


Error Loader::saveAccountParams(Pointer<Account> a,
                                     SimpleConfig &cfg,
                                     cfgPtr group1){
    Error err;
    list<updJob> jobs;
    list<updJob>::const_iterator it;
    string t;

    // save account params
    cfg.setVariable("accountnumber",
                    a.ref().accountId(),
                    group1);
    cfg.setVariable("accountsuffix",
                    a.ref().accountSuffix(),
                    group1);
    cfg.setIntVariable("countrycode",
                       a.ref().bank().ref().countryCode(),
                       group1);
	cfg.setVariable("institute",
					a.cast<AccountImpl>().ref().instituteCode(),
                    group1);					
//     cfg.setVariable("institute",
//                     a.ref().bank().ref().bankCode(),
//                     group1);
    cfg.setVariable("accountname",
                    a.ref().accountName(),
                    group1);
    /*cfg.setVariable("userid",
                    a.ref().userId(),
                    group1);*/
    cfg.setVariable("name1",
                    a.ref().name1(),
                    group1);
    cfg.setVariable("name2",
                    a.ref().name2(),
                    group1);
    cfg.setVariable("currency",
                    a.ref().currency(),
                    group1);
    t = Limit::typeToChar(a.ref().limit().type());
    if (t[0]==0)
        t=" ";
    cfg.setVariable("limittype",
                    t,
                    group1);
    cfg.setVariable("limitvalue",
                    a.ref().limit().value().toString(),
                    group1);
    cfg.setIntVariable("limitdays",
                       a.ref().limit().daysForTimeLimit(),
                       group1);

    // save updjobs
    jobs=(dynamic_cast<AccountImpl&>(a.ref())).allowedJobs();
    for (it=jobs.begin();
         it!=jobs.end();
         it++) {
        err=saveUpdJob(*it,cfg,group1);
        if (!err.isOk())
            return err;
    } // for
    // done
    return Error();
}


Error Loader::saveBalance(const Balance &b,
                                  SimpleConfig &cfg,
                                  cfgPtr where){

    cfg.setBoolVariable("isdebit", b.isDebit(), where);
    cfg.setVariable("value", b.value().toString(), where);
    cfg.setVariable("date",b.date().toString(), where);
    cfg.setVariable("time",b.time().toString(), where);
    return Error();
}


Error Loader::saveAccountBalance(const AccountBalance &b,
                                         SimpleConfig &cfg,
                                         cfgPtr where){
    Error err;
    cfgPtr group1;

    // save booked balance
    group1=cfg.createGroup("booked-balance",where);
    if (!group1.isValid())
        return Error("Loader::saveAccountBalance()",
                         ERROR_LEVEL_INTERNAL,
                         0,
                         ERROR_ADVISE_ABORT,
                         "Could not create group",
                         "booked-balance");
    err=saveBalance(b.bookedBalance(),cfg,group1);
    if (!err.isOk())
        return err;

    // save noted balance
    group1=cfg.createGroup("noted-balance",where);
    if (!group1.isValid())
        return Error("Loader::saveAccountBalance()",
                         ERROR_LEVEL_INTERNAL,
                         0,
                         ERROR_ADVISE_ABORT,
                         "Could not create group",
                         "noted-balance");
    err=saveBalance(b.notedBalance(),cfg,group1);
    if (!err.isOk())
        return err;

    cfg.setVariable("currency",b.currency(),where);
    cfg.setVariable("bankline",b.bankLine().toString(),where);
    cfg.setVariable("disposable",b.disposable().toString(), where);
    cfg.setVariable("disposed",b.disposed().toString(), where);
    cfg.setVariable("date",b.date().toString(),where);
    cfg.setVariable("time",b.time().toString(),where);

    return Error();
}


Error Loader::saveStandingOrder(const StandingOrder &o,
                                        SimpleConfig &cfg,
                                        cfgPtr w) {
	list<string> description = o.description();
	list<string> otherNames = o.otherName();
	list<string>::iterator it;

    cfg.setVariable("jobid", o.jobIdentification(), w);
    cfg.setVariable("id", o.ourAccountId(), w);
    cfg.setVariable("institute", o.ourBankCode(), w);
    cfg.setVariable("otherid", o.otherAccountId(), w);
    cfg.setVariable("othersuffix", o.otherSuffix(), w);
    cfg.setVariable("otherbank", o.otherBankCode(), w);
    cfg.setIntVariable("othercountry", o.otherCountryCode(), w);
    cfg.setVariable("value", o.value().toString(), w);
    cfg.setIntVariable("code", o.transactionCode(), w);
    cfg.setVariable("firstdate", o.firstExecutionDate().toString(), w);
    cfg.setVariable("lastdate", o.lastExecutionDate().toString(), w);
    cfg.setVariable("nextdate", o.executionDate().toString(), w);
    cfg.setIntVariable("cycle", o.cycle(), w);
    cfg.setIntVariable("period", o.period(), w);
    cfg.setIntVariable("execday", o.execDay(), w);

    for (it=description.begin(); it!=description.end(); it++)
        cfg.setVariable("description", (*it), w);

	if (otherNames.size() > 0)	  
	  cfg.setVariable("othername", (* otherNames.begin()), w);
	if (otherNames.size() > 1)
	  cfg.setVariable("othername", (* ++otherNames.begin()), w);

    return Error();
}



Error Loader::saveAccount(Pointer<Account> a,
                                  SimpleConfig &cfg,
                                  cfgPtr where,
                                  unsigned int flags) {
    Error err;
    cfgPtr group1;
    list<Pointer<Customer> > users;
    list<Pointer<Customer> >::const_iterator it;
    list<Transaction> xa;
    list<Transaction>::const_iterator xit;
    list<StandingOrder>::const_iterator sit;

    // save account params
    group1=cfg.createGroup("params",where);
    if (!group1.isValid())
        return Error("Loader::saveAccount()",
                         ERROR_LEVEL_INTERNAL,
                         0,
                         ERROR_ADVISE_ABORT,
                         "Could not create group",
                         "params");
    err=saveAccountParams(a,cfg,group1);
    if (!err.isOk())
        return err;

    group1=cfg.createGroup("balance",where);
    if (!group1.isValid())
        return Error("Loader::saveAccount()",
                         ERROR_LEVEL_INTERNAL,
                         0,
                         ERROR_ADVISE_ABORT,
                         "Could not create group",
                         "balance");
    err=saveAccountBalance(a.ref().balance(),cfg,group1);
    if (!err.isOk())
        return err;

    cfg.setBoolVariable("managed",a.ref().managed(),where);

    // save authorized customer
    users=a.ref().authorizedCustomers();
    for (it=users.begin();
         it!=users.end();
         it++)
        cfg.setVariable("customers", (*it).ref().custId(), where);

    // save the transactions
    if (flags & HBCILOADER_WITH_TRANSACTIONS) {
        xa=a.ref().transactions();
        for (xit=xa.begin(); xit!=xa.end(); xit++) {
            group1=cfg.createGroup("transaction",where);
            if (!group1.isValid())
                return Error("Loader::saveAccount()",
                                 ERROR_LEVEL_INTERNAL,
                                 0,
                                 ERROR_ADVISE_ABORT,
                                 "Could not create group",
                                 "transaction");
            err=saveTransaction(*xit,cfg,group1);
            if (!err.isOk())
                return err;
        } // for
    } // if withTransactions

    // save standing orders
    for (sit=a.ref().standingOrders().begin();
         sit!=a.ref().standingOrders().end();
         sit++) {
        group1=cfg.createGroup("standing_order",where);
        if (!group1.isValid())
            return Error("Loader::saveAccount()",
                             ERROR_LEVEL_INTERNAL,
                             0,
                             ERROR_ADVISE_ABORT,
                             "Could not create group",
                             "message");
        err=saveStandingOrder(*sit,cfg,group1);
        if (!err.isOk())
            return err;
    } // for


    return Error();
}


Error Loader::saveCustomer(Pointer<Customer> c,
                                   SimpleConfig &cfg,
                                   cfgPtr where){
    cfg.setVariable("id",c.ref().custId(),where);
    cfg.setVariable("custName",c.ref().custName(),where);
    return Error();
}


Error Loader::saveUser(Pointer<User> c,
		       SimpleConfig &cfg,
		       cfgPtr where){
  list<Pointer<Customer> > custs;
  list<Pointer<Customer> >::iterator cit;
  cfgPtr group1;
  Error err;
  int nextGroupNr;

  //fprintf(stderr,"Saving user %s\n",c.ref().userId().c_str());

  cfg.setVariable("id",c.ref().userId(),where);
  cfg.setVariable("userName",c.ref().userName(),where);
  cfg.setIntVariable("version",c.ref().version(),where);
  cfg.setBoolVariable("knowsUPDJobs", c.ref().knowsSupportedJobs(), where);

  // save medium
  if (c.ref().medium().isValid()) {
    group1=cfg.createGroup("medium",
			   where);
    if (!group1.isValid())
      return Error("Loader::saveUser()",
		   ERROR_LEVEL_INTERNAL,
		   0,
		   ERROR_ADVISE_ABORT,
		   "Could not create group",
		   "medium");
    err=_hbci->mediumToConfig(c.ref().medium(),cfg,group1);
    if (!err.isOk())
      return err;

  } // if medium

  // save customers
  custs=c.ref().customers();
  nextGroupNr=0;
  for (cit=custs.begin(); cit!=custs.end(); cit++) {
    group1=cfg.createGroup("customer/"+
			   String::num2string(nextGroupNr++),
			   where);
    if (!group1.isValid())
      return Error("Loader::saveUser()",
		   ERROR_LEVEL_INTERNAL,
		   0,
		   ERROR_ADVISE_ABORT,
		   "Could not create group",
		   "customer"+(*cit).ref().custId());
    err=saveCustomer(*cit,cfg,group1);
    if (!err.isOk())
      return err;
  } // for

  return Error();
}


Error Loader::saveBankJob(const bpdJob &b,
                                  SimpleConfig &cfg,
                                  cfgPtr w){
    cfg.setVariable("segmentcode",b.segmentCode(),w);
    cfg.setIntVariable("segmentversion",b.segmentVersion(),w);
    cfg.setIntVariable("jobspermsg",b.jobsPerMessage(),w);
    cfg.setIntVariable("minsigcount",b.minSigCount(),w);
    cfg.setVariable("parameter",b.parameter(),w);

    return Error();
}


Error Loader::saveBankParams(Pointer<Bank> b,
                                     SimpleConfig &cfg,
                                     cfgPtr w){
    list<string> sl;
    //list<string>::const_iterator it;
    list<int> il;
    list<int>::const_iterator it2;
    list<bpdJob> jobs;
    list<bpdJob>::const_iterator it3;
    cfgPtr group1;
    Error err;

    BankImpl &ba = dynamic_cast<BankImpl&> (b.ref());

    cfg.setIntVariable("version", ba.version(),w);
    cfg.setIntVariable("country", ba.countryCode(),w);
    cfg.setVariable("code",ba.bankCode(),w);
    cfg.setVariable("name",ba.name(),w);
    il=ba.languages();
    for (it2=il.begin(); it2!=il.end(); it2++)
        cfg.setIntVariable("languages",*it2,w);
    il=ba.supportedVersions();
    for (it2=il.begin(); it2!=il.end(); it2++)
        cfg.setIntVariable("hbciversions",*it2,w);
    cfg.setIntVariable("maxmsgsize",ba.maxMessageSize(),w);
    cfg.setIntVariable("maxdifferentactions",
                       ba.maxDifferentActions(),w);
    cfg.setIntVariable("language",ba.language(),w);
    cfg.setIntVariable("type",ba.type(),w);
    cfg.setVariable("addr",ba.addr(),w);
    cfg.setVariable("suffix",ba.suffix(),w);
    cfg.setVariable("filter",ba.filter(),w);
    cfg.setIntVariable("filterversion",ba.filterVersion(),w);

    jobs=ba.supportedJobs();
    for (it3=jobs.begin(); it3!=jobs.end(); it3++){
        group1=cfg.createGroup("job",w);
        if (!group1.isValid())
            return Error("Loader::saveBankParams()",
                             ERROR_LEVEL_INTERNAL,
                             0,
                             ERROR_ADVISE_ABORT,
                             "Could not create group",
                             "job");
        err=saveBankJob(*it3,cfg,group1);
        if (!err.isOk())
            return err;
    } // for

    return Error();
}


Error Loader::saveInstituteMessage(const instituteMessage &msg,
                                           SimpleConfig &cfg,
                                           cfgPtr w) {
    cfgPtr group1;
    Error err;

    cfg.setVariable("date",msg.date().toString(),w);
    cfg.setVariable("time",msg.time().toString(),w);
    cfg.setVariable("subject",msg.subject(),w);
    cfg.setVariable("text",msg.text(),w);
    cfg.setBoolVariable("read",msg.read(),w);

    return err;
}


Error Loader::saveBank(Pointer<Bank> b,
                               SimpleConfig &cfg,
                               cfgPtr w,
                               unsigned int flags) {
    cfgPtr group1;
    Error err;
    list<Pointer<User> > custs;
    list<Pointer<User> >::const_iterator cit;
    list<Pointer<Account> > accs;
    list<Pointer<Account> >::const_iterator ait;
    list<instituteMessage>::const_iterator mit;
    int nextGroupNr;

    BankImpl &ba = dynamic_cast<BankImpl&> (b.ref());

    cfg.setIntVariable("hbciversion",ba.hbciVersion(),w);

    // save bank params
    group1=cfg.createGroup("params",w);
    if (!group1.isValid())
        return Error("Loader::saveBank()",
                         ERROR_LEVEL_INTERNAL,
                         0,
                         ERROR_ADVISE_ABORT,
                         "Could not create group",
                         "params");
    err=saveBankParams(b,cfg,group1);
    if (!err.isOk())
        return err;

    // save customers
    custs=ba.users();
    nextGroupNr=0;
    for (cit=custs.begin(); cit!=custs.end(); cit++) {
        group1=cfg.createGroup("user/"
                               +String::num2string(nextGroupNr++),
                               w);
        if (!group1.isValid())
            return Error("Loader::saveBank()",
                             ERROR_LEVEL_INTERNAL,
                             0,
                             ERROR_ADVISE_ABORT,
                             "Could not create group",
                             "customer");
        err=saveUser(*cit,cfg,group1);
        if (!err.isOk())
            return err;
    } // for

    // save accounts
    accs=ba.accounts();
    nextGroupNr=0;
    for (ait=accs.begin(); ait!=accs.end(); ait++) {
        group1=cfg.createGroup("account/"
                               +String::num2string(nextGroupNr++),
                               w);
        if (!group1.isValid())
            return Error("Loader::saveBank()",
                             ERROR_LEVEL_INTERNAL,
                             0,
                             ERROR_ADVISE_ABORT,
                             "Could not create group",
                             "account");
        err=saveAccount(*ait,cfg,group1,flags);
        if (!err.isOk())
            return err;
    } // for

    // save institute messages
    if (flags & HBCILOADER_WITH_MESSAGES) {
        for (mit=b.ref().messages().begin();
             mit!=b.ref().messages().end();
             mit++) {
            group1=cfg.createGroup("message",w);
            if (!group1.isValid())
                return Error("Loader::saveBank()",
                                 ERROR_LEVEL_INTERNAL,
                             0,
                                 ERROR_ADVISE_ABORT,
                                 "Could not create group",
                                 "message");
            err=saveInstituteMessage(*mit,cfg,group1);
            if (!err.isOk())
                return err;
        } // for
    }
    // done
    return Error();
}


Error Loader::saveAll(SimpleConfig &cfg,
                              cfgPtr w,
                              unsigned int flags) {
    list<Pointer<Bank> > banks;
    list<Pointer<Bank> >::const_iterator bit;
    cfgPtr group1;
    Error err;
    int nextGroupNr;

    cfg.setIntVariable("lasttransactionid",API::lastTransactionId(),w);
    cfg.setIntVariable("nextjobid",Outbox::nextId(),w);
    cfg.setIntVariable("sockettimeout",(int)Connection::timeout(),w);

    banks=_hbci->banks();
    nextGroupNr=0;
    for (bit=banks.begin(); bit!=banks.end(); bit++) {
        if ((*bit).ref().bankCode().empty())
            return Error("Loader::saveBank()",
                             ERROR_LEVEL_INTERNAL,
                             0,
                             ERROR_ADVISE_ABORT,
                             "bank has no id !");
        group1=cfg.createGroup("bank/"+
                               String::num2string(nextGroupNr++),
                               w);
        if (!group1.isValid())
            return Error("Loader::saveAll()",
                             ERROR_LEVEL_INTERNAL,
                             0,
                             ERROR_ADVISE_ABORT,
                             "Could not create group",
                             "bank/"+String::num2string(nextGroupNr-1));
        err=saveBank(*bit,cfg,group1);
        if (!err.isOk())
            return err;
    } // for

    return Error();
}


Error Loader::loadTransaction(Transaction &t,
			      SimpleConfig &cfg,
			      cfgPtr w){
  cfgPtr val;
  unsigned int tid;

  tid=cfg.getIntVariable("tid",0,w);
  if (tid==0) {
    // no id assigned, so do it now
    tid=API::nextTransactionId();
  }
  t.setId(tid);
  t.setOurBankCode(cfg.getVariable("institute","",w));
  t.setOurAccountId(cfg.getVariable("id","",w));

  t.setOtherBankCode(cfg.getVariable("otherinstitute","",w));
  t.setOtherAccountId(cfg.getVariable("otherid","",w));
  val=cfg.findVariable("othername",w);
  if (val.isValid()) {
    // add othername
    val.child();
    while(val.isValid()) {
      t.addOtherName((*val).data);
      val++;
    } // while
  }
  t.setPrimanota(cfg.getVariable("primanota","",w));
  t.setTransactionKey(cfg.getVariable("key","",w));
  t.setTransactionCode(cfg.getIntVariable("code",51,w));
  val=cfg.findVariable("description",w);
  if (val.isValid()) {
    // add description
    val.child();
    while(val.isValid()) {
      t.addDescription((*val).data);
      val++;
    } // while
  }
  t.setTransactionText(cfg.getVariable("text","",w));
  t.setCustomerReference(cfg.getVariable("customerreference","",w));
  t.setBankReference(cfg.getVariable("bankreference","",w));
  t.setDate(Date(cfg.getVariable("date","",w)));
  t.setValutaDate(Date(cfg.getVariable("valutadate","",w)));
  t.setValue(Value(cfg.getVariable("value","0,:EUR",w)));
  t.setOriginalValue(Value(cfg.getVariable("originalvalue","0,:EUR",w)));
  if (t.originalValue().getCurrency().empty())
    t.setOriginalValue(Value(t.originalValue().getValue(),"EUR"));
  t.setCharge(Value(cfg.getVariable("charge","0,:EUR",w)));
  if (t.charge().getCurrency().empty())
    t.setCharge(Value(t.charge().getValue(),"EUR"));
  return Error();
}



Error Loader::loadBalance(Balance &b,
                                  SimpleConfig &cfg,
                                  cfgPtr w){

    b.setDebit(cfg.getBoolVariable("isdebit",true,w));
    b.setValue(Value(cfg.getVariable("value","",w)));
    b.setDate(Date(cfg.getVariable("date","",w)));
    b.setTime(Time(cfg.getVariable("time","",w)));

    return Error();
}


Error Loader::loadAccountBalance(AccountBalance &ab,
                                         SimpleConfig &cfg,
                                         cfgPtr w){
    Error err;
    cfgPtr group;

    // load noted balance
    group=cfg.findGroup("noted-balance",w,false);
    if (group.isValid()) {
        Balance b;
        err=loadBalance(b, cfg,group);
        if (!err.isOk())
            return err;
        ab.setNotedBalance(b);
    } // if group exists

    // load noted balance
    group=cfg.findGroup("booked-balance",w,false);
    if (group.isValid()) {
        Balance b;
        err=loadBalance(b, cfg,group);
        if (!err.isOk())
            return err;
        ab.setBookedBalance(b);
    } // if group exists

    ab.setBankLine(Value(cfg.getVariable("bankline","0,:EUR",w)));
    ab.setDisposable(Value(cfg.getVariable("disposable","0,:EUR",w)));
    ab.setDisposed(Value(cfg.getVariable("disposed","0,:EUR",w)));
    ab.setDate(Date(cfg.getVariable("date","",w)));
    ab.setTime(Time(cfg.getVariable("time","",w)));

    return Error();
}


Error Loader::loadUpdJob(Pointer<Account> a,
                                 SimpleConfig &cfg,
                                 cfgPtr w){
    updJob j;
    string t;
    AccountImpl &aimpl = dynamic_cast<AccountImpl&>(a.ref());

    j.setSegmentCode(cfg.getVariable("segmentcode","",w));
    j.setMinSigCount(cfg.getIntVariable("minsigcount",0,w));
    t=cfg.getVariable("limittype"," ",w);
    if (!t.empty()) {
        if (t[0]==0 || t[0]==3)
            j.setLimitType(0);
        else
            j.setLimitType(t[0]);
    }
    j.setLimitValue(Value(cfg.getVariable("limitvalue","0,:EUR",w)));
    j.setLimitDays(cfg.getIntVariable("limitdays",0,w));
    aimpl.addJob(j);

    return Error();
}


Error Loader::loadAccountParams(Pointer<Account> a,
                                        SimpleConfig &cfg,
                                        cfgPtr w){
    string t;
    Error err;
    cfgPtr group1;
    AccountImpl &aimpl = dynamic_cast<AccountImpl&>(a.ref());

    aimpl.setAccountId(cfg.getVariable("accountnumber","",w));
    aimpl.setAccountSuffix(cfg.getVariable("accountsuffix","",w));
    aimpl.setCountryCode(cfg.getIntVariable("countrycode",280,w));
    aimpl.setInstituteCode(cfg.getVariable("institute","",w));
    aimpl.setAccountName(cfg.getVariable("accountname","",w));
    aimpl.setUserId(cfg.getVariable("userid","",w));
    aimpl.setName1(cfg.getVariable("name1","",w));
    aimpl.setName2(cfg.getVariable("name2","",w));
    aimpl.setCurrency(cfg.getVariable("currency","EUR",w));
    t=cfg.getVariable("limittype"," ",w);
    if (!t.empty()) {
	    if (t[0]==0 || t[0]==3)
            aimpl.setLimitType(0);
        else
            aimpl.setLimitType(t[0]);
    }
    aimpl.setLimitValue(Value(cfg.getVariable("limitvalue","0,:EUR",w)));
    aimpl.setLimitDays(cfg.getIntVariable("limitdays",0,w));

    // load jobs
    group1=w;
    group1.child();
    while(group1.isValid()) {
        if (-1!=parser::cmpPattern((*group1).data,"job",false) &&
            (*group1).type==CONFIG_TYPE_GROUP){
            err=loadUpdJob(a, cfg,group1);
            if (!err.isOk())
                return err;
        } // if correct group
        group1++;
    } // while

    return Error();
}


Error Loader::loadStandingOrder(StandingOrder &o,
                                        SimpleConfig &cfg,
                                        cfgPtr w) {

    cfgPtr val;

	// add all names
    val = cfg.findVariable("othername", w);
    if (val.isValid()) {
        // add othername
        val.child();
        while(val.isValid()) {
            o.addOtherName((*val).data);
            val++;
        } // while
    }

	// add all description lines
    val = cfg.findVariable("description", w);
    if (val.isValid()) {
        // add description
        val.child();
        while(val.isValid()) {
            o.addDescription((*val).data);
            val++;
        } // while
    }

    o.setJobIdentification(cfg.getVariable("jobid", "", w));
	o.setOurAccountId(cfg.getVariable("id", "", w));
	o.setOurBankCode(cfg.getVariable("institute", "", w));
	o.setOurCountryCode(cfg.getIntVariable("country", 280, w));
    o.setOtherAccountId(cfg.getVariable("otherid", "", w));
    o.setOtherSuffix(cfg.getVariable("othersuffix", "", w));
    o.setOtherBankCode(cfg.getVariable("otherbank", "", w));
    o.setOtherCountryCode(cfg.getIntVariable("othercountry", 280, w));
    o.setValue(Value(cfg.getVariable("value", "0,:EUR", w)));
    o.setTransactionCode(cfg.getIntVariable("code", 52, w));
    o.setFirstExecutionDate(Date(cfg.getVariable("firstdate", "", w)));
    o.setLastExecutionDate(Date(cfg.getVariable("lastdate", "", w)));
    o.setExecutionDate(Date(cfg.getVariable("nextdate", "", w)));
    o.setCycle(cfg.getIntVariable("cycle", 1, w));
    o.setPeriod(cfg.getIntVariable("period",
                                   StandingOrder::EXECUTE_MONTHLY, w));
    o.setExecDay(cfg.getIntVariable("execday", 1, w));

    return Error();
}



Error Loader::loadAccount(Pointer<Bank> bank,
                                  SimpleConfig &cfg,
                                  cfgPtr w,
                                  unsigned int flags) {
    Pointer<Account> a;
    Pointer<Customer> cust;
    Error err;
    cfgPtr group, group1;
    cfgPtr val;
    Pointer<BankImpl> bimpl = bank.cast<BankImpl>();

    // Create new AccountImpl object
    AccountImpl *newAccount = new AccountImpl(bimpl);
    // To save typing, use the reference in the following
    AccountImpl &aimpl = *newAccount;
    // Save the upcast pointer to it. Also, the Pointer will take
    // care of deletion.
    a = newAccount;

    // load params
    group=cfg.findGroup("params",w,false);
    if (group.isValid()) {
        err=loadAccountParams(a, cfg,group);
        if (!err.isOk())
            return err;
    } // if group exists

    // load balance
    group=cfg.findGroup("balance",w,false);
    if (group.isValid()) {
        AccountBalance bal;
        err=loadAccountBalance(bal, cfg,group);
        if (!err.isOk())
            return err;
        aimpl.setBalance(bal);
    } // if group exists

    // load authorized customers
    val=cfg.findVariable("customers",w);
    if (!val.isValid())
        // fallback
        val=cfg.findVariable("users",w);
    if (val.isValid()) {
        // add customer
        val.child();
        while(val.isValid()) {
            // find customer
            cust=bank.ref().findCustomer((*val).data);
            // add customer
            if (cust.isValid())
                aimpl.addAuthorizedCustomer(cust);
            else {
//#if DEBUGMODE>0
                fprintf(stderr,"Unknown customer \"%s\"\n",
                        (*val).data.c_str());
//#endif
            }
            val++;
        } // while
    }

    aimpl.setManaged(cfg.getBoolVariable("managed",true,w));

    // load transactions
    if (flags & HBCILOADER_WITH_TRANSACTIONS) {
        group1=w;
        group1.child();
        while(group1.isValid()) {
            if (-1!=parser::cmpPattern((*group1).data,"transaction",false)) {
                Transaction t;
                err=loadTransaction(t,cfg,group1);
                if (!err.isOk())
                    return err;
                aimpl.addTransaction(t);
            } // if transaction group
            group1++;
        } // while
    } // if withTransactions

    // load standing orders
    group=w;
    group.child();
    while(group.isValid()) {
        if (-1!=parser::cmpPattern((*group).data,"standing_order",false) &&
            (*group).type==CONFIG_TYPE_GROUP) {
            StandingOrder sto;

            // load standing order
            err=loadStandingOrder(sto,cfg,group);
            if (!err.isOk())
                return err;
            // add standing order to this account
            aimpl.addStandingOrder(sto);
        } // if standing order
        group++;
    } // while

    // add account to the bank
    bank.ref().addAccount(a);

    return Error();
}


Error Loader::loadCustomer(Pointer<User> u,
                                   Pointer<Customer> &cu,
                                   SimpleConfig &cfg,
                                   cfgPtr w){
    string id;
    string custName;

    id=cfg.getVariable("id","",w);
    custName=cfg.getVariable("custName","",w);
    if (id.empty())
        return Error("Loader::loadCustomer()",
                         ERROR_LEVEL_INTERNAL,
                         0,
                         ERROR_ADVISE_ABORT,
                         "customer id is empty");

    cu=_hbci->customerFactory(u, id, custName);
    return Error();
}


Error Loader::loadUser(Pointer<Bank> b,
                       SimpleConfig &cfg,
                       cfgPtr w){
  Pointer<User> u;
  Pointer<Customer> cu;
  Pointer<Medium> m;
  cfgPtr group, group1;
  Error err;
  string mname;

  string id;
  string userName;
  int version;
  bool knowsUPDJobs;
  string t;

  // sample all data we need
  id=cfg.getVariable("id","",w);
  if (id.empty())
    return Error("Loader::loadUser()",
		 ERROR_LEVEL_INTERNAL,
		 0,
		 ERROR_ADVISE_ABORT,
		 "customer with empty id");

  version=cfg.getIntVariable("version",0,w);
  userName = cfg.getVariable("userName","",w);
  knowsUPDJobs = cfg.getBoolVariable("knowsUPDJobs", false, w);

  // check for "[medium]" group
  group=cfg.findGroup("medium",w,false);
  if (!group.isValid()) {
    // group does not exist, so simply use the user's group
    group=w;
  }

  // look for a medium of that name
  mname=cfg.getVariable("mediumname","",group);
  if (!mname.empty())
    m=_hbci->findMedium(mname);

  // check if the medium already exists
  if (!m.isValid()) {
    // it does not, so create it
    try {
      m=_hbci->mediumFromConfig(cfg, group);
    }
    catch (Error xerr) {
      return Error("Loader::loadUser", xerr);
    }
  }

  // create user
  u=_hbci->userFactory(b,m,id,version, userName, knowsUPDJobs);
  m.ref().setOwner(u);

  // add user to the bank
  b.ref().addUser(u);

  // load all customers
  group=cfg.findGroup("customer",w,false);
  if (!group.isValid())
    group=cfg.findGroup("user",w,false);
  if (group.isValid()) {
    group1=group;
    group1.child();
    while(group1.isValid()) {
      err=loadCustomer(u,cu,cfg,group1);
      if (!err.isOk())
	return err;
      // add customer to list
      u.ref().addCustomer(cu);
      group1++;
    } // while there are customers
  } // if group exists

  // check whether there is at least one customer
  if (u.ref().customers().size()==0) {
    // there is none, so create this customer
    cu =_hbci->customerFactory(u, u.ref().userId(), "");
    u.ref().addCustomer(cu);
  }

  return Error();
}


Error Loader::loadBankJob(Pointer<Bank> b,
                          SimpleConfig &cfg,
                          cfgPtr w){
    bpdJob j;
    BankImpl &ba = dynamic_cast<BankImpl&> (b.ref());

    j.setSegmentCode(cfg.getVariable("segmentcode","",w));
    j.setSegmentVersion(cfg.getIntVariable("segmentversion",0,w));
    j.setJobsPerMessage(cfg.getIntVariable("jobspermsg",0,w));
    j.setMinSigCount(cfg.getIntVariable("minsigcount",0,w));
    j.setParameter(cfg.getVariable("parameter","",w));
    ba.addJob(j);

    return Error();
}


Error Loader::loadBankParams(Pointer<Bank> b,
                                     SimpleConfig &cfg,
                                     cfgPtr w){
    cfgPtr val;
    cfgPtr group1;
    Error err;

    BankImpl &ba = dynamic_cast<BankImpl&> (b.ref());

    ba.setVersion(cfg.getIntVariable("version",0,w));
    ba.setCountryCode(cfg.getIntVariable("country",280,w));
    ba.setBankCode(cfg.getVariable("code","",w));
    ba.setName(cfg.getVariable("name","[unnamed]",w));
    val=cfg.findVariable("languages",w);
    if (val.isValid()) {
        // add languages
        val.child();
        while(val.isValid()) {
            ba.addLanguage(atoi((*val).data.c_str()));
            val++;
        } // while
    }
    val=cfg.findVariable("hbciversions",w);
    if (val.isValid()) {
        // add versions
        val.child();
        while(val.isValid()) {
            ba.addVersion(atoi((*val).data.c_str()));
            val++;
        } // while
    }
    ba.setMaxMessageSize(cfg.getIntVariable("maxmsgsize",0,w));
    ba.setMaxDifferentActions(cfg.getIntVariable("maxdifferentactions",0,w));
    ba.setLanguage(cfg.getIntVariable("language",1,w));
    ba.setType(cfg.getIntVariable("type",2,w));
    ba.setAddr(cfg.getVariable("addr","",w));
    ba.setSuffix(cfg.getVariable("suffix","",w));
    ba.setFilter(cfg.getVariable("filter","",w));
    ba.setFilterVersion(cfg.getIntVariable("filterversion",0,w));

    // load all jobs
    group1=w;
    group1.child();
    while(group1.isValid()) {
        if (parser::cmpPattern((*group1).data,"job",false) &&
            (*group1).type==CONFIG_TYPE_GROUP){
            err=loadBankJob(b, cfg,group1);
            if (!err.isOk())
                return err;
        } // if correct group
        group1++;
    } // while

    return Error();
}


Error Loader::loadInstituteMessage(instituteMessage &msg,
                                           SimpleConfig &cfg,
                                           cfgPtr w) {
    msg.setDate(Date(cfg.getVariable("date","",w)));
    msg.setTime(Time(cfg.getVariable("time","",w)));
    msg.setSubject(cfg.getVariable("subject","",w));
    msg.setText(cfg.getVariable("text","",w));
    msg.setRead(cfg.getBoolVariable("read",false,w));

    return Error();
}


Error Loader::loadBank(SimpleConfig &cfg,
                               cfgPtr w,
                               unsigned int flags) {
    cfgPtr group;
    cfgPtr group1;
    Error err;
    Pointer<Bank> bank;
    Pointer<BankImpl> bimpl;

    // Create new BankImpl object
    bimpl = new BankImpl(_hbci);
    // type safely cast it to a pointer to the base class
    bank = bimpl.cast<Bank>();

    bimpl.ref().setHbciVersion(cfg.getIntVariable("hbciversion",0,w));

    // load all params
    group=cfg.findGroup("params",w,false);
    if (group.isValid()) {
        err=loadBankParams(bank, cfg,group);
        if (!err.isOk())
            return err;
    } // if group exists

    // load all users
    group=cfg.findGroup("user",w,false);
    if (!group.isValid())
        group=cfg.findGroup("customer",w,false);
    if (!group.isValid())
        // fall back for older version of OpenHBCI
        group=cfg.findGroup("customer",w,false);
    if (group.isValid()) {
        group1=group;
        group1.child();
        while(group1.isValid()) {
            err=loadUser(bank, cfg,group1);
            if (!err.isOk())
                return err;
            // add user to list
            group1++;
        } // while there are users
    } // if group exists

    // load all accounts
    group=cfg.findGroup("account",w,false);
    if (group.isValid()) {
        group1=group;
        group1.child();
        while(group1.isValid()) {
            err=loadAccount(bank, cfg,group1,flags);
            if (!err.isOk())
                return err;
            // add user to list
            group1++;
        } // while there are users
    } // if group exists

    // load all institute messages
    if (flags & HBCILOADER_WITH_MESSAGES) {
        group=w;
        group.child();
        while(group.isValid()) {
            if (-1!=parser::cmpPattern((*group).data,"message",false) &&
                (*group).type==CONFIG_TYPE_GROUP) {
                instituteMessage msg;
                err=loadInstituteMessage(msg,cfg,group);
                if (!err.isOk())
                    return err;
                msg.setCountry(bimpl.ref().countryCode());
                msg.setBankCode(bimpl.ref().bankCode());
                bimpl.ref().addInstituteMessage(msg);
            } // if message group
            group++;
        } // while
    }

    // add bank
    _hbci->addBank(bank);

    return Error();

}


Error Loader::loadAll(SimpleConfig &cfg,
                              cfgPtr w,
                              unsigned int flags) {
    cfgPtr group;
    cfgPtr group1;
    Error err;


    API::setLastTransactionId(cfg.getIntVariable("lasttransactionid",1,w));
    Outbox::setNextId(cfg.getIntVariable("nextjobid",1,w));
    Connection::setTimeOut(cfg.getIntVariable("sockettimeout",30,w));

    // load all banks
    group=cfg.findGroup("bank",w,false);
    if (group.isValid()) {
        group1=group;
        group1.child();
        while(group1.isValid()) {
            err=loadBank(cfg,group1,flags);
            if (!err.isOk())
                return err;
            // add user to list
            group1++;
        } // while there are users
    } // if group exists

    return Error();
}



Error 
Loader::writeTransactionFile(const string &name,
			     const list<Transaction> &xactions){
    SimpleConfig cfg;
    cfgPtr group;
    Error err;
    list<Transaction>::const_iterator it;

    // create config
    cfg.setMode(HBCIAPP_CONFIG_MODE);

    for (it=xactions.begin();
         it!=xactions.end();
         it++) {
        group=cfg.createGroup("transaction",cfg.root());
        if (!group.isValid())
            return Error("Loader::writeTransactionFile()",
                             ERROR_LEVEL_INTERNAL,
                             0,
                             ERROR_ADVISE_ABORT,
                             "Could not create group",
                             "transaction");
        err=saveTransaction(*it,cfg,group);
        if (!err.isOk())
            return err;
    }

    // then write file
    if (name.empty())
      err=cfg.writeFile(File::StdTypeStdOut);
    else
      err=cfg.writeFile(name);

    // done
    return err;
}


Error Loader::readTransactionFile(const string &name,
				  list<Transaction> &xactions){
    SimpleConfig cfg;
    cfgPtr group;
    Error err;

    // read config file
    cfg.setMode(HBCIAPP_CONFIG_MODE);

    if (name.empty())
      err=cfg.readFile(File::StdTypeStdIn);
    else
      err=cfg.readFile(name);
    if (!err.isOk())
      return err;

    // load transactions from config
    group=cfg.root();
    group.child();
    while(group.isValid()) {
        if (-1!=parser::cmpPattern((*group).data,"transaction",false)) {
            Transaction t;
            err=loadTransaction(t,cfg,group);
            if (!err.isOk())
                return err;
            xactions.push_back(t);
        } // if transaction group
        group++;
    } // while

    // done
    return err;
}


Error 
Loader::writeStandingOrderFile(const string &name,
			       const list<StandingOrder> &sto){
    SimpleConfig cfg;
    cfgPtr group;
    Error err;
    list<StandingOrder>::const_iterator it;

    // create config
    cfg.setMode(HBCIAPP_CONFIG_MODE);

    for (it=sto.begin();
         it!=sto.end();
         it++) {
        group=cfg.createGroup("standing_order",cfg.root());
        if (!group.isValid())
            return Error("Loader::writStandingOrdereFile()",
                             ERROR_LEVEL_INTERNAL,
                             0,
                             ERROR_ADVISE_ABORT,
                             "Could not create group",
                             "standing_order");
        err=saveStandingOrder(*it,cfg,group);
        if (!err.isOk())
            return err;
    }

    // then write file
    if (name.empty())
      err=cfg.writeFile(File::StdTypeStdOut);
    else
      err=cfg.writeFile(name);

    // done
    return err;
}


Error Loader::readStandingOrderFile(const string &name,
				    list<StandingOrder> &sto){
  SimpleConfig cfg;
  cfgPtr group;
  Error err;

  // read config file
  cfg.setMode(HBCIAPP_CONFIG_MODE);

  if (name.empty())
    err=cfg.readFile(File::StdTypeStdIn);
  else
    err=cfg.readFile(name);
  if (!err.isOk())
    return err;
  if (!err.isOk())
    return err;

  // load transactions from config
  group=cfg.root();
  group.child();
  while(group.isValid()) {
    if (-1!=parser::cmpPattern((*group).data,"standing_order",false)) {
      StandingOrder s;
      err=loadStandingOrder(s,cfg,group);
      if (!err.isOk())
	return err;
      sto.push_back(s);
    } // if transaction group
    group++;
  } // while

  // done
  return err;
}



} // namespace HBCI
