#include "osl/record/kisen.h"
#include "osl/record/csaIOError.h"
#include "osl/apply_move/applyMove.h"
#include "osl/pieceStand.h"
#include "osl/misc/iconvConvert.h"
#include <boost/filesystem/convenience.hpp>
#include <boost/foreach.hpp>
#include <iostream>

namespace osl
{
namespace record
{
  Position KisenUtils::convertPosition( int pos ){
    assert(1<=pos && pos<=0x51);
    int y=((pos-1)/9)+1, x=((pos-1)%9)+1;
    return Position(x,y);
  }
  int KisenUtils::convertPosition(Position pos)
  {
    return ((pos.y() - 1) * 9 + 1) + pos.x() - 1;
  }

  Move KisenUtils::convertMove(SimpleState const& state,int c0,int c1){
    Move move;

    if(1<=c1 && c1<=0x51){
      Position from=convertPosition(c1),to;
      Piece fromPiece=state.getPieceOnBoard(from);
      if (! fromPiece.isPiece()) 
	throw CsaIOError("Position error");
      assert(fromPiece.isPiece());
      assert(fromPiece.owner()==state.getTurn() ||
	     (std::cerr << c1 << "," << from << "," << fromPiece << std::endl,0)
	     );
      bool isPromote=false;
      if(1<=c0 && c0<=0x51){
	to=convertPosition(c0);
      }
      else if(0x65<=c0 && c0<=0xb5){
	to=convertPosition(c0-0x64);
	isPromote=true;
      }
      else{
	throw CsaIOError("c0 range error");
      }
      Piece toPiece=state.getPieceAt(to);
      if (! toPiece.isEmpty() && toPiece.owner()!=alt(state.getTurn()))
	throw CsaIOError("inconsintent move (to)");
      Ptype ptype=fromPiece.ptype();
      if(isPromote)ptype=promote(ptype);
      const Ptype captured = toPiece.ptype();
      if (captured == KING)
	return Move::INVALID();
      move=Move(from,to,ptype,captured,isPromote,state.getTurn());
    }
    else{
      assert(0x65<=c1);
      assert(1<=c0&&c0<=0x51);
      Position to=convertPosition(c0);
      Ptype ptype=PTYPE_EMPTY;
      int piece_on_stand = c1;
      const Ptype ptypes[]={ROOK,BISHOP,GOLD,SILVER,KNIGHT,LANCE,PAWN};
      for(size_t i=0;i<sizeof(ptypes)/sizeof(Ptype);i++){
	int count=state.countPiecesOnStand(state.getTurn(),ptypes[i]);
	if(count>0){
	  if(piece_on_stand>0x64){
	    piece_on_stand-=count;
	    if(piece_on_stand<=0x64) ptype=ptypes[i];
	  }
	}
      }
      assert(ptype!=PTYPE_EMPTY ||
	     (std::cerr << state << to << " " << c1
	      << " " << piece_on_stand << std::endl, false));
      move=Move(to,ptype,state.getTurn());
    }
    if (! state.isValidMove(move,true)) {
      std::cerr << "warning: bad move in kisen\n" << state << move << "\n";
      return Move();
    }
    assert(state.isValidMove(move,true) ||
	   (std::cerr << state << move << std::endl, false));
    return move;
  }


  KisenFile::KisenFile(const std::string& fileName) 
    :ifs(fileName.c_str()),initialState(HIRATE), fileName(fileName) 
  {
    if (! ifs)
      throw CsaIOError("KisenFile not found");
    ifs.seekg(0,std::ios::end);
    assert((ifs.tellg() % 512)==0);
    numberOfGames=ifs.tellg()/512;
  }

  const vector<Move> KisenFile::getMoves(size_t index)
  {
    assert(index<size());
    vector<Move> moves;
    //    std::cerr << "Game[" << index << "]" << std::endl;
    ifs.seekg(index*512,std::ios::beg);
    CArray<unsigned char, 512> cbuf;
    ifs.read(reinterpret_cast<char *>(&cbuf[0]),512);
    SimpleState state(HIRATE);
    //
    Player turn=BLACK;
    for(size_t turnCount=0; 
	(turnCount*2 < cbuf.size())
	  && cbuf[turnCount*2]!=0 && cbuf[turnCount*2+1]!=0;
	turnCount++, turn=alt(turn)){
      if(turnCount==KisenFile::maxMoves || cbuf[  turnCount *2 ] == 0 || cbuf[ turnCount * 2 + 1 ] == 0 ){ break; }
      int c0=cbuf[turnCount*2], c1=cbuf[turnCount*2+1];
      if (moves.empty() && c0 == 0xff && c1 == 0xff) // komaochi
	break;
      const Move move=KisenUtils::convertMove(state,c0,c1);
      if (move.isInvalid())
	break;
      moves.push_back(move);
      ApplyMoveOfTurn::doMove(state, move);
      assert(state.isConsistent( true ) );
    }
    return moves;
  }
#ifndef MINIMAL
  const std::string KisenFile::ipxFileName(const std::string& filename) 
  {
    namespace bf = boost::filesystem;
    const bf::path ipxfilename = bf::change_extension(bf::path(filename), ".ipx");
    return ipxfilename.file_string();
  }

  KisenIpxFile::KisenIpxFile(const std::string& fileName) 
    :ifs(fileName.c_str()), file_name(fileName) 
  {
    if (! ifs)
      throw CsaIOError("KisenIpxFile not found");
    ifs.seekg(0,std::ios::end);
    assert((ifs.tellg() % 256)==0);
    numberOfGames=ifs.tellg()/256;
  }
  const std::string KisenIpxFile::getPlayer(size_t index,Player pl)
  {
    assert(index<size());
    vector<Move> moves;
    ifs.seekg(index*256,std::ios::beg);
    CArray<unsigned char, 256> cbuf;
    ifs.read(reinterpret_cast<char *>(&cbuf[0]),256);
    int startIndex=0;
    if(pl==WHITE)startIndex=14;
    CArray<char,15> buf;
    buf[14]='\0';
    strncpy(&buf[0],reinterpret_cast<char *>(&cbuf[startIndex]),14);
    return std::string(&buf[0]);
  }
  unsigned int KisenIpxFile::getRating(size_t index,Player pl)
  {
    assert(index<size());
    vector<Move> moves;
    ifs.seekg(index*256,std::ios::beg);
    CArray<unsigned char, 256> cbuf;
    ifs.read(reinterpret_cast<char *>(&cbuf[0]),256);
    int startIndex=0324;
    if(pl==WHITE)startIndex=0326;
    return cbuf[startIndex]+256*cbuf[startIndex+1];
  }
  unsigned int KisenIpxFile::getResult(size_t index)
  {
    assert(index<size());
    ifs.seekg(index*256,std::ios::beg);
    CArray<unsigned char, 256> cbuf;
    ifs.read(reinterpret_cast<char *>(&cbuf[0]),256);
    return cbuf[64+48+6];
  }
  const std::string KisenIpxFile::getTitle(size_t index,Player pl)
  {
    assert(index<size());
    vector<Move> moves;
    ifs.seekg(index*256,std::ios::beg);
    CArray<unsigned char, 256> cbuf;
    ifs.read(reinterpret_cast<char *>(&cbuf[0]),256);
    int startIndex=28;
    if(pl==WHITE)startIndex+=8;
    CArray<char,9> buf;
    buf[8]='\0';
    strncpy(&buf[0],reinterpret_cast<const char*>(&cbuf[startIndex]),8);
    return std::string(&buf[0]);
  }

  KisenPlusFile::KisenPlusFile(const std::string& fileName) 
    :ifs(fileName.c_str()),initialState(HIRATE) 
  {
    if (! ifs)
      throw CsaIOError("KisenPlusFile not found");
    ifs.seekg(0,std::ios::end);
    assert((ifs.tellg() % 2048)==0);
    numberOfGames=ifs.tellg()/2048;
  }

  const vector<Move> KisenPlusFile::getMoves(size_t index)
  {
    vector<Move> moves;
    vector<int> times;
    getMoves(index, moves, times);
    return moves;
  }

  void KisenPlusFile::getMoves(size_t index,
			       vector<Move>& moves, vector<int>& times)
  {
    assert(index<size());
    //    std::cerr << "Game[" << index << "]" << std::endl;
    ifs.seekg(index*2048,std::ios::beg);
    CArray<unsigned char, 2048> cbuf;
    ifs.read(reinterpret_cast<char *>(&cbuf[0]),2048);
    SimpleState state(HIRATE);
    for (size_t i = 0; 
	 i < 2048 && cbuf[i]!=0 && cbuf[i+1]!=0;
	 i += 8)
    {
      int c0 = cbuf[i];
      int c1 = cbuf[i + 1];
      bool is_promote = false;
      Move move;

      if (c0 > 100)
      {
	is_promote = true;
	c0 = 256 - c0;
      }

      Position to(c0 % 10, c0 / 10);

      if (c1 < 10)
      {
	// drop
	move = Move(to,
		    PieceStand::order[c1 - 1],
		    state.getTurn());
      }
      else
      {
	Position from(c1 % 10, c1 / 10);
	Ptype type = state.getPieceAt(from).ptype();
	if (is_promote)
	  type = promote(type);
	move = Move(from, to,
		    type, state.getPieceAt(to).ptype(),
		    is_promote, state.getTurn());
      }
      moves.push_back(move);
      times.push_back(cbuf[i + 7] * 60 + cbuf[i + 6]);
      ApplyMoveOfTurn::doMove(state, move);
      assert(state.isConsistent( true ) );
    }
  }
#endif
} // namespace record
} // namespace osl

osl::record::
KisenFile::~KisenFile()
{
}
#ifndef MINIMAL
osl::record::
KisenIpxFile::~KisenIpxFile()
{
}

void osl::record::
OKisenStream::save(SimpleState state, const vector<Move> &moves)
{
  if (!(state == SimpleState(HIRATE)))
  {
    std::cerr << "Can not save non-HIRATE record" << std::endl;
    return;
  }
  const int max_length = std::min(256, static_cast<int>(moves.size()));
  for (int i = 0; i < max_length; ++i)
  {
    const Move move = moves[i];
    if (!move.isDrop())
    {
      int from = KisenUtils::convertPosition(move.from());
      int to = KisenUtils::convertPosition(move.to());
      if (move.isPromote())
      {
	to += 100;
      }
      os << static_cast<char>(to) << static_cast<char>(from);
    }
    else
    {
      int to = KisenUtils::convertPosition(move.to());
      int count = 1;
      BOOST_FOREACH(Ptype ptype, PieceStand::order)
      {
	if (ptype == move.ptype())
	{
	  break;
	}
	count += state.countPiecesOnStand(move.player(), ptype);
      }
      count += 100;
      os << static_cast<char>(to) << static_cast<char>(count);
    }
    ApplyMoveOfTurn::doMove(state, moves[i]);
  }
  for (int i = max_length; i < 256; ++i)
  {
    os << '\0' << '\0';
  }
}

void osl::record::
OKisenStream::save(Record *record)
{
  vector<Move> moves;
  vector<int> time;
  record->getMoves(moves, time);
  SimpleState state = record->getInitialState();
  save(state, moves);
}

void osl::record::
KisenIpxWriter::writeString(const std::string &name, size_t length)
{
  for (size_t i = 0; i < length; ++i)
  {
    if (i < name.length())
    {
      os << name[i];
    }
    else
    {
      os << '\0';
    }
  }
}

void osl::record::
KisenIpxWriter::writeRating(int rating)
{
  int high = rating / 256;
  int low = rating % 256;
  os << static_cast<char>(low) << static_cast<char>(high);
}

void osl::record::
KisenIpxWriter::save(const Record &record,
		     int black_rating, int white_rating,
		     const std::string &black_title,
		     const std::string &white_title)
{
  // total 256 bytes
  // Player name: 14 bytes each
#ifndef _WIN32
  writeString(misc::IconvConvert::convert("EUC-JP", "SJIS", record.getPlayer(BLACK)), 14);
  writeString(misc::IconvConvert::convert("EUC-JP", "SJIS", record.getPlayer(WHITE)), 14);
#endif
  writeString(black_title, 8);
  writeString(white_title, 8);
  for (int i = 44; i < 118; ++i)
  {
    os << '\0';
  }
  vector<Move> moves;
  vector<int> time;
  record.getMoves(moves, time);
  // TODO: sennichite, jishogi
  if (moves.size() <= 256)
  {
    if (moves.size() % 2 == 0)
      os << static_cast<char>(KisenIpxFile::WHITE_WIN);
    else
      os << static_cast<char>(KisenIpxFile::BLACK_WIN);
  }
  else
  {
    if (moves.size() % 2 == 0)
      os << static_cast<char>(KisenIpxFile::WHITE_WIN_256);
    else
      os << static_cast<char>(KisenIpxFile::BLACK_WIN_256);
  }
  for (int i = 119; i < 214; ++i)
  {
    os << '\0';
  }
  writeRating(black_rating);
  writeRating(white_rating);
  for (int i = 218; i < 256; ++i)
  {
    os << '\0';
  }
}
#endif
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
