/* FixedCapacityVector.t.cc
 */
#include "osl/misc/fixedCapacityVector.h"
#include <cppunit/TestCase.h>
#include <cppunit/extensions/HelperMacros.h>
#include <set>

class FixedCapacityVectorTest : public CppUnit::TestFixture {
  CPPUNIT_TEST_SUITE(FixedCapacityVectorTest);
  CPPUNIT_TEST(testUniq);
  CPPUNIT_TEST(testErase);
  CPPUNIT_TEST(testPushBack);
  CPPUNIT_TEST(testDestruct);
  CPPUNIT_TEST(testCopy);
  CPPUNIT_TEST_SUITE_END();
public:
  void testUniq();
  void testErase();
  void testPushBack();
  void testDestruct();
  void testCopy();
};

CPPUNIT_TEST_SUITE_REGISTRATION(FixedCapacityVectorTest);
using namespace osl;

void FixedCapacityVectorTest::testPushBack()
{
  FixedCapacityVector<int,6> f123;
  f123.push_back(1); f123.push_back(2); f123.push_back(3);

  FixedCapacityVector<int,3> f456;
  f456.push_back(4); f456.push_back(5); f456.push_back(6);
  
  f123.push_back(f456.begin(), f456.end());
  CPPUNIT_ASSERT_EQUAL((size_t)6, f123.size());
  for (int i=0; i<6; ++i)
    CPPUNIT_ASSERT_EQUAL(f123[i], i+1);
}

void FixedCapacityVectorTest::testUniq()
{
  std::multiset<int> s;
  FixedCapacityVector<int,1000> v;
  for (size_t i=0; i<700; ++i)
  {
    const int e = random() % 128;
    s.insert(e);
    v.push_back(e);
  }
  CPPUNIT_ASSERT_EQUAL(s.size(), v.size());
  std::set<int> u(s.begin(), s.end());
  CPPUNIT_ASSERT(u.size() < v.size());
  v.unique();
  CPPUNIT_ASSERT_EQUAL(u.size(), v.size());
  std::sort(v.begin(), v.end());
  CPPUNIT_ASSERT(std::equal(u.begin(), u.end(), v.begin()));
}

void FixedCapacityVectorTest::testErase()
{
  FixedCapacityVector<int,4> v;
  v.push_back(1);
  v.push_back(2);
  v.push_back(3);
  v.push_back(1);
  CPPUNIT_ASSERT_EQUAL((size_t)4, v.size());
  v.erase(1);
  CPPUNIT_ASSERT_EQUAL((size_t)2, v.size());
  CPPUNIT_ASSERT_EQUAL(2, v[0]);
  CPPUNIT_ASSERT_EQUAL(3, v[1]);
  v.erase(3);
  CPPUNIT_ASSERT_EQUAL((size_t)1, v.size());
  v.erase(2);
  CPPUNIT_ASSERT_EQUAL((size_t)0, v.size());
}

struct Counting
{
  static int count;
  Counting() { ++count; }
  Counting(const Counting&) { ++count; }
  ~Counting() { --count; }
};
int Counting::count = 0;

void FixedCapacityVectorTest::testDestruct()
{
  CPPUNIT_ASSERT_EQUAL(0, Counting::count);
  {
    Counting object;
    CPPUNIT_ASSERT_EQUAL(1, Counting::count);
  }
  CPPUNIT_ASSERT_EQUAL(0, Counting::count);

  {
    int64_t storage;
    CPPUNIT_ASSERT_EQUAL(0, Counting::count);
    misc::construct(reinterpret_cast<Counting*>(&storage), Counting());
    CPPUNIT_ASSERT_EQUAL(1, Counting::count);
    misc::destroy(reinterpret_cast<Counting*>(&storage));;
    CPPUNIT_ASSERT_EQUAL(0, Counting::count);
  }
  CPPUNIT_ASSERT_EQUAL(0, Counting::count);
  
  BOOST_STATIC_ASSERT((boost::has_trivial_destructor<FixedCapacityVector<int,4>::iterator::value_type>::value));
  BOOST_STATIC_ASSERT((! boost::has_trivial_destructor<FixedCapacityVector<Counting,4>::iterator::value_type>::value));

  {
    FixedCapacityVector<Counting,4> v;
    Counting object;
    CPPUNIT_ASSERT_EQUAL(1, Counting::count);
    v.push_back(object);
    CPPUNIT_ASSERT_EQUAL(2, Counting::count);
    v.pop_back();
    CPPUNIT_ASSERT_EQUAL(1, Counting::count);
    v.push_back(object);
    v.push_back(object);
    CPPUNIT_ASSERT_EQUAL(3, Counting::count);
    v.clear();
    CPPUNIT_ASSERT_EQUAL(1, Counting::count);
  }
  CPPUNIT_ASSERT_EQUAL(0, Counting::count);
}

void FixedCapacityVectorTest::testCopy()
{
  {
    FixedCapacityVector<int,2> v, v2;
    v.push_back(100);
    v.push_back(200);
    CPPUNIT_ASSERT(! (v == v2));
    v2 = v;
    CPPUNIT_ASSERT(v == v2);
  }
  {
    FixedCapacityVector<int,2> v, v2;
    v.push_back(100);
    v.push_back(200);
    v2.push_back(1);
    v2.push_back(2);
    CPPUNIT_ASSERT(! (v == v2));
    v2 = v;
    CPPUNIT_ASSERT(v == v2);
  }
}

/* ------------------------------------------------------------------------- */
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
