
////////////////////////////////////////////////////////////////////////////////
// lite.cpp:
// A demonstration app for cllite.
// Author: stephan@s11n.net
// License: Public Domain
////////////////////////////////////////////////////////////////////////////////

#include <stdlib.h> // getenv()
#include <PACKAGE_NAMESPACE/cllite.h> // cllite framework
#include <PACKAGE_NAMESPACE/cl_debuggering_macros.h> // COUT/CERR

/**
   First we define a class hierarchy which we'd like to load.
   Notice that their interfaces do not reflect that they are
   classloadable.
*/
struct A {
        A(){
                // We echo the ctors so we can see when classes
                // are instantiated by class_loader.
                COUT << "A()"<<std::endl;
        }
        virtual ~A(){
                // So we can see that the proper dtors are being called.
                // This helps verify that we've created the proper types.
               COUT << "~A()"<<std::endl;
        }
        bool no_op;
};
struct B : public A {
        B(){
                COUT << "B()"<<std::endl;
        }
        virtual ~B(){
                COUT << "~B()"<<std::endl;
        }

};
struct C : public B {
        C(){
                COUT << "C()"<<std::endl;
        }
        virtual ~C(){
                COUT << "~C()"<<std::endl;
        }
};


////////////////////////////////////////////////////////////////////////
// Define USE_REGISTER to 0 to disable the use of CLASSLOADER_REGISTER
// and use manual factory registration instead. This is for testing
// both approaches, and they should function identically.
#define USE_REGISTER 0
#if USE_REGISTER
////////////////////////////////////////
// A's classloader:
PACKAGE_NAMESPACE_CLASSLOADER_REGISTER1(A);
PACKAGE_NAMESPACE_CLASSLOADER_REGISTER2(A,B);
PACKAGE_NAMESPACE_CLASSLOADER_REGISTER2(A,C);
PACKAGE_NAMESPACE_CLASSLOADER_ALIAS(A,"C","default_document_class",1);
////////////////////////////////////////
// B's classloader:
PACKAGE_NAMESPACE_CLASSLOADER_REGISTER1(B);
PACKAGE_NAMESPACE_CLASSLOADER_REGISTER2(B,C);
////////////////////////////////////////
// C's classloader:
PACKAGE_NAMESPACE_CLASSLOADER_REGISTER1(C);
PACKAGE_NAMESPACE_CLASSLOADER_ALIAS(C,"C","default_document_class",1);
PACKAGE_NAMESPACE_CLASSLOADER_ALIAS(C,"C","bogo::C",2);
#endif // USE_REGISTER
// if we USE_REGISTER, we're done with the classloader registration
// process!
////////////////////////////////////////////////////////////////////////


#include "LoadableClass.h" // sample BaseType for demonstating DLLs

int main( int argc, char ** argv )
{

        int dlevel = 0;
        for( int i = 0; i < argc; ++i )
        {
                if( std::string("-d") == argv[i] )
                {
                        dlevel = 1;
                }
        }
        PACKAGE_NAMESPACE::class_loader_debug_level( dlevel );
        if( ! dlevel )
        {
                CERR << "Use -d to enable extra debug output.\n";
        }

        ////////////////////////////////////////////////////////////
        // all of the free functions shown here are defined in
        // the cllite namespace:
        using namespace cllite;

        ////////////////////////////////////////////////////////////
        // Example of altering the class search path:
        // make sure PWD is in there:
        class_path().add_path( getenv("PWD") );


#if ! USE_REGISTER
        ////////////////////////////////////////////////////////////
        // Register default factories for all types with the base
        // classloaders. This is not necessary if you've used the
        // appropriate PACKAGE_NAMESPACE_CLASSLOADER_REGISTER
        // macro(s).
        //////////////////////////////////////////////////////////////
        // A's classloader:
        register_base<A>( "A" ); // equiv. to register_factory<A,A>("A")
        register_factory<A,B>( "B" );
        register_factory<A,C>( "C" );
        alias<A>( "default_document_class", "C" );
        //////////////////////////////////////////////////////////////
        // B's classloader:
        register_base<B>( "B" );
        register_factory<B,C>( "C" );
        // register_factory<B,A>( "A" ); // ERROR: no conversion for A* to B*
        //////////////////////////////////////////////////////////////
        // C's classloader:
        register_base<C>( "C" );
        // register_factory<C,A>( "A" ); // ERROR: no conversion from A* to C*
        // register_factory<C,B>( "B" ); // ERROR: no conversion from B* to C*
        alias<C>( "bogo::C", "C" );
        alias<C>( "default_document_class", "C" );
        ////////////////////////////////////////////////////////////
        // Note that the ERROR lines will fail at compile time.
        ////////////////////////////////////////////////////////////
#endif // USE_REGISTER


        ////////////////////////////////////////////////////////////
        // TESTLOAD(): helper macro to test classloading.
        // OBJ = the object to assign to.
        // Base == base TYPE of OBJ
        // ClassnameE = search key, as a quoted string
        // ShouldpasS = if true, aborts app when the test fails.
#define TESTLOAD(OBJ,Base,ClassnamE,ShouldpasS) \
	COUT << "BEGIN TEST:\n" << # OBJ << " = classload<"<<#Base<<">("<<ClassnamE<<");"<<std::endl; \
	OBJ = classload<Base>( ClassnamE ); \
 	std::cout << "   = " << std::hex<<OBJ<<std::endl; \
	if( ShouldpasS && (!OBJ) ) { COUT << "TEST FAILED! :(" << std::endl; exit(1); } \
	else COUT << "TEST PASSED :)" << std::endl; \
	if( OBJ ) delete( OBJ );


        ////////////////////////////////////////////////////////////////////////
        // test out classload<A>():
        A * a = 0;
        TESTLOAD(a,A,"A",true); // creates A*
        TESTLOAD(a,A,"B",true); // creates B*, disguised as A*
        TESTLOAD(a,A,"C",true); // creates C*, disguised as A*
        TESTLOAD(a,A,"X",false); // NULL: key "X" was never registered
        TESTLOAD(a,A,"default_document_class",true); // creates C*, due to alias
        ////////////////////////////////////////////////////////////////////////

        ////////////////////////////////////////////////////////////////////////
        // test out classload<B>():
        B * b = 0;
        TESTLOAD(b,B,"B",true); // creates B*
        TESTLOAD(b,B,"A",false); // NULL: key "A" was never registered
        TESTLOAD(b,B,"C",true); // creates a C*, disguised as a B*
        ////////////////////////////////////////////////////////////////////////

        ////////////////////////////////////////////////////////////////////////
        // test out classload<C>():
        C * c = 0;
        TESTLOAD(c,C,"C",true); // creates a C*
        TESTLOAD(c,C,"bogo::C",true); // creates a C*, due to an alias
        TESTLOAD(c,C,"default_document_class",false); // NULL. Not a registered
                                                      // alias/factory.
        ////////////////////////////////////////////////////////////////////////


        ////////////////////////////////////////////////////////////////////////
        // Now try some DLL loading. The base-most class (LoadableClass) is known
        // at compile-time, but LoadableSubClass is not: it is known here only
        // by a name, and the DLL loader loads it for us:
        LoadableClass * lc = 0;
        TESTLOAD(lc,LoadableClass,"LoadableClass",true);
        TESTLOAD(lc,LoadableClass,"LoadableSubClass",true); // a DLL, we hope
        alias<LoadableClass>( "AnAlias", "LoadableSubClass" ); // let's see what happens...
        TESTLOAD(lc,LoadableClass,"AnAlias",true); // creates a LoadableSubClass*
        TESTLOAD(lc,LoadableClass,"ANonAlias",false); // NULL. Unregistered classname/alias
        ////////////////////////////////////////////////////////////////////////


        COUT << "Exiting main()." << std::endl;
        return 0;
}

