/*
 * libSpiff - XSPF playlist handling library
 *
 * Copyright (C) 2007, Sebastian Pipping / Xiph.Org Foundation
 * All rights reserved.
 *
 * Redistribution  and use in source and binary forms, with or without
 * modification,  are permitted provided that the following conditions
 * are met:
 *
 *     * Redistributions   of  source  code  must  retain  the   above
 *       copyright  notice, this list of conditions and the  following
 *       disclaimer.
 *
 *     * Redistributions  in  binary  form must  reproduce  the  above
 *       copyright  notice, this list of conditions and the  following
 *       disclaimer   in  the  documentation  and/or  other  materials
 *       provided with the distribution.
 *
 *     * Neither  the name of the Xiph.Org Foundation nor the names of
 *       its  contributors may be used to endorse or promote  products
 *       derived  from  this software without specific  prior  written
 *       permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS  IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT  NOT
 * LIMITED  TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND  FITNESS
 * FOR  A  PARTICULAR  PURPOSE ARE DISCLAIMED. IN NO EVENT  SHALL  THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL,    SPECIAL,   EXEMPLARY,   OR   CONSEQUENTIAL   DAMAGES
 * (INCLUDING,  BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES;  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT  LIABILITY,  OR  TORT (INCLUDING  NEGLIGENCE  OR  OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * Sebastian Pipping, sping@xiph.org
 */

/**
 * @file SpiffPropsWriter.h
 * Interface of SpiffPropsWriter.
 */

#ifndef SPIFF_PROPS_WRITER_H
#define SPIFF_PROPS_WRITER_H



#include "SpiffDataWriter.h"
#include <list>
#include <utility>

namespace Spiff {



class SpiffXmlFormatter;
class SpiffProps;
class SpiffWriter;
class SpiffPropsWriterPrivate;



/**
 * Writes playlist properties to an XML formatter.
 */
class SpiffPropsWriter : public SpiffDataWriter {

private:
	/// @cond DOXYGEN_NON_API
	SpiffPropsWriterPrivate * const d; ///< D pointer
	/// @endcond

public:
	/**
	 * Creates a new playlist property writer.
	 *
	 * @param props		Playlist properties to write
	 */
	SpiffPropsWriter(SpiffProps * props);

	/**
	 * Copy constructor.
	 *
	 * @param source  Source to copy from
	 */
	SpiffPropsWriter(const SpiffPropsWriter & source);

	/**
	 * Assignment operator.
	 *
	 * @param source  Source to copy from
	 */
	SpiffPropsWriter & operator=(const SpiffPropsWriter & source);

	/**
	 * Destroys this playlist property writer.
	 */
	~SpiffPropsWriter();

protected:
	/**
	 * Initializes the playlist properties writer.
	 * Must be called before writing.
	 *
	 * @param output	Output formatter to write to
	 * @param version	XSPF version to produce
	 */
	void init(SpiffXmlFormatter & output, int version);

	/**
	 * Writes the opening playlist tag.
	 */
	void writeStartPlaylist();

	/**
	 * Writes the opening trackList tag.
	 */
	void writeStartTracklist(bool trackListEmpty);

	/**
	 * Writes the closing trackList tag.
	 */
	void writeEndTracklist();

	/**
	 * Writes the closing playlist tag.
	 */
	void writeEndPlaylist();

	/**
	 * Writes the list of attributions.
	 */
	void writeAttribution();

	/**
	 * Writes the date property.
	 */
	void writeDate();

	/**
	 * Writes the identifier property.
	 */
	void writeIdentifier();

	/**
	 * Writes the license property.
	 */
	void writeLicense();

	/**
	 * Writes the location property.
	 */
	void writeLocation();

	/**
	 * Writes the closing playlist tag.
	 */
	void writePlaylistClose();

	/**
	 * Writes the opening playlist tag.
	 */
	void writePlaylistOpen();

	/**
	 * Writes the closing trackList tag.
	 */
	void writeTrackListClose();

	/**
	 * Writes the opening trackList tag.
	 */
	void writeTrackListOpen();

private:
	/**
	 * Preregisters an XML namespace.
	 *
	 * @param uri				Namespace URI
	 * @param prefixSuggestion	Suggested prefix
	 */
	void registerNamespace(const XML_Char * uri, const XML_Char * prefixSuggestion);

	friend class SpiffWriter;

};



}

#endif // SPIFF_PROPS_WRITER_H
