/*  tiffep - TI File Format Engine & Proxy for TI calculators
 *  Copyright (C) 2000  Romain Lievin
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h> // fork, execl, getpid
#ifdef __WIN32__
#include <windows.h>
#include <process.h> // spwanl (= fork + execl), getpid
#endif
#include <glib.h>
#include <signal.h> // for managing some signals

#include "../src/tiffep.h"
#include "../src/tiffep_str.h"
#include "../src/tiffep_ver.h"

#define POLLING_TIME 500 //ms

int version(void);
int load_default_options();

// Global options
OPTIONS options;
TiffepShm s;

void signal_handler(int sig_no)
{
  DISPLAY2("Signal SIGINT (Ctrl+C) caught...\n");
  DISPLAY2("Ressources destroyed.\n");
  unlink(SOCKET_NAME);
  shm_detach(&s);
  shm_destroy(&s);
  
  exit(0);
}

/*
 * Main function
 */
int main(int argc, char *argv[], char **arge)
{
#ifdef __WIN32__
  HMODULE hModule;
  DWORD dWord;
  char sBuffer[4*MAXCHARS];
  char *dirname;
#endif

  volatile TiffepMsg *msg;
  gchar *plugin_path;
  pid_t pid;
  gchar *filename;
  int cmd = CMD_NONE;

  /* Install a signal handler */
  signal(SIGINT, signal_handler);

  /* Display version */
  version();

  /* Load default options */
#ifdef __WIN32__
  DISPLAY2("Process ID (pid): %d\n", _getpid());
#elif defined(__LINUX__)
  DISPLAY2("Process ID (pid): %d\n", getpid());
#endif
  DISPLAY2("-> Initializing... \n");
  DISPLAY2("   Load default options\n");
  load_default_options();
  
  /* 
     Init the path for the Windows version 
  */
#if defined(__WIN32__)
  // Retrieve the directory of the executable */
  hModule = GetModuleHandle("tilp.exe");
  dWord = GetModuleFileName(hModule, sBuffer, 4*MAXCHARS);
  dirname = g_dirname(sBuffer);
  DISPLAY2("Current path: <%s>\n", dirname);
  options.current_dir = g_strdup(dirname);
  g_free(dirname);
#else
  options.current_dir = g_strdup("");
#endif

  DISPLAY2("   Done.\n");
  DISPLAY2("-> Running... \n");

  /*
    Create ressources
  */
  if(shm_create(&s, SHM_NAME, TiffepMsgSize, 0))
  {
      exit(1);
  }

  if(shm_attach(&s, 0))
    {
      exit(1);
    }

  msg = (TiffepMsg *)(s.addr);	// a encapsuler !
  msg->cmd = CMD_NONE;

  /* Listen connections */
  while(1)
    {  
      /* Process messages */
      DISPLAY2("   Waiting messages\n");
      do
	{
#if defined(__LINUX__)
	  usleep(POLLING_TIME * 1000);
#elif defined(__WIN32__)
	  Sleep(POLLING_TIME);
#endif
	}
      while(!cmd_is_available(s));
      //DISPLAY2("   Message available. Processing...\n");
      cmd = msg->cmd;      // retrieve cmd

      switch(cmd)
	{
	case CMD_ECHO: // echo is acknowledged by server only
	  cmd_remove(s);
	  DISPLAY2("%3i: Receiving CMD_ECHO...\n", CMD_ECHO);
	  TRY(cmd_send_ok(s));
	  break;
	case CMD_EXEC_PLUGIN: // exec a plugin
	  cmd_remove(s);
	  DISPLAY2("%3i: Receiving EXEC_PLUGIN...\n", CMD_EXEC_PLUGIN);
	  TRY(cmd_send_ack(s));

	  TRY(cmd_recv_string(s, &plugin_path));
	  TRY(cmd_recv_string(s, &filename));

	  TRY(cmd_recv_ok(s));

#ifdef __UNIX__
	  /* Once you have the path, we can fork */
	  pid = fork();
	  if(pid != 0)
	    {
	      DISPLAY2("   Fork: create a child, PID=%i\n", pid);
	    }
	  else
	    {
	      DISPLAY2("-> Child process execing <%s> with arg <%s>\n",
		      plugin_path, filename);
	      if(execl(plugin_path, plugin_path, filename, NULL) == -1)
		{
		  DISPLAY2("!!! exec failed !!!\n");
		  print_last_error("exec");
		  
		}
	      exit(0);
	    }
#elif defined(__WIN32__)
	DISPLAY2("-> Child process execing <%s> with arg <%s>\n",
		      plugin_path, filename);
	      if(_spawnl(_P_NOWAIT, plugin_path, plugin_path, filename, NULL) == -1)
		{
		  DISPLAY2("!!! exec failed !!!\n");
		  print_last_error("exec");
		  
		}
#endif

	  break;
	default:
	  //DISPLAY2("Ignored.\n");
	  //DISPLAY2("Unknown command: %i\n", cmd);
	  break;
	}
    }

  shm_detach(&s);
  shm_destroy(&s);
  DISPLAY2("   Done.\n");

  return 0;
}

/*
 * Display a short help
 */
int help(void)
{

  DISPLAY2(_("\n"));
  DISPLAY2(_("tiffep %s, (C) Romain Lievin 1999-2000\n"), VERSION);
  //DISPLAY2(_("THIS PROGRAM COMES WITH ABSOLUTELY NO WARRANTY\n"));
  //DISPLAY2(_("PLEASE READ THE DOCUMENTATION FOR DETAILS\n"));
  DISPLAY2(_("usage: tiffep [-options] [filename]\n"));
  DISPLAY2("\n");
  DISPLAY2(_("-h, --help    display this information page and exit\n"));
  DISPLAY2(_("-v, --version display the version information and exit\n"));
  DISPLAY2(_("-r, --reg     register the application in the registry\n"));
  DISPLAY2(_("\n"));

  exit(0);
  return 0;
}

/*
 * Display the program version
 */
int version(void)
{
  DISPLAY2(_("tiffep - Version %s, (C) Romain Lievin 2000\n"), VERSION);
  DISPLAY2(_("THIS PROGRAM COMES WITH ABSOLUTELY NO WARRANTY\n"));
  DISPLAY2(_("PLEASE READ THE DOCUMENTATION FOR DETAILS\n"));

  return 0;
}

static /*inline*/ int strexact(char *p1, char *p2)
{
  return (strstr(p1,p2) && strstr(p2,p1));
}

/*
 * Scan the command line, extract arguments and init variables
 */
int scan_cmdline(int argc, char **argv)
{
  int cnt;
  char *p;
  //char *q;
  char msg[80];
  
  for(cnt=1; cnt<argc; cnt++)
    {
      p=argv[cnt];
      if(*p=='-' ) 
	{
	  // an option
	  p++;
	}
      else
	{
	  // a filename to send
	  //printf("Filename to send: <%s>\n", p);
	}
      strcpy(msg, p);
      if(strstr  (msg, "reg    "   ))
	{

	}

      /*
      if(strstr  (msg, "cmdline"   ))
        {
          working_mode = MODE_CMD;
        }
      if(strstr  (msg, "console"   ))
	{
	  working_mode = MODE_CON;
	}
      if(strstr  (msg, "gui="      ))
	{
	  q = msg+4;
	  if(!strcmp(q, "console")) working_mode = MODE_CON;
	  if(!strcmp(q, "gtk")) working_mode = MODE_GTK;
	  if(!strcmp(q, "mfc")) working_mode = MODE_MFC;
	}
      */
      if(strexact(msg, "-help"     )) help();
      if(strexact(msg, "-version"  )) { version(); exit(0); }
    }

  return 0;
}

/*****************************************************************
 * If TiFFEP is compiled in console mode (_CONSOLE), then we use 
 * the 'main' entry point.
 * If TiLP is compiled as a windowed application (_WINDOWS), then 
 * we use the 'WinMain' entry point.
 *****************************************************************
 */
#if defined(__WIN32__) && defined(_WINDOWS)// && !defined(_CONSOLE)
int APIENTRY WinMain(HINSTANCE hInstance,
                     HINSTANCE hPrevInstance,
                     LPSTR     lpCmdLine,
                     int       nCmdShow)
{
	return main(__argc, __argv, NULL);
}
#endif

/*
 * Load default options
 */
int load_default_options()
{
  options.current_dir = g_strdup("");
  options.running = FALSE;
  options.registry = NULL;
  
  return 0;
}
