/*----------------------------------------------------------------------------

   libtunepimp -- The MusicBrainz tagging library.  
                  Let a thousand taggers bloom!
   
   Copyright (C) Robert Kaye 2003
   
   This file is part of libtunepimp.

   libtunepimp is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   libtunepimp is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with libtunepimp; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

   $Id: filelookup.h,v 1.23 2005/02/21 01:10:28 robert Exp $

----------------------------------------------------------------------------*/
#ifndef FILELOOKUP_H
#define FILELOOKUP_H

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <vector>
#include <string>
using namespace std;
#include <musicbrainz/mb_c.h>
#include "defs.h"
#include "lookuptools.h"
#include "thread.h"
#include "semaphore.h"
#include "metadata.h"

#define DB printf("%s:%d\n", __FILE__, __LINE__);

const int fArtistId = 1;
const int fArtistLid = 2;
const int fAlbumId = 4;
const int fAlbumList = 8;
const int fTrackId = 16;
const int fTrackList = 32;
const int fAlbumTrackId = 64;
const int fAlbumTrackList = 128;
const int fFuzzy = 256;

class Track;

class TPResult
{
    public:

               TPResult(void) { rel = 0; };
      virtual ~TPResult(void) { };

       void          setRelevance(int rel)
                     { this->rel = rel; };
       int           getRelevance(void)
                     { return rel; };

       void          setId(const string &id)
                     { this->id = id; };
       const string &getId(void)
                     { return id; };

       void          setName(const string &name)
                     { this->name = name; };
       const string &getName(void)
                     { return name; };

       void          operator=(const TPResult &other)
                     { rel = other.rel;
                       name = other.name;
                       id = other.id;
                     }
    protected:

       int           rel;
       string        name, id;
};

class TPArtistResult : public TPResult
{
    public:


       void          setSortName(const string &sortName)
                     { this->sortName = sortName; };
       const string &getSortName(void)
                     { return sortName; };

       void          operator=(const TPArtistResult &other)
                     { rel = other.rel;
                       name = other.name;
                       id = other.id;
                       sortName = other.sortName;
                     }
    private:

       string        sortName;
};

class TPAlbumResult : public TPResult
{
    public:

                TPAlbumResult(void)
                       { 
                         numTracks = numCDIndexIds = 0; 
                         type = eAlbumType_Error;
                         status = eAlbumStatus_Error;
                         isVA = false; 
                         isNA = false; 
                         releaseYear = releaseMonth = releaseDay = 0;
                       };
       virtual ~TPAlbumResult(void) {}; 

       void            setArtist(const TPArtistResult &artist)
                       { this->artist = artist; };
       const TPArtistResult &getArtist(void)
                       { return artist; };

       void            setNumTracks(int numTracks)
                       { this->numTracks = numTracks; };
       int             getNumTracks(void)
                       { return numTracks; };
                      
       void            setNumCDIndexIds(int numCDIndexIds)
                       { this->numCDIndexIds = numCDIndexIds; };
       int             getNumCDIndexIds(void)
                       { return numCDIndexIds; };
                      
       void            setType(TPAlbumType type)
                       { this->type = type; };
       TPAlbumType     getType(void)
                       { return type; };

       void            setStatus(TPAlbumStatus status)
                       { this->status = status; };
       TPAlbumStatus   getStatus(void)
                       { return status; };

       void            setVariousArtists(bool isVA)
                       { this->isVA = isVA; };
       int             getVariousArtists(void)
                       { return isVA; };

       void            setNonAlbum(bool isNA)
                       { this->isNA = isNA; };
       int             getNonAlbum(void)
                       { return isNA; };
                      
       void            setReleaseYear(int releaseYear)
                       { this->releaseYear = releaseYear; };
       int             getReleaseYear(void)
                       { return releaseYear; };
                      
       void            setReleaseMonth(int releaseMonth)
                       { this->releaseMonth = releaseMonth; };
       int             getReleaseMonth(void)
                       { return releaseMonth; };
                      
       void            setReleaseDay(int releaseDay)
                       { this->releaseDay = releaseDay; };
       int             getReleaseDay(void)
                       { return releaseDay; };
                      
       void            setReleaseCountry(string releaseCountry)
                       { this->releaseCountry = releaseCountry; };
       string          getReleaseCountry(void)
                       { return releaseCountry; };
                      
    private:

       int             numTracks, numCDIndexIds;
       int             releaseYear, releaseMonth, releaseDay;
       string          releaseCountry;
       bool            isVA;
       bool            isNA;
       TPAlbumStatus   status;
       TPAlbumType     type;
       TPArtistResult  artist;
};

class TPAlbumTrackResult : public TPResult
{
    public:

                TPAlbumTrackResult(void) {}; 
       virtual ~TPAlbumTrackResult(void) {}; 

       const TPArtistResult &getArtist(void)
                            { return artist; };
       void                 setArtist(const TPArtistResult &artist)
                            { this->artist = artist; };

       const TPAlbumResult &getAlbum(void)
                            { return album; };
       void                 setAlbum(const TPAlbumResult &album)
                            { this->album = album; };

                            // Duration is in milliseconds
       void                 setDuration(unsigned long duration)
                            { this->duration = duration; };
       unsigned long        getDuration(void)
                            { return duration; };

       void                 setTrackNum(int trackNum)
                            { this->trackNum = trackNum; };
       int                  getTrackNum(void)
                            { return trackNum; };

       void                 setNumTRMIds(int numTRMIds)
                            { this->numTRMIds = numTRMIds; };
       int                  getNumTRMIds(void)
                            { return numTRMIds; };

    private:

       int             numTRMIds, trackNum;
       unsigned long   duration;
       TPArtistResult  artist;
       TPAlbumResult   album;
};

class LookupFile : public Lookup
{
    public:

                 LookupFile(TunePimp *pimp) : Lookup(pimp) { type = eNone; };
        virtual ~LookupFile(void) {};

        void          setArgs(const string   &trm, 
                              const string   &fileName,
                              const Metadata &data)
                      {
                          this->trm = trm;
                          this->data = data;
                          this->fileName = fileName;
                      };
        TPResultType  getResultType(void) 
                      { 
                          return type; 
                      };
        void          getResults(vector<TPResult *> &results, Metadata &mdata)
                      {
                          results = this->results;
                          mdata = this->data;
                      };

        LookupStatus  lookup(void);

    private:

        void          extractMatch(musicbrainz_t o);
        int           extractArtistList(musicbrainz_t o);
        int           extractAlbumList(musicbrainz_t o);
        int           extractTrackList(musicbrainz_t o);

    private:

        vector<TPResult *> results;
        TPResultType       type;
        string             trm, fileName;
        Metadata           data;
};

class TunePimp;
class FileCache;

class FileLookupThread : public Thread
{
    public:

                 FileLookupThread(TunePimp *tunePimpArg, FileCache *cacheArg);
        virtual ~FileLookupThread(void);

        void     wake(void);
        void     threadMain(void);

        void     setServer(const string &server, short port)
                    {
                        this->server = server;
                        this->port = port;
                    };
        void     setProxy(const string &proxyServer, short proxyPort)
                    {
                        this->proxyServer = proxyServer;
                        this->proxyPort = proxyPort;
                    };

    private:

        TunePimp       *tunePimp;
        FileCache      *cache;
        bool            exitThread;
        Semaphore      *sem;
        string          server, proxyServer;
        short           port, proxyPort;
};

#endif
