/***************************************************************************
                     libxbox-i2c.c - I2C functions for libxbox
                             -------------------
    begin                : Thu Dec 09 23:02:12 BST 2004
    copyright            : (C) 2004 by David Pye
    email                : dmp@davidmpye.dyndns.org
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <stdbool.h>
#include <string.h>

#include "libxbox-eeprom.h"
#include "libxbox-i2c.h"
#include "xboxi2c-dev.h"
#include "libxbox.h"

//Internal SMBus accessor functions
//=============================================================================
static int xbox_open_i2c_dev() {
        int fd;
	if ((fd = open("/dev/i2c/0", O_RDWR)) > 0) return fd;
	if ((fd = open("/dev/i2c-0",O_RDWR)) > 0) return fd;
	fprintf(stderr,"Unable to open I2C devices /dev/i2c/0 and /dev/i2c-0:%s\n",strerror(errno));
	return -1;
}

static bool xbox_smc_write(unsigned char reg, unsigned char cmd) {
	bool result=false;
	int write;
	int i2cfd = xbox_open_i2c_dev();
	if (i2cfd>=0) {
		if (ioctl(i2cfd,I2C_SLAVE_FORCE,XBOX_SMC_ADDR) >= 0) {
			write = i2c_smbus_write_byte_data(i2cfd, reg, cmd);
			if (write!=-1) result=true;
		}
		close(i2cfd);
	}
	if (!result) {
		fprintf(stderr,"SMC I2C write reg 0x%02X, cmd 0x%02X failed:%s\n", reg, cmd, strerror(errno));
	}
	usleep(25);
	return result;
}

static bool xbox_smc_read(unsigned char reg, unsigned char *data) {
	bool result=false;
	char c;
	int i2cfd = xbox_open_i2c_dev();
	if (i2cfd>0) {
		if (ioctl(i2cfd,I2C_SLAVE_FORCE,XBOX_SMC_ADDR) >= 0) {
			c = i2c_smbus_read_byte_data(i2cfd, reg);
			if (c!=-1) result=true;
			*data = c;
		}
		close(i2cfd);
	}
	if (!result) {
		fprintf(stderr,"SMC I2C read reg 0x%02X failed:%s\n", reg, strerror(errno));
	}
	usleep(25);
	return result;
}

//EEPROM functions
//=============================================================================
bool xbox_read_eeprom(EEPROMDATA *eepromdata) {
	int i2cfd = xbox_open_i2c_dev();
	bool retval=false;
	int i=0;
	if (i2cfd>0) {
		if (ioctl(i2cfd,I2C_SLAVE_FORCE,XBOX_EEPROM_ADDR) >= 0) {
			for (i=0; i<255; ++i) {
				((unsigned char *)eepromdata)[i] = i2c_smbus_read_byte_data(i2cfd, i);
			}
			retval = true;
		}
		close(i2cfd);
	}
	usleep(25);
	return retval;
}

bool xbox_write_eeprom(EEPROMDATA *eepromdata) {
	int i2cfd = xbox_open_i2c_dev();
	bool retval=false;
	int i=0;
	unsigned char *data = (unsigned char *)eepromdata;
	if (i2cfd>0) {
		if (ioctl(i2cfd,I2C_SLAVE_FORCE,XBOX_EEPROM_ADDR) >= 0) {
			for (i=0; i<255; ++i) {
				i2c_smbus_write_byte_data(i2cfd, i, data[i]);
			}
			retval = true;
		}
		close(i2cfd);
	}
	usleep(25);
	return retval;
}

//Thermal mgmt functions
//=============================================================================
bool xbox_temperatures(unsigned int *cpu, unsigned int *mcpx) {
	return (xbox_smc_read(CPU_TEMP_REG, (char*)cpu) && xbox_smc_read(MCPX_TEMP_REG, (char*)mcpx));
}

bool xbox_fanspeed(unsigned int *percentage) {
	if (xbox_smc_read(SMC_FAN_READBACK_REG, (char *)percentage)) {
		*percentage *= 2;
		return true;
	}
	else return false;
}

bool xbox_set_fanspeed_manual(unsigned int percentage) {
	return (xbox_smc_write(SMC_FAN_CTL_MODE, 0x01) && xbox_smc_write(SMC_FAN_SPEED_REG, percentage/2));
}


bool xbox_set_fanspeed_auto() {
	return (xbox_smc_write(SMC_FAN_SPEED_REG, 10) && xbox_smc_write(SMC_FAN_CTL_MODE, 0x00));
}

//DVD functions
//=============================================================================
bool xbox_enable_reset_on_eject() {
	//FIXME - check this is correct
	return xbox_smc_write(SMC_RST_ON_EJECT_REG, 0x01);
}

bool xbox_eject_dvd_drive() {
	return xbox_smc_write(SMC_DVD_CTL_REG, SMC_DVD_EJECT_CMD);
}

bool xbox_load_dvd_drive() {
	return xbox_smc_write(SMC_DVD_CTL_REG, SMC_DVD_LOAD_CMD);
}

//Power control functions
//=============================================================================
bool xbox_poweroff() {
	return xbox_smc_write(SMC_POWER_CTL_REG, SMC_POWEROFF_CMD);
}

bool xbox_reboot_slow() {
	return xbox_smc_write(SMC_POWER_CTL_REG, SMC_REBOOT_SLOW_CMD);
}

bool xbox_reboot_fast() {
	return xbox_smc_write(SMC_POWER_CTL_REG, SMC_REBOOT_FAST_CMD);
}

//Probes
//=============================================================================
bool xbox_smc_version(char *ver) {
	int i;
	bool result=false;
	//Reset the counter
	if (xbox_smc_write(0x01,0x01)) {
		for (i=0; i<3; ++i)
			xbox_smc_read(0x01, &ver[i]);
		result = true;
	}
	return result;
}

bool xbox_tv_encoder(XBOX_TV_ENCODER *enc) {
	int i2cfd=xbox_open_i2c_dev();
	if (i2cfd>=0) {
		if (ioctl(i2cfd,I2C_SLAVE_FORCE, XBOX_ENC_CONEXANT_ADDR)>=0 && i2c_smbus_read_byte_data(i2cfd, 0x00)>=0) {
			*enc = ENCODER_CONEXANT;
		}
		else if (ioctl(i2cfd,I2C_SLAVE_FORCE,XBOX_ENC_FOCUS_ADDR)>=0 && i2c_smbus_read_byte_data(i2cfd, 0x00)>=0) {
			*enc = ENCODER_FOCUS;
		}
		else if (ioctl(i2cfd,I2C_SLAVE_FORCE,XBOX_ENC_XCALIBUR_ADDR)>=0 && i2c_smbus_read_byte_data(i2cfd, 0x00)>=0) {
			*enc = ENCODER_XCALIBUR;
		}
		else *enc = ENCODER_UNKNOWN;	
	}
	close(i2cfd);
	return (*enc != ENCODER_UNKNOWN);
}

bool xbox_av_cable(XBOX_AVIP_CABLE *cable) {
	unsigned int cableType=0;
	bool result=false;
	*cable = CABLE_UNKNOWN;
	if (xbox_smc_read(0x04, (char*)&cableType)) {
		result = true;
		switch (cableType) {
			case 0:
				*cable = CABLE_RGB_SCART;
				break;
			case 1:
				*cable = CABLE_HDTV;
				break;
			case 2:
				*cable = CABLE_VGA_SOG;
				break;
			case 4: 
				*cable = CABLE_SVIDEO;
				break;
			case 6:
				*cable = CABLE_COMPOSITE;
				break;
			case 7:
				*cable = CABLE_VGA;
				break;
		}
	}
	return result;
}

bool xbox_led_color(unsigned char *c) {
	return (xbox_smc_read(SMC_LED_COLOR_REG, c));
}

bool xbox_set_led_color_manual(unsigned char c) {
	return (xbox_smc_write(SMC_LED_COLOR_REG, c) && xbox_smc_write(SMC_LED_MODE_REG, 0x01));
}


bool xbox_set_led_color_auto() {
	return (xbox_smc_write(SMC_LED_COLOR_REG, 0xFF) && xbox_smc_write(SMC_LED_MODE_REG, 0x00));
}
