/***************************************************************************
 *
 * This file is covered by a dual licence. You can choose whether you
 * want to use it according to the terms of the GNU GPL version 2, or
 * under the terms of Zorp Professional Firewall System EULA located
 * on the Zorp installation CD.
 *
 * $Id: cap.c,v 1.9 2003/09/10 11:46:58 bazsi Exp $
 *
 *
 * Author  : Bazsi
 * Auditor :
 * Last audited version:
 * Notes:
 *
 ***************************************************************************/

#include <zorp/zorplib.h>
#include <zorp/cap.h>
#include <zorp/log.h>

#if ZORPLIB_ENABLE_CAPS

const gchar *zorp_caps = "cap_net_admin,cap_net_bind_service=p";

/**
 * cap_modify:
 * @capability: capability to turn off or on
 * @onoff: specifies whether the capability should be enabled or disabled
 *
 * This function modifies the current permitted set of capabilities by
 * enabling or disabling the capability specified in @capability.
 *
 * Returns: whether the operation was successful.
 **/
gboolean 
cap_modify(int capability, int onoff)
{
  cap_t caps;

  z_enter();
  if (!zorp_caps)
    {
      z_leave();
      return TRUE;
    }
  
  caps = cap_get_proc();
  if (!caps)
    {
      z_leave();
      return FALSE;
    }


  if (cap_set_flag(caps, CAP_EFFECTIVE, 1, &capability, onoff) == -1)
    {
      cap_free(caps);
      z_leave();
      return FALSE;
    }

  if (cap_set_proc(caps) == -1)
    {
      cap_free(caps);
      z_leave();
      return FALSE;
    }
  cap_free(caps);
  z_leave();
  return TRUE;
}

/**
 * cap_save:
 *
 * Save the set of current capabilities and return it. The caller might
 * restore the saved set of capabilities by using cap_restore().
 *
 * Returns: the current set of capabilities
 **/
cap_t 
cap_save(void)
{
  cap_t res;
  
  z_enter();
  if (!zorp_caps)
    {
      z_leave();
      return NULL;
    }
  res = cap_get_proc();
  z_leave();
  return res;
}

/**
 * cap_restore:
 * @r: capability set saved by cap_save()
 *
 * Restore the set of current capabilities specified by @r.
 *
 * Returns: whether the operation was successful.
 **/
gboolean
cap_restore(cap_t r)
{
  gboolean rc;

  z_enter();
  if (!zorp_caps)
    {
      z_leave();
      return TRUE;
    }
  rc = cap_set_proc(r) != -1;
  cap_free(r);
  z_leave();
  return rc;
}

#endif
