/*
Magpie - reference librarian for Debian systems
Copyright (C) 2000  Bear Giles <bgiles@coyotesong.com>

This program is free software; you may redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the license, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

static const char rcsid[] = "$Id$";

/*****
This program generates a series of cross-referenced HTML pages
based on the content of the apt and dpkg caches.

This software contains tags recognized by cxref(1).
*****/
#include <assert.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <locale.h>
#include <time.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <pwd.h>
#include <grp.h>
#include "magpie.h"

char timestamp[30];

extern int mkdir (const char *, mode_t);

/*+
For now, we hard-link our modules.
+*/
extern struct magpie_module mod_html;
extern struct magpie_module mod_debian;
extern struct magpie_module mod_core;
extern struct magpie_module mod_xml;
extern struct magpie_module mod_docbook;
extern struct magpie_module mod_debiandoc;
extern struct magpie_module mod_texinfo;

extern struct magpie_module mod_sections;
extern struct magpie_module mod_priorities;
extern struct magpie_module mod_keywords;
extern struct magpie_module mod_installed;
extern struct magpie_module mod_tasks;
extern struct magpie_module mod_names;
extern struct magpie_module mod_maintainers;
extern struct magpie_module mod_sources;
extern struct magpie_module mod_sizes;
extern struct magpie_module mod_installed_sizes;
extern struct magpie_module mod_md5sums;
extern struct magpie_module mod_conffiles;

struct magpie_module *modules[] = {
	
	&mod_html,		/* produce HTML output */
	&mod_debian,	/* read Debian database files */
	&mod_core,		/* generate reference material, 'current' format */
//	&mod_xml,		/* generate reference material, XML format */
	&mod_docbook,	/* generate reference material, docbook format */
//	&mod_debiandoc,	/* generate reference material, debiandoc format */
//	&mod_texinfo,	/* generate reference material, texinfo format */

#if 1
	&mod_sections,
	&mod_priorities,
	&mod_keywords,
	&mod_installed,
	&mod_tasks,
	&mod_names,
	&mod_maintainers,
	&mod_sources,
	&mod_sizes,
	&mod_installed_sizes,
	&mod_md5sums,
	&mod_conffiles,
#endif
	NULL
};

extern int rmrf (const char *);


/*+
The main procedure.
+*/
int main (int argc, char *argv[])
{
	FILE *fp;
	char pathname[256], buffer[256];
	static const char prefix[] = "/var/cache/magpie";
	time_t now;
	char user[] = "magpie";
	char group[] = "magpie";
	uid_t uid;
//	gid_t gid;
	struct passwd *pw;
	struct group *gr;
	struct magpie_module **m;

	if (mkdir (prefix, 0755) == -1 && errno != EEXIST) {
		sprintf (buffer, "mkdir (%s)", prefix);
		perror (buffer);
		exit (0);
	}

	if (chdir (prefix) == -1) {
		sprintf (buffer, "chdir (%s)", prefix);
		perror (buffer);
		exit (0);
	}

	/* change effective uid to 'magpie' here, if we're running
	 * as root and that user is known.  We don't surrender root
	 * privileges since we can use them later.
	 */
	uid = getuid();
	if (uid == 0 && (pw = getpwnam (user)) != NULL) {
		gr = getgrnam (group);
		if (gr) {
			chown (prefix, pw->pw_uid, gr->gr_gid);
			setgid (gr->gr_gid);
		}
		else {
			chown (prefix, pw->pw_uid, -1);
		}
		seteuid (pw->pw_uid);
	}

	setlocale (LC_ALL, NULL);

	time (&now);
	strcpy (timestamp, ctime (&now));

	rmrf (prefix);

	/* read the database(s) */
	for (m = modules; *m; m++) {
		if ((*m)->database)
			(*m)->database ();
	}

	/* generate 'reference' material */
	for (m = modules; *m; m++) {
		if ((*m)->init)
			(*m)->init ();
	}

	sprintf (pathname, "./index.html");
	fp = fopen (pathname, "w");
	if (fp == NULL) {
		perror ("fopen (./index.html)");
	}
	else {
		mp_doc_open (fp, "");

		mp_abstract (fp, "\
These pages contain information on all packages known to either\n\
apt(8) or dpkg(8), with extensive cross-referencing.\n");

		mp_title (fp, 2, "Annotated Indexes, All Packages");
		for (m = modules; *m; m++) {
			if ((*m)->annotated_index)
				(*m)->annotated_index (fp, MAGPIE_ALL_PACKAGES);
		}
	
		mp_title (fp, 2, "Annotated Indexes, Installed Packages");
		for (m = modules; *m; m++) {
			if ((*m)->annotated_index)
				(*m)->annotated_index (fp, MAGPIE_INSTALLED_PACKAGES);
		}

		mp_title (fp, 2, "Unannotated Indexes, All Packages");
		mp_list_open (fp);
		for (m = modules; *m; m++) {
			if ((*m)->unannotated_index)
				(*m)->unannotated_index (fp, MAGPIE_ALL_PACKAGES);
		}
		mp_list_close (fp);

		mp_title (fp, 2, "Unannotated Indexes, Installed Packages");
		mp_list_open (fp);
		for (m = modules; *m; m++) {
			if ((*m)->unannotated_index)
				(*m)->unannotated_index (fp, MAGPIE_INSTALLED_PACKAGES);
		}
		mp_list_close (fp);

		mp_title (fp, 2, "Miscellaneous Material");
		mp_list_open (fp);
		for (m = modules; *m; m++) {
			if ((*m)->miscellaneous)
				(*m)->miscellaneous (fp);
		}
		mp_list_close (fp);

		mp_doc_close (fp);
		fclose (fp);
	}

	for (m = modules; *m; m++) {
		if ((*m)->cleanup)
			(*m)->cleanup ();
	}

	return 0;
}
