#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/time.h>

#include "mdldapconfig.h"
#include "mdldap.h"

#include <lber.h>
#include <ldap.h>

mdldaprec *get_user_ldap(mdldapconfig *cfg,char *username)
{
  LDAP *ldap;
  LDAPMessage *result;
  LDAPMessage *entry;
  char *tmp_filter,*filter;
  char *dn;
  char **values;
  int size;
  mdldaprec *rec = NULL;
  int needbind = 0;
  char *mail_filter;
  struct timeval tv;

  ldap = ldap_init(cfg->hostname,cfg->port);
  if ( !ldap )
    return NULL;

  if ( cfg->binddn && cfg->bindpw )
    {
      needbind = 1;
      if ( ldap_simple_bind_s(ldap,cfg->binddn,cfg->bindpw) != 0 )
	return NULL;
    }

  if ( !strcasecmp(cfg->search_method,"mail") )
    {
      mail_filter = (char*)malloc(strlen(cfg->mail_attr)+5);
      snprintf(mail_filter,strlen(cfg->mail_attr)+4,"%s=%%s",cfg->mail_attr);
    } else {
      mail_filter = (char*)malloc(strlen(cfg->uid_attr)+5);
      snprintf(mail_filter,strlen(cfg->uid_attr)+4,"%s=%%s",cfg->uid_attr);
    }

  size = strlen(mail_filter)+strlen(username);
  filter = (char*)malloc(size+1);
  snprintf(filter,size,mail_filter,username);

  if ( cfg->filter )
    {
      size += strlen(cfg->filter) + 4;
      tmp_filter = (char*)malloc(size+1);
      snprintf(tmp_filter,size,"(%s(%s))",cfg->filter,filter);
      free(filter);
      filter = strdup(tmp_filter);
      free(tmp_filter);
    }

  tv.tv_sec = cfg->timeout;
  tv.tv_usec = 0;

  if ( ldap_search_st(ldap,cfg->basedn,LDAP_SCOPE_SUBTREE,filter,NULL,0,&tv,&result) != LDAP_SUCCESS )
    {
      if ( needbind )
	ldap_unbind(ldap);
      free(mail_filter);
      free(filter);
      return NULL;
    }

  entry = ldap_first_entry(ldap,result);

  if ( !entry )
    {
      if ( needbind )
	ldap_unbind(ldap);
      free(mail_filter);
      free(filter);
      return NULL;
    }

  rec = (mdldaprec*)malloc(sizeof(mdldaprec)+1);

  dn = ldap_get_dn(ldap,entry);
  rec->dn = strdup(dn);

#if HAVE_LDAP_MEMFREE
  ldap_memfree(dn);
#else
  free(dn);
#endif

  values=ldap_get_values(ldap,entry,cfg->maildir_attr);
  if ( ldap_count_values(values) > 0 )
    {
      rec->maildir = strdup(values[0]);
      ldap_value_free(values);
    } else
      rec->maildir = NULL;
  

  values=ldap_get_values(ldap,entry,cfg->mail_attr);
  if ( ldap_count_values(values) > 0 ) {
      rec->mail = strdup(values[0]);
      ldap_value_free(values);
  } else {
      rec->mail = NULL;
  }

  values=ldap_get_values(ldap,entry,cfg->uidnumber_attr);
  if ( ldap_count_values(values) > 0 )
    {
      rec->uidnumber = atoi(values[0]);
      ldap_value_free(values);
    } else {
      rec->uidnumber = cfg->default_uidnumber;
    }


  values=ldap_get_values(ldap,entry,cfg->gidnumber_attr);
  if ( ldap_count_values(values) > 0 )
    {
      rec->gidnumber = atoi(values[0]);
      ldap_value_free(values);
    } else {
      rec->gidnumber = cfg->default_gidnumber;
    }

  values=ldap_get_values(ldap,entry,cfg->homedirectory_attr);
  if ( ldap_count_values(values) > 0 ) {
  	rec->homedirectory = strdup(values[0]);
  	ldap_value_free(values);
  } else {
  	rec->homedirectory = NULL;
  }

  values=ldap_get_values(ldap,entry,cfg->quota_attr);
  if ( ldap_count_values(values) > 0 )
    {
      rec->quota = strdup(values[0]);
      ldap_value_free(values);
    } else {
      rec->quota = NULL;
    }
  
  if ( needbind )
    ldap_unbind(ldap);

  free(mail_filter);
  free(filter);

  return rec;
}

void free_ldap_rec(mdldaprec **rec)
{
  if ( rec[0]->dn )
    free(rec[0]->dn);

  if ( rec[0]->maildir )
    free(rec[0]->maildir);

  if ( rec[0]->mail )
    free(rec[0]->mail);

  if ( rec[0]->homedirectory )
    free(rec[0]->homedirectory);
  
  if ( rec[0]->quota )
    free(rec[0]->quota);

  free(*rec);

  *rec = NULL;
}
