#   MailScanner - SMTP E-Mail Virus Scanner
#   Copyright (C) 2001  Julian Field
#
#   $Id: info.pl,v 1.2 2001/12/10 16:49:50 nwp Exp $
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#   The author, Julian Field, can be contacted by email at
#      Jules@JulianField.net
#   or by paper mail at
#      Julian Field
#      Dept of Electronics & Computer Science
#      University of Southampton
#      Southampton
#      SO17 1BJ
#      United Kingdom
#

use strict;
use MIME::Parser;

# Walk the MIME tree for each message in the list we are passed,
# and work out number of parts to each message. Construct a new
# email header string from this info.
# Each message has a minimum of 1 part (may be main message text).
# Passes out the headers as a hash.
# Passes out a hash of entity->filename for every section of every message.
# Returns a hash of messageid and filename --> MIME entity.
sub EntitiesInfo {
  my($entities, $NumberParts, $Entity2Parent, $Entity2File) = @_;

  my($id, $entity, %File2Entity);
  while(($id, $entity) = each %$entities) {
    $NumberParts->{$id} = CountParts($entity) || 1;
    # Make each one a ref to an empty hash
    $File2Entity{$id} = {};
    $File2Entity{$id}{""} = $entity; # Root of this message
    BuildFile2EntityAndEntity2File($entity, $File2Entity{$id},
                                   $Entity2File);
    BuildEntity2Parent($entity, $Entity2Parent);
  }
  return \%File2Entity; # Stop it blowing it out into a list...
}

# For the MIME entity given, work out the number of message parts.
# Recursive.
sub CountParts {
  my($entity) = @_;
  my(@parts, $total, $part);

  @parts = $entity->parts;
  $total += int(@parts);
  foreach $part (@parts) {
    $total += CountParts($part);
  }
  return $total;
}

# Build a hash of filename to entity. This will let us replace infected
# entities later. Key is "messageid,filename". Value is the entity.
# Passed in an entity and a ref to a hash of results.
# Fills the hash of filename-->entity.
# Now also builds the reverse to map entity-->filename.
sub BuildFile2EntityAndEntity2File {
  my($entity, $file2entity, $entity2file) = @_;

  my(@parts, $body, $headfile, $part, $path);

  # Find the body for this entity
  $body = $entity->bodyhandle;
  if (defined($body) && defined($body->path)) {   # data is on disk:
    $path = $body->path;
    $path =~ s#^.*/([^/]*)$#$1#;
    $file2entity->{$path} = $entity;
    $entity2file->{$entity} = $path;
  }
  # And the head, which is where the recommended filename is stored
  # This is so we can report infections in the filenames which are
  # recommended, even if they are evil and we hence haven't used them.
  $headfile = $entity->head->recommended_filename;
  $file2entity->{$headfile} = $entity
    if $headfile && !$file2entity->{$headfile};

  # And for all its children
  @parts = $entity->parts;
  foreach $part (@parts) {
    BuildFile2EntityAndEntity2File($part, $file2entity, $entity2file);
  }
}

# Build a hash that gives the parent of any entity (except for root ones which
# will be undef).
sub BuildEntity2Parent {
  my($entity, $Entity2Parent, $parent) = @_;

  my(@parts, $part);

  $Entity2Parent->{$entity} = $parent;
  @parts = $entity->parts;
  foreach $part (@parts) {
    $Entity2Parent->{$part} = $entity;
    BuildEntity2Parent($part, $Entity2Parent, $entity);
  }
}

# Various debugging output routines
sub PrintParts {
  my($NumParts) = @_;
  my($id);

  foreach $id (keys %$NumParts) {
    print "Message $id has " . $NumParts->{$id} . " parts\n";
  }
}
sub PrintFilenames {
  my($File2Entity) = @_;
  my($id, $fnames, @filenames);

  print "\nFilenames:\n";
  while(($id, $fnames) = each %$File2Entity) {
    print "Message $id has files ";
    @filenames = keys %$fnames;
    print join(', ', @filenames) . "\n";
  }
}
sub PrintInfectedSections {
  my($InfectionReports, $File2Entity) = @_;
  my($id, $parts, $file, $entity);

  print "\nInfected sections are:\n";
  while(($id, $parts) = each %$InfectionReports) {
    foreach $file (keys %$parts) {
      $entity = $File2Entity->{$id}{$file};
      $entity->dump_skeleton;
    }
  }
}
sub PrintParents {
  my($Entity2Parent) = @_;
  my($key, $value);

  print "\nParent tree:\n";
  while(($key, $value) = each %$Entity2Parent) {
    print "Parent of $key is $value\n";
  }
}

1;
