/*!
    @file           DBMCli_EventHandlerExtendDB.hpp
    @author         MarcW
    @brief          handles DB_ABOVE_LIMIT events with extending the DB - Specification

\if EMIT_LICENCE

    ========== licence begin  GPL
    Copyright (c) 2003-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end

\endif
*/

#ifndef _DBMCli_EventHandlerExtendDB_HPP_
#define _DBMCli_EventHandlerExtendDB_HPP_

/*
  -----------------------------------------------------------------------------
  includes
  -----------------------------------------------------------------------------
 */
#include "hin100.h"
#include "hin105.h"
#include "SAPDBCommon/ErrorsAndMessages/SAPDBErr_MessageList.hpp"
#include "SAPDB/DBM/Cli/DBMCli_EventHandler.hpp"

/*!
    @brief status of the DB

    Description of the status of the database. Should and can a new datavolume be added?
    @see checkDB()
*/
typedef enum { 
    DBMCLI_DBEXTSTATE_CANADD = 0, /*!< a volume can be added */
	DBMCLI_DBEXTSTATE_NOTNEC = 1, /*!< not necessary to add one */
	DBMCLI_DBEXTSTATE_NOTONL = 2  /*!< db is not online, so no recheck possible */
} DBMCli_DBExtState;


/*!
    @brief event handler for DB_ABOVE_LIMIT

    This event handler requires a serialized event and DB connect data on the command line
    (which is passed in the constructor). An optional parameter "l" for the filling limit
    (between 60% and 95%) is also evaluated.
    The actions taken are documented in a log file (by default dbmevthndl_extendDB.prt in
    DB's rundirectory) and on stdout/stderr. Name and location of the logfile can be overridden
    with parameter "lf".

    If an event DB_ABOVE_LIMIT is found on the command line, it's value2 field is checked. If it's
    bigger that the limit, a new datavolume of the size of the last datavolume is added. By default,
    the new volume is added in the location of the last datavolume (this location can be overridden
    with DBM server parameter "AutoExtDir").
*/
class DBMCli_EventHandlerExtendDB: public DBMCli_EventHandler {
    public:
        /*!
            @brief Constructor

            Constructor taking a command line and a message list.
            @param argc         [IN] same as in main
            @param argv         [IN] same as in main
            @param oMessageList [OUT] message list for passing information out
        */
        DBMCli_EventHandlerExtendDB
            (int argc, char * argv[], SAPDBErr_MessageList& oMessageList);

        /*! @brief Destructor */
        ~DBMCli_EventHandlerExtendDB();

        /*!
            @brief handling the event
            
            Called from main function.
        */
        void HandleEvent();

    private:

        /*!
            @brief recheck degree of filling
            
            Recheck the degree of filling again using the database object (more accurate
            than the event data).
            @return indication what to do: add a volume or not
            @see DBMCli_DBExtState
        */
        DBMCli_DBExtState checkDB();

        /*!
            @brief add a new volume

            Adds a new datavolume to the database. Relies on the precondition, that the limit
            of the number of datavolumes is not reached yet.
            @return indicator whether operation was successful
        */
        bool addVolume();

        //SAPDB_Int m_nPercentPerNewVolume;
        /*!
            @brief filing limit

            Limit of degree of filling (percent). If actual degree of filling is bigger,
            a new datavolume is added.
        */
        SAPDB_Int m_nLimit;

        /*
            @brief verbosity

            If true, information messages are created (in log and on stdout), otherwise
            only error messages are created in log and stderr).
        */
        bool m_bVerbose;

        // to compare with m_nLimit
        static const SAPDB_Int ms_limitMax;
        static const SAPDB_Int ms_limitDefault;
        static const SAPDB_Int ms_limitMin;
        //static const SAPDB_Int ms_nSizeDefault;

        static const DBMCli_String ms_AutoExtDir;

        //static const DBMCli_String ms_LimitKey;
        //static const DBMCli_String ms_SizeKey;
        static const DBMCli_String ms_EventName;

        // logging
        tin100_GlobalLog* m_Logger;
        bool m_doLogging;
        static const DBMCli_String sRundirectory;
        static const DBMCli_String sPrtfile;

        /*!
            @brief create an error message

            Takes line (__LINE__) and file (__FILE__) where error occurred and a format
            string and string arguments. If logging was initialized successfully, not only
            stderr is written but also the log file.
        */
        void dumpError(const int line,
                       const char* file, 
                       const char* format,
                       const char* s0 = NULL,
                       const char* s1 = NULL,
                       const char* s2 = NULL,
                       const char* s3 = NULL,
                       const char* s4 = NULL ) const;

        /*!
            @brief create an info message

            Takes a format string and string arguments. If logging was initialized
            successfully, not only stdout is written but also the log file. Only
            functional in verbose mode.
        */
        void dumpText( const char* format,
                       const char* s0 = NULL,
                       const char* s1 = NULL,
                       const char* s2 = NULL,
                       const char* s3 = NULL,
                       const char* s4 = NULL ) const;
};

#endif /* _DBMCli_EventHandlerExtendDB_HPP_ */
