/*!---------------------------------------------------------------------
  @file           RTEConf_ParameterAccess.hpp
  @author         FrankH, JoergM, RobinW
  @brief          RTE configuration parameters: class for kernel configurable parameter access
  @see            example.html ...
  
  first created  2000-03-22  17:49

\if EMIT_LICENCE

    ========== licence begin  GPL
    Copyright (c) 2000-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end


\endif
---------------------------------------------------------------------*/


#ifndef CONF_PARAMETER_ACCESS_HPP
#define CONF_PARAMETER_ACCESS_HPP


/*===========================================================================*
 *  INCLUDES                                                                 *
 *===========================================================================*/

#include "SAPDBCommon/SAPDB_Types.hpp"
#include "RunTime/RTE_Types.hpp"
#include "RunTime/RTE_ISystem.hpp"
#include "RunTime/RTE_MessageList.hpp"
#include "RunTime/RTE_Message.hpp"
#include "RunTime/RTE_Messages.hpp"

#include "RunTime/System/RTESys_IO.h"  // file access routines 

#include "Container/Container_List.hpp"
/*===========================================================================*
 *  DEFINES                                                                  *
 *===========================================================================*/

/* 6 is cryptic... found in gsp00.h */
#define RTECONF_CRYPT_ARRAY_SIZE 6

#define RTECONF_BYTECODE_FOR_INTEGER_   ((SAPDB_Byte)0x11)
#define RTECONF_BYTECODE_FOR_REAL_      ((SAPDB_Byte)0x22)
#define RTECONF_BYTECODE_FOR_STRING_    ((SAPDB_Byte)0x33)
#define RTECONF_BYTECODE_FOR_CRYPTINFO_ ((SAPDB_Byte)0x44)

/* this should be used if no version is to be specified */
#define RTECONF_CURRENT_VERSION         ((SAPDB_Int4)0)

#define RTECONF_MAXSTRINGLENGTH         ((SAPDB_Int4)256)
#define RTECONF_MAXNAMELENGTH           ((SAPDB_Int4) 32)


/* number of lists for the characters from a to z */
#define NUMBER_OF_CHARS_FROM_A_TO_Z_    ('Z'-'A'+1)
/* number of lists for the characters from 0 to 9 */
#define NUMBER_OF_CHARS_FROM_0_TO_9_    ('9'-'0'+1)

/* the total number of lists: one for names shorter than KEY_INDEX_, one for each character from A to Z, one for each character
   from 0 to 9 and one list for the underscore */
#define TOTAL_NUMBER_OF_LISTS_ (1 + NUMBER_OF_CHARS_FROM_A_TO_Z_ + NUMBER_OF_CHARS_FROM_0_TO_9_ + 1)

/*===========================================================================*
 *  MACROS                                                                   *
 *===========================================================================*/

#define STRCMP_UTF8(string1,string2)            strcmp((SAPDB_Char *)(string1),(SAPDB_Char *)(string2))
#define STRNCMP_UTF8(string1,string2,length)    strncmp((SAPDB_Char *)(string1),(SAPDB_Char *)(string2),length)
#define STRLEN_UTF8(string)                     strlen((SAPDB_Char *)(string))
#define STRCPY_UTF8(string1,string2)            strcpy((SAPDB_Char *)(string1),(SAPDB_Char *)(string2))
 
 /*===========================================================================*
 *  CLASSES, STRUCTURES, TYPES, UNIONS ...                                   *
 *===========================================================================*/

/* These functions do the same thing: they return a copy of the string that has been given as the parameter */
/* The memory for the copy is allocated in the functions. Two different functions are provided to avoid the need */
/* for casting */
SAPDB_Char *RTEConf_MakeACopyOfString(SAPDB_Char *urString);
const SAPDB_UTF8 *RTEConf_MakeACopyOfString(const SAPDB_UTF8 *urString);

class RTEConf_ParameterIterator;
class RTEConf_ParameterRecord;
/*!--------------------------------------------------------------------
  @class          RTEConf_Parameter
   @description    Parameter access class


  This class allows to handle XParam-Files and its individual parameters.

 --------------------------------------------------------------------*/

class RTEConf_Parameter
{
public:
    typedef enum 
    { 
        t_Integer = RTECONF_BYTECODE_FOR_INTEGER_, 
        t_Real = RTECONF_BYTECODE_FOR_REAL_, 
        t_String = RTECONF_BYTECODE_FOR_STRING_, 
        t_CryptInfo = RTECONF_BYTECODE_FOR_CRYPTINFO_, 
        t_Invalid
    } Type;

    typedef SAPDB_UTF8 const    * Name;
    typedef SAPDB_Int8            Integer;
    typedef SAPDB_Real8           Real;
    typedef SAPDB_UTF8            StringElement;
    typedef StringElement const * String;
    typedef SAPDB_Int4            CryptInfo[RTECONF_CRYPT_ARRAY_SIZE];

    typedef SAPDB_Byte			* ByteArray;

	typedef RTE_ISystem::DateTime TimeStamp;

    /*!----------------------------------------------------------------
       @description    ctor


  Construct the object, but does not do any I/O. This is done in Read() member function.

     ----------------------------------------------------------------*/


    RTEConf_Parameter(const SAPDB_Char *dbname, SAPDB_Int4 version,SAPDB_Int4 maxLengthOfName,SAPDB_Int4 maxLengthOfStringValue);

    /*!----------------------------------------------------------------
       @description    copy ctor


  Construct the object using another object as template.
  The copies use independent storage for parameters, so value change
  on template afterwards have no effect on the object.
  Does not do any I/O, and copies state of dirty flag from template!
       @argument       templateParameter [in] the template parameter class

     ----------------------------------------------------------------*/

    RTEConf_Parameter(const RTEConf_Parameter &templateParameter);

    /*!----------------------------------------------------------------
       @description    assignment operator


  Copy all the content from template parameter object, but keep original filename.
  Does not do any I/O, but sets dirty flag!
       @argument       templateParameter [in] the template parameter class

     ----------------------------------------------------------------*/

    RTEConf_Parameter & operator=(const RTEConf_Parameter &templateParameter);

    /*!----------------------------------------------------------------
       @description    dtor


  Frees any memory allocated inside the object but does no I/O activity.
  I/O must be enforced by a Write() call.

     ----------------------------------------------------------------*/

    ~RTEConf_Parameter();

    /*!----------------------------------------------------------------
       @description    Read the Parameter file and convert into internal memory structure


  This does read in the complete parameter set and builds up the internal memory image.
  If no file was found, no error is returned, but dataRead flag is set to false.
       @param          dataRead [out] flag to indicate file was read (true) or does not exist (false)
       @param          err [out] error list filled, if false is returned
       @return value   true if fileNotFound is filled, false if err is filled

     ----------------------------------------------------------------*/

    SAPDB_Bool Read(SAPDB_Bool &dataRead, SAPDBErr_MessageList & err);

    /*!----------------------------------------------------------------
       @description    Write the Parameter file from internal memory structure if needed


  This does write out the complete parameter set from internal memory image. If the
  dirty flag was never set, it does nothing. If the file does not exist it is created.

     ----------------------------------------------------------------*/

    SAPDB_Bool Write(SAPDBErr_MessageList & err);

    /*!----------------------------------------------------------------
       @description    Add a block of binary data (structured like a parameter file) to an
                       internal storage. 
                       This can be used to feed the parameter class with the contents 
                       of a parameter file that are not stored as a file in a file system
                       (but - for example - stored inside a backup).
                       AddBinaryData() can be called more than once, all blocks are stored
                       internally in a list. After all blocks are put into the calls by calls
                       to AddBinaryData(), InterpretBinaryData() can be called to fill the
                       internal parameter set of the class with the contents of the binary 
                       blocks. 


       @param          binaryData [in] pointer to a block of binary data
       @param          length [in] size of the binary data block
       @param          err [out] error list filled, if false is returned
       @return value   true if type was filled, false if err was filled

     ----------------------------------------------------------------*/

    SAPDB_Bool AddBinaryData(SAPDB_Byte *binaryData,SAPDB_UInt4 length,SAPDBErr_MessageList & err);

    /*!----------------------------------------------------------------
       @description    Fill the internal parameter set of the class with the data 
                       that has been stored by subsequent calls to AddBinaryData().
                       The internal storage that has been filled by the AddBinaryData()
                       calls is destroyed by InterpretBinaryData().

       @param          dataRead [out] true if at least one parameter could be read from the stored binary data
       @param          err [out] error list filled, if false is returned
       @return value   true if type was filled, false if err was filled

     ----------------------------------------------------------------*/

    SAPDB_Bool InterpretBinaryData(SAPDB_Bool &dataRead,SAPDBErr_MessageList & err);

    /*!----------------------------------------------------------------
       @description    Get type of given parameter


  Search given parameter with its name and return the type found
       @param          name [in] parameter name
       @param          type [out] parameter type found if any
       @param          err [out] error list filled, if false is returned
       @return value   true if type was filled, false if err was filled

     ----------------------------------------------------------------*/

    SAPDB_Bool GetType (Name name, Type & type,       SAPDBErr_MessageList & err);

    /*!----------------------------------------------------------------
       @description    Get a named and typed value
       @param          name [in] parameter name
       @param          value [out] typed value reference. The space for a 
                       string value has to be allocated by the caller!
                       To receive a reference to a string parameter
                       stored within the class, use the GetReference()
                       method!
       @param          err [out] error list filled, if false is returned
       @return value   true if value was filled, false if err was filled
       ??Prototypes<docTools.DocEntry.PrototypesEntry instance at 8f76b8>

     ----------------------------------------------------------------*/

    SAPDB_Bool GetValue(Name name, Integer & value,   SAPDBErr_MessageList & err);
    SAPDB_Bool GetValue(Name name, Real & value,      SAPDBErr_MessageList & err);
    SAPDB_Bool GetValue(Name name, String  value,     SAPDBErr_MessageList & err);
    SAPDB_Bool GetValue(Name name, CryptInfo & value, SAPDBErr_MessageList & err);

    /*!----------------------------------------------------------------
       @description    Get a named and typed value
       @param          name [in] parameter name
       @param          value [out] typed value reference. The space for a 
                       string value has to be allocated by the caller!
                       To receive a reference to a string parameter
                       stored within the class, use the GetReference()
                       method!
       @param          err [out] error list filled, if false is returned
       @return value   true if value was filled, false if err was filled
       ??Prototypes<docTools.DocEntry.PrototypesEntry instance at 8f76b8>

     ----------------------------------------------------------------*/

    SAPDB_Bool GetTypeAndValueDirect(Name name, Type & type, Integer & intValue, Real & realValue, String  stringValue, CryptInfo & cryptValue,  SAPDBErr_MessageList & err);

    /*!----------------------------------------------------------------
       @description    Get a named and typed value
       @param          name [in] parameter name
       @param          value [out] typed value reference. This method exists for
                       string parameters only as there would be no
                       difference to the GetValue() method for other
                       types of parameters
       @param          err [out] error list filled, if false is returned
       @return value   true if value was filled, false if err was filled

     ----------------------------------------------------------------*/

    SAPDB_Bool GetReference(Name name, String & reference, SAPDBErr_MessageList & err);

    /*!----------------------------------------------------------------
       @description    (Client only ???)Set a named and typed value in memory image


  This call does not produce any I/O. The given values are stored internally only.
  Any string references to 'old value' are invalid after setting a string.
  Setting an value to a different type is an error. It must be deleted first.
  Setting a value with a unknown name, creates that value.
  Any modification on internal memory image sets the 'dirty' flag
       @param          name [in] parameter name
       @param          value [in] typed value reference
       @param          err [out] error list filled, if false is returned
       @return value   true if value was filled, false if err was filled

     ----------------------------------------------------------------*/

    SAPDB_Bool SetValue(Name name, const Integer & value,   SAPDBErr_MessageList & err);
    SAPDB_Bool SetValue(Name name, const Real & value,      SAPDBErr_MessageList & err);
    SAPDB_Bool SetValue(Name name, const String & value,    SAPDBErr_MessageList & err);
    SAPDB_Bool SetValue(Name name, const CryptInfo & value, SAPDBErr_MessageList & err);

    /*!----------------------------------------------------------------
       @description    (Client only ???)Set the update online attribute for a given parameter in memory image.
  The parameter must exist in the memory image, either read from the parameter file or created by the SetValue()
  method.

  This call does not produce any I/O. The given attribute is stored internally only.
       @param          name [in] parameter name
       @param          updateOnlinePossible [in] true if the paramater can be updated in online mode, false if not
       @param          err [out] error list filled, if false is returned
       @return value   true if attribute was set, false if err was filled

     ----------------------------------------------------------------*/

    SAPDB_Bool SetUpdateOnline(Name name, SAPDB_Bool updateOnlinePossible,   SAPDBErr_MessageList & err);

    /*!----------------------------------------------------------------
       @description    Return the update-online-possible attribute for a given parameter

       @param          name [in] parameter name

       @return value   true if the parameter can be updated in online mode, false if not
                       If the parameter is not yet included in the parameter set, true
                       is returned.

     ----------------------------------------------------------------*/
    SAPDB_Bool UpdateOnlinePossible(Name name);

    /*!----------------------------------------------------------------
       @description    (Client only)Request if any parameter modification was done


  If the parameter set was modified after initial read or by 'assignment', but before any
  write action the internal dirty flag is set.
       @return value   true if internal dirty flag is set, false if no I/O is needed.

     ----------------------------------------------------------------*/

    SAPDB_Bool IsDirty()
	{
		return m_DirtyFlag;
	}

    /*!----------------------------------------------------------------
       @description    (Client only)Request to force parameter set write


  With this call the internal dirty flag is set, which is used in Write() call to decide,
  if any I/O is needed.

     ----------------------------------------------------------------*/

    void MarkAsDirty()
	{
		m_DirtyFlag = true;
	};

    /*!----------------------------------------------------------------
       @description    (Client only)Delete a named value in memory image


  This call does not produce any I/O. The given name is searched and if found
  the reference in memory is removed.
  Any string references to 'old value' are invalid after deleting a string.
       @param          name [in] parameter name
       @param          err [out] error list filled, if false is returned
       @return value   true if parameter was deleted, false if not found or other problem

     ----------------------------------------------------------------*/

    SAPDB_Bool Delete(Name name, SAPDBErr_MessageList & err);

    /*!----------------------------------------------------------------
       @description    (Client only)Get the timestamp of last verify action


  If the parameter set was verified if the SAPDB Kernel was able to reach ONLINE state with it.
  This is marked by SAPDB by setting a time stamp inside the parameter file. This timestamp
  is read here from memory image.
       @param          timestamp [out] verification time
       @return value   true if the parameter has not been changed since the time indicated by timeStamp, 
              false if changed have been made in the meantime

     ----------------------------------------------------------------*/

    SAPDB_Bool GetLastVerifyTime(TimeStamp & timeStamp);

    /*!----------------------------------------------------------------
       @description    (SAPDB kernel only)Mark the parameter file as verified


  If the parameter set was verified if the SAPDB Kernel was able to reach ONLINE state with it.
  This function is called by SAPDB and sets the timestamp in the memory image and does the I/O
  to the parameter file.
       @param          err [out] error list filled, if false is returned
       @return value   true if marking timestamp succeeded, false if I/O error occured

     ----------------------------------------------------------------*/

    SAPDB_Bool MarkAsVerified(SAPDBErr_MessageList & err);

    /*!----------------------------------------------------------------
       @description    (SAPDB kernel only)Get the flag that indicates kernel crash


  If SAPDB Kernel crashed before resetting parameter file crash flag, this call returns true.
  This call should be able to return the old state, even after SetCrashFlag() was called...
       @return value   true if previous kernel crashed, false if previous kernel reached OFFLINE state correctly

     ----------------------------------------------------------------*/

    SAPDB_Bool GetOriginalCrashFlag()
	{
		return m_CrashFlag;
	};

    /*!----------------------------------------------------------------
       @description    (SAPDB kernel only)Mark the parameter file crash flag


  If the kernel starts, it sets this flag to make the next kernel able to detect a crash. This is
  possible, since SAPDB kernel uses MarkStoppedCorrectly() to reset the flag. Since the function
  has to survive any crash, it must do I/O directly, without update of 'original' crash flag.
  This call does not influence the internal 'dirty' state!
       @param          err [out] error list filled, if false is returned
       @return value   true if marking crash flag succeeded, false if I/O error occured

     ----------------------------------------------------------------*/

    SAPDB_Bool SetCrashFlag(SAPDBErr_MessageList & err);

    /*!----------------------------------------------------------------
       @description    (SAPDB kernel only)Reset the parameter file crash flag


  Just before the SAPDB kernel reached OFFLINE state, it calls this function to reset the crash flag.
  This call must do I/O directly, without update of 'original' crash flag.
  This call does not influence the internal 'dirty' state!
       @param          err [out] error list filled, if false is returned
       @return value   true if resetting crash flag succeeded, false if I/O error occured

     ----------------------------------------------------------------*/

    SAPDB_Bool MarkStoppedCorrectly(SAPDBErr_MessageList & err);

    /*!----------------------------------------------------------------
       @description    indicate if the parameter file is in old format
       @return value   true if a parameter file in old format has been read.

     ----------------------------------------------------------------*/

    SAPDB_Bool FileInOldFormat()
    {
        return m_FileInOldFormat;
    };
    /*!----------------------------------------------------------------
       @description    Get an iterator objected used to scan parameters


  The iterator is used primarily to scan through a complete parameter set, without knowledge of
  its content. An iterator does not get invalid, if a Delete() was done on the parameterset. But
  mixing up deletes with iterations can be very confusing...
       @param          beginningOfParameterName [in] if != NULL, the iterator contains only those
                       parameters of which the name begins with the string beginningOfParameterName points to.
                       If == NULL, the iterator contains all parameters of the parameter set. This is the
                       default, if this parameter is omitted.
       @return value   An iterator useful to scan parameter set content

     ----------------------------------------------------------------*/

    RTEConf_ParameterIterator GetIterator(SAPDB_UTF8 *beginningOfParameterName = NULL);

private:

    friend class RTEConf_ParameterIterator;
    friend class RTEConf_ParameterRecord;
    friend class CharacterBuffer;

    SAPDB_Int4 CalcIndex(SAPDB_Char *Identifier);


    static SAPDB_Bool m_IntelByteOrder;

    static void ByteOrderSwapInt(SAPDB_Int4 *pTheInt4);
    static void ByteOrderSwapInt(SAPDB_Int8 *pTheInt8);
    static void ByteOrderSwapReal(SAPDB_Real8 *pTheReal8);

    RTEConf_ParameterRecord *LookupRecordByName(Name name);

    void CheckIfStringValueTooLong(const SAPDB_UTF8 *stringValue,SAPDB_Bool &valueTooLong);
    void InsertRecord(Name name,SAPDB_Bool updateOnlinePossible,RTEConf_Parameter::Type type,SAPDB_Byte *pcValue,SAPDB_Bool &nameTooLong,SAPDB_Bool &valueTooLong);

    void ReadOldStyleXParamRecord(Name name,Type & Type,SAPDB_Byte *Value,tRTESys_IOHandle fileHandle=NULL);
    SAPDB_Bool ReadNewStyleXParamRecord(SAPDB_Int4 version,Name name,SAPDB_Bool &updateOnlinePossible,Type & Type,SAPDB_Byte *Value,SAPDB_Bool *pfEndReachedCorrectly,tRTESys_IOHandle fileHandle = NULL);
    void ReadFromBinaryData ( SAPDB_Byte *recordBuffer,SAPDB_Int4 bytesToRead,SAPDB_Int4 &bytesRead,SAPDB_Bool &returnStatus );
    void ReadStringFromBinaryData(String string,SAPDB_UInt4 maxLength,SAPDB_Bool *endMarkerRead);
    void ReadStringFromFile(tRTESys_IOHandle fileHandle,String string,SAPDB_UInt4 maxLength,SAPDB_Bool *endMarkerRead);

    SAPDB_Bool WriteCrashFlag(SAPDB_Byte ucValue,SAPDBErr_MessageList & err);

    SAPDB_Int2 ReadInt2FromByteArray(SAPDB_Byte *pucCharacterArray);
    void       WriteInt2ToByteArray (SAPDB_Byte *pucCharacterArray, SAPDB_Int2 nNum);

    SAPDB_Bool WriteInt2ToFile(SAPDB_Int2 theInt,tRTESys_IOHandle fileHandle,SAPDBErr_MessageList & err);

    void CopyRecords(const RTEConf_Parameter &templateParameter);
    void DeleteRecords();

    SAPDB_Bool BuildFileNameIfNecessary(SAPDBErr_MessageList & err);

    SAPDB_Char *TypeName(RTEConf_Parameter::Type type);

    RTEConf_ParameterRecord *m_ParameterRecords[TOTAL_NUMBER_OF_LISTS_];

    RTE_Path m_FileName;

    TimeStamp m_VerifyTimeStamp;

    SAPDB_Bool m_DirtyFlag;

    SAPDB_Bool m_CrashFlag;

    SAPDB_UInt4 m_maxLengthOfName;
    SAPDB_UInt4 m_maxLengthOfStringValue;

    tsp00_DbNamec	m_dbnamec;
    SAPDB_Int4      m_version;    

    SAPDB_Bool MakeSureParamFileIsInNewFormat(tRTESys_IOHandle &fileHandle,SAPDBErr_MessageList & err );
    SAPDB_Bool m_FileHasBeenRead;
    SAPDB_Bool m_FileInOldFormat;

    struct BinaryDataBlock
    {
        SAPDB_Byte *data;
        SAPDB_UInt4 length;
    };

    Container_List < BinaryDataBlock > m_binaryData;

    SAPDB_Int4 m_positionInBinaryData;
};



class RTEConf_ParameterIterator
{
public:
    /*!----------------------------------------------------------------
       @description    Get the next name and type combination is parameter set


  This call returns the name and type information for the next parameter found. At end of list
  the corresponding flag is set. Ignoring this flag leads to an error for the next call.
       @param          name [out] name of next parameter
       @param          type [out] type of next parameter
       @param          endOfList [out] flag to indicate end of list reached
       @param          err [out] error list filled, if false is returned
       @return value   true if name, type and endOfList are filled, false if err is filled

     ----------------------------------------------------------------*/

    SAPDB_Bool Next(RTEConf_Parameter::Name & name, RTEConf_Parameter::Type & type, SAPDB_Bool &endOfList, SAPDBErr_MessageList & err );

    /*!----------------------------------------------------------------
       @description    ctor


  Construct an iterator object, making a snapshot of the parameter set
       @argument       param [in] the parameter class to make a snapshot of
       @argument       beginningOfParameterName [in] if != NULL, the iterator contains only those
                       parameters of which the name begins with the string beginningOfParameterName points to.
                       If == NULL, the iterator contains all parameters of the parameter set. This is the
                       default, if this parameter is omitted.

     ----------------------------------------------------------------*/

    RTEConf_ParameterIterator(RTEConf_Parameter &param,SAPDB_UTF8 *beginningOfParameterName=NULL);

    /*!----------------------------------------------------------------
       @description    copy ctor


  Construct the object using another object as template.
  The copies use independent storage for parameters, so value change
  on template afterwards have no effect on the object.
       @argument       templateIterator [in] the template parameter iterator class

     ----------------------------------------------------------------*/

    RTEConf_ParameterIterator(const RTEConf_ParameterIterator &templateIterator);

    /*!----------------------------------------------------------------
       @description    assignment operator


  Copy all the content from template parameter iterator object
       @argument       templateIterator [in] the template parameter iterator class

     ----------------------------------------------------------------*/

    RTEConf_ParameterIterator & operator=(const RTEConf_ParameterIterator &templateIterator);

    /*!----------------------------------------------------------------
       @description    dtor


  Frees any memory allocated inside the object

     ----------------------------------------------------------------*/

    ~RTEConf_ParameterIterator();

    /*!----------------------------------------------------------------
       @description    operator new


  uses RTEMem_RteAllocator to allocate size bytes.
  If allocation fails, RTECrash() is called
       @argument       size [in] number of bytes to allocate

     ----------------------------------------------------------------*/

    void *operator new(size_t size);

    /*!----------------------------------------------------------------
       @description    operator delete


  uses RTEMem_RteAllocator to deallocate the memory block at address.
       @argument       address [in] base address of the memory block to deallocate

     ----------------------------------------------------------------*/

    void operator delete(void *address);
private:

    RTEConf_ParameterRecord *m_FirstRecord;
    RTEConf_ParameterRecord *m_LastVisitedRecord;

    void IterateThroughOneList(RTEConf_Parameter &param,SAPDB_Int4 listToIterate,SAPDB_UTF8 *beginningOfParameterName,SAPDB_Int4 lengthToCompare);
    void DeleteIteratorRecords();
    void CopyIteratorRecords(const RTEConf_ParameterIterator &templateIterator);
    

};



/*!--------------------------------------------------------------------
  @class          RTEConf_ParameterRecord
   @description    abstract base class for parameter records


  This class is used to store one single parameter record. It is an abstract base class, derived classes exist for 
  each parameter type (integer, real, string, cryptinfo).
  Only used internally.

 --------------------------------------------------------------------*/

class RTEConf_ParameterRecord
{
public:
    RTEConf_ParameterRecord():m_Name(NULL),m_Next(NULL),m_UpdateOnline(false)
	{
	};

	virtual ~RTEConf_ParameterRecord();

    RTEConf_ParameterRecord *m_Next;

	void *operator new(size_t size);

	void operator delete(void *address);

private:	
    friend class RTEConf_Parameter;
    friend class RTEConf_ParameterIterator;

	RTEConf_Parameter::Name m_Name;
    SAPDB_Bool              m_UpdateOnline;
protected:
	RTEConf_Parameter::Type m_Type;
};



/*!--------------------------------------------------------------------
  @class          RTEConf_ParameterRecordInteger
   @description    derived class for integer parameter


  This class is used to store one single parameter record of type integer. 
  Only used internally.

 --------------------------------------------------------------------*/

class RTEConf_ParameterRecordInteger : public RTEConf_ParameterRecord
{
public:
	RTEConf_ParameterRecordInteger()
	{
		m_intValue = 0;
		m_Type = RTEConf_Parameter::t_Integer;
	};

	~RTEConf_ParameterRecordInteger()
	{
	};

	RTEConf_Parameter::Integer Value(void)
	{
		return m_intValue;
	};

	SAPDB_Bool IsDifferent(RTEConf_Parameter::Integer intValue)
	{
		return (m_intValue != intValue);
	};

    void Assign(void *intValue);

private:
	RTEConf_Parameter::Integer m_intValue;
};



/*!--------------------------------------------------------------------
  @class          RTEConf_ParameterRecordReal
   @description    derived class for real parameter


  This class is used to store one single parameter record of type real.
  Only used internally.

 --------------------------------------------------------------------*/

class RTEConf_ParameterRecordReal : public RTEConf_ParameterRecord
{
public:
	RTEConf_ParameterRecordReal()
	{
		m_realValue = 0.0;
		m_Type = RTEConf_Parameter::t_Real;
	};

	~RTEConf_ParameterRecordReal()
	{
	};

	RTEConf_Parameter::Real Value(void)
	{
		return m_realValue;
	};

	SAPDB_Bool IsDifferent(RTEConf_Parameter::Real realValue)
	{
		return (m_realValue != realValue);
	};

    void Assign(void *realValue);

private:
	RTEConf_Parameter::Real m_realValue;
};



/*!--------------------------------------------------------------------
  @class          RTEConf_ParameterRecordString
   @description    derived class for string parameter


  This class is used to store one single parameter record of type string. 
  Only used internally.

 --------------------------------------------------------------------*/

class RTEConf_ParameterRecordString : public RTEConf_ParameterRecord
{
public:
	RTEConf_ParameterRecordString()
	{
		m_stringValue = NULL;
		m_Type = RTEConf_Parameter::t_String;
	};

	~RTEConf_ParameterRecordString();

    RTEConf_Parameter::String Value(void)
	{
		return m_stringValue;
	};

	SAPDB_Bool IsDifferent(RTEConf_Parameter::String stringValue)
	{
        return (STRCMP_UTF8(m_stringValue, stringValue) != 0);
	};

    void Assign(RTEConf_Parameter::String stringValue);

private:
	RTEConf_Parameter::String m_stringValue;
};



/*!--------------------------------------------------------------------
  @class          RTEConf_ParameterRecordCryptInfo
   @description    derived class for cryptInfo parameter


  This class is used to store one single parameter record of type CryptInfo. 
  Only used internally.

 --------------------------------------------------------------------*/

class RTEConf_ParameterRecordCryptInfo : public RTEConf_ParameterRecord
{
public:
	RTEConf_ParameterRecordCryptInfo()
        : m_Value(0)
	{
		m_Type = RTEConf_Parameter::t_CryptInfo;
	};

	~RTEConf_ParameterRecordCryptInfo();

	RTEConf_Parameter::CryptInfo *Value(void)
	{
		return m_Value;
	};

	SAPDB_Bool IsDifferent(const RTEConf_Parameter::CryptInfo *cryptInfoValue)
	{
		return (memcmp(m_Value, cryptInfoValue, sizeof(RTEConf_Parameter::CryptInfo)) != 0);
	};

    void Assign(RTEConf_Parameter::CryptInfo *cryptInfoValue);

private:
	RTEConf_Parameter::CryptInfo *m_Value;
};




#endif // CONF_PARAMETER_ACCESS_HPP
