/*!
    @file     SharedSQL_ICommand.hpp
    @ingroup  SharedSQL
    @author   DirkT
    @brief    SQLCommands cached by the SharedSQL_CommandCache
    @see            

\if EMIT_LICENCE
    ========== licence begin  GPL
    Copyright (c) 2000-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end
\endif
*/

#ifndef SHAREDSQL_ICOMMAND_HPP
#define SHAREDSQL_ICOMMAND_HPP

/*===========================================================================*
 *  INCLUDES                                                                 *
 *===========================================================================*/

#include "SQLManager/SharedSQL/SharedSQL_CommandStatistics.hpp"
#include "SQLManager/SharedSQL/SharedSQL_Types.hpp"

/*===========================================================================*
 *  CLASSES, STRUCTURES, TYPES, UNIONS ...                                   *
 *===========================================================================*/

class SharedSQL_ICommandCache;
class SharedSQL_IPlan;
class SAPDBMem_IRawAllocator;

/*---------------------------------------------------------------------------*/
/*!
    @class          SharedSQL_CommandInfo
    @brief          Information about the stored SharedSQL_Command, used the SystemView COMMANDSTATISTICS
 */
class SharedSQL_CommandInfo
{
public:
    /// Unique ID to identify the command
	SharedSQL_CommandID				mCommandID;
    /// Flag to indicate wheather or not the command is an internal one
    SAPDB_Bool                      mInternal;
    /// Hash value of the command
	SharedSQL_HashValue				mHashValue;
    /// The hash table slot the command is stored in
    SharedSQL_HashValue             mSlot;
    //
    /// The context of the command
    SharedSQL_SQLContext            mSQLContext;
    //
    /// The Status (New, Dropped, Preparing, Prepared, Invalid)
    SharedSQL_CommandStatus	        mStatus;
    /// Flag to indicate that the sql statement is unloaded to the temporary file
	SAPDB_Byte			    	    mFlgExtern;
    /// ...
    SAPDB_Int4                      mPrepareCount; 	
    /// ...
    SAPDB_Int4                      mParseIDCount;
    /// ...
    SAPDB_Int4                      mExecuteCount;
    //
    /// Size of the sql statement
    SAPDB_Int4                      mStmtSize;
    /// Pointer to the sql statement (NULL if the statement is unloaded to the temporary file)
    void*	                        mStmt;
    /// Size of the execution plan
    SAPDB_Int4                      mPlanSize;
    //
    /// Statistical information
    SharedSQL_CommandStatistics     mStatistics;
};  // SharedSQL_CommandInfo

/*---------------------------------------------------------------------------*/
/*!
    @interface      SharedSQL_ICachedCommand
    @brief          Interface to SharedSQL_CachedCommand
 */
class SharedSQL_ICachedCommand
{

public:
    /*!
        @brief  Returns an interface to the command cache the command is stored in
        @return SharedSQL_ICommandCache&
    */
    virtual SharedSQL_ICommandCache& CommandCache( void ) = 0;

    /*!
        @brief  Returns the sql statement
        @return SharedSQL_SQLStmt&
    */
    virtual SharedSQL_SQLStmt& SQLStmt( void ) = 0;

    /*!
        @brief  Returns the sql context
        @return SharedSQL_SQLContext&
    */
    virtual SharedSQL_SQLContext& SQLContext( void ) = 0;

    /*!
        @brief  Returns an interface to the execution plan
        @return SharedSQL_IPlan&
    */
    virtual SharedSQL_IPlan& Plan( void ) = 0;


    /*!
        @brief  Returns the statistical information
        @return SharedSQL_CommandStatistics&
    */
    virtual SharedSQL_CommandStatistics& Statistics( void ) = 0;


    /*!
        @brief  Returns the status of the command
        @return SharedSQL_CommandStatus
    */
    virtual SharedSQL_CommandStatus GetStatus( void ) = 0;

    /*!
        @brief  Sets the status of the command (only to be used the SharedSQL_CommandCache)
        @param  Status [in]
    */
    virtual void SetStatus( SharedSQL_CommandStatus Status ) = 0;


    /*!
        @brief  Returns FlgExtern that indicated wheather or not the sql statement is unloaded to the temporary file
        @return SAPDB_Byte
    */
    virtual SAPDB_Byte GetFlgExtern( void ) = 0;

    /*!
        @brief  Sets FlgExtern of the command (only to be used the SharedSQL_CommandCache)
        @param  FlgExtern [in]
    */
    virtual void SetFlgExtern( SAPDB_Byte FlgExtern ) = 0;

    /*!
        @brief  Returns FlgLRUListOld that indicated wheather or not the command belongs to the OLD part of the LRU list
        @return SAPDB_Bool
    */
    virtual SAPDB_Bool GetFlgLRUListOld( void ) = 0;

    virtual void SetFlgLRUListOld( SAPDB_Bool Flg ) = 0;

    /*!
        @brief  Returns the unique ID that identifys the command
        @return SharedSQL_CommandID
    */
    virtual SharedSQL_CommandID GetCommandID( void ) = 0;

    /*!
        @brief  Returns the hash value 
        @return SharedSQL_HashValue
    */
	virtual SharedSQL_HashValue	GetHashValue( void ) = 0;

    /*!
        @brief  Returns wheather or not the command is an internal one
        @return SAPDB_Bool
    */
    virtual SAPDB_Bool IsInternal( void ) = 0;

    /*!
        @brief  Sets FlgInternal of the command (only to be used the SharedSQL_CommandCache)
        @param  Internal [in]
    */
    virtual void SetInternal(SAPDB_Bool Internal) = 0;


    /*!
        @brief  Returns the sql statement
        @return SharedSQL_SQLStmt
    */
    virtual const SharedSQL_SQLStmt GetSQLStmt( void ) = 0;


    /*!
        @brief  Returns the CommandInfo
        @param  I [out]
    */
    virtual void GetCommandInfo( SharedSQL_CommandInfo& I ) = 0;

    /*!
        @brief  Returns the reet version of the CommandInfo
        @param  I [out]
    */
    virtual void GetCommandInfoReset( SharedSQL_CommandInfo& I ) = 0;

    /*!
        @brief  Resets the statistical data
    */
    virtual void ResetStatisticsInfo( void ) = 0;


    /*!
        @brief  Returns the next command within the same slot of the hash table
        @return SharedSQL_ICachedCommand*
    */
    virtual SharedSQL_ICachedCommand* NextHashCommand( void ) const = 0;


    /*!
        @brief  Returns the prepare counter
        @return SAPDB_Int4
    */
    virtual SAPDB_Int4 GetPrepareCount( void ) = 0;

    /*!
        @brief  Increment prepare counter (only to be used by SharedSQL_PrepareHandle)
    */
    virtual void IncPrepareCount( void ) = 0;

    /*!
        @brief  Decrement prepare counter (only to be used by SharedSQL_PrepareHandle)
    */
    virtual void DecPrepareCount( void ) = 0;


    /*!
        @brief  Returns the parseid counter
        @return SAPDB_Int4
    */
    virtual SAPDB_Int4 GetParseIDCount( void ) = 0;

    /*!
        @brief  Increment parseid counter (only to be used by SharedSQL_ParseIDHandle)
    */
    virtual void IncParseIDCount( void ) = 0;

    /*!
        @brief  Decrement parseid counter (only to be used by SharedSQL_ParseIDHandle)
    */
    virtual void DecParseIDCount( void ) = 0;


    /*!
        @brief  Returns the execution
        @return SAPDB_Int4
    */
    virtual SAPDB_Int4 GetExecuteCount( void ) = 0;

    /*!
        @brief  Increment execution counter (only to be used by SharedSQL_ExecuteHandle)
    */
    virtual void IncExecuteCount( void ) = 0;

    /*!
        @brief  Decrement execution count (only to be used by SharedSQL_ExecuteHandle)
    */
    virtual void DecExecuteCount( void ) = 0;


    /*!
        @brief  Lock the command
        @param  exclusive [in]
    */
    virtual void EnterLock( bool exclusive ) = 0;

    /*!
        @brief  Try to lock the command
        @param  exclusive [in]
        @return true, if lock could be entered, else false
    */
    virtual bool TryEnterLock( bool exclusive ) = 0;

    /*!
        @brief  Leave the lock
        @param  exclusive [in]
    */
    virtual void LeaveLock( bool exclusive ) = 0;


    /// ...
    virtual void IncCacheAccessCountExecute( void ) = 0;
    /// ...
    virtual void IncCacheSuccessfulAccessCountExecute( void ) = 0;
    /// ...
    virtual void IncCacheUnSuccessfulAccessCountExecute( void ) = 0;

};  // SharedSQL_ICachedCommand


/* ------------------------------------------------------------------------- */

#endif  /* SHAREDSQL_ICOMMAND_HPP */

