""" A wizard for creating a new project resource. """


# Standard library imports.
from os.path import join

# Enthought library imports.
from enthought.naming.api import Context
from enthought.traits.api import Instance

# Plugin imports.
from enthought.envisage.project.template import Template

# Local imports.
from new_project_page import NewProjectPage
from new_resource_wizard import NewResourceWizard
from resource_factory import ResourceFactory


class ProjectFactory(ResourceFactory):
    """ A factory that creates projects from templates. """

    #### 'ProjectFactory' interface ###########################################

    # The project template.
    template = Instance(Template)

    ###########################################################################
    # 'ResourceFactory' interface.
    ###########################################################################

    def create_resource(self):
        """ Creates the resource. """

        # The root of the workspace.
        from enthought.envisage import get_application
        workspace = get_application().service_registry.get_service(
                  'enthought.envisage.project.IWorkspace'
            )
        
        # Create the project (this creates the project directory).
        project = workspace.create_project(self.name)

        # Copy the selected project template.  Using 'force=True' makes sure
        # that the copy will succeed even though the project directory already
        # exists.
        self.template.target = join(workspace.path, self.name)
        self.template.generate(force=True)

        # Refresh the project to reflect its new contents (put there by the
        # template).
        project.refresh()
        
        return
        

class NewProjectWizard(NewResourceWizard):
    """ A wizard for creating a new project resource. """

    #### 'Window' interface ###################################################

    # The window title.
    title = 'New Project'

    #### 'NewResourceWizard' interface ########################################

    # The resource factory that the wizard is configuring.
    factory = Instance(ProjectFactory, ())
    
    #### 'NewProjectWizard' interface #########################################

    # The root of the project template hierarchy to be displayed in the tree.
    root = Instance(Context)

    ###########################################################################
    # 'NewResourceWizard' interface.
    ###########################################################################

    def reset(self):
        """ Reset the wizard to the initial state. """
        
        # The pages in the wizard.
        self.pages = [
            NewProjectPage(
                obj  = self.factory,
                root = self.root
            )
        ]

        return

#### EOF ######################################################################
