#-------------------------------------------------------------------------------
#
#  A factory class for creating RepositoryRoot objects for use with the
#  Repository class based on a file referenced by an environment variable.
#
#  Written by: David C. Morrill
#
#  Date: 03/30/2006
#
#  (c) Copyright 2006 by Enthought, Inc.
#
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
#  Imports:
#-------------------------------------------------------------------------------

import logging

from os \
    import environ

from os.path \
    import isdir

from enthought.traits.api \
    import Str

from repository_root \
    import RepositoryRoot

from repository_root_factory \
    import RepositoryRootFactory


# Setup a logger for this module.
logger=logging.getLogger(__name__)


#-------------------------------------------------------------------------------
#  'EnvironmentVariableRepositoryRootFactory' class:
#-------------------------------------------------------------------------------

class EnvironmentVariableRepositoryRootFactory ( RepositoryRootFactory ):

    #---------------------------------------------------------------------------
    #  Trait definitions:
    #---------------------------------------------------------------------------

    # The name of the environment variable to use:
    variable = Str( 'ENVISAGE_REPOSITORY' )

    # The relative priority of RepositoryRoots created by this factory:
    priority = 1000000

    #---------------------------------------------------------------------------
    #  'RepositoryRootFactory' interface:
    #---------------------------------------------------------------------------

    def roots ( self ):
        """ Returns the list of RepositoryRoot objects this factory creates.
        """
        file  = None
        roots = []
        try:
            file_name = environ[ self.variable ]
            file      = open( file_name, 'rb' )
            i         = j = 0
            for line in file:
                j   += 1
                line = line.strip()
                if (line != '') and (line[:1] != '#'):
                    col = line.find( '=' )
                    if col >= 0:
                        name = line[ : col ].strip()
                        path = line[ col + 1: ].strip()
                        if name != '':
                            if isdir( path ):
                                roots.append(
                                    RepositoryRoot(
                                        name      = name,
                                        path      = path,
                                        permanent = True,
                                        priority  = i - self.priority ) )
                                i += 1
                            else:
                                logger.error( "Repository: '%s' is not a "
                                    "directory (Line %d in %s for %s)" % (
                                    path, j, file_name, self.variable ) )
                        else:
                            logger.error( "Repository: No repository name "
                                    "specified (Line %d in %s for %s)" % (
                                    j, file_name, self.variable ) )
                    else:
                        logger.error( "Repository: Invalid syntax: '%s' (Line "
                                      "%d in %s for %s" % (
                                      line, j, file_name, self.variable ) )
        except:
            # Since we expect an exception to be raised if the environment
            # variable isn't set, we just warn about it here.  Note that if
            # the environmant variable isn't set at all, then the file_name
            # isn't set and the first log message raises its own exception.
            try:
                logger.warn( "Repository: Error reading '%s' for %s",
                             file_name, self.variable )
            except:
                logger.warn( "Repository: Environment variable '%s' not set",
                             self.variable )

        if file is not None:
            file.close()

        return roots

