""" The resource type for files. """


# Standard library imports.
from os.path import join, split

# Enthought library imports.
from enthought.io import File
from enthought.traits.api import Str

# Local imports.
from file_node_type import FileNodeType
from resource_type import ResourceType


class FileResourceType(ResourceType):
    """ The resource type for files. """

    #### 'FileResourceType' interface #########################################

    # The file extension recognized by this type.
    ext = Str

    ###########################################################################
    # 'ResourceType' interface.
    ###########################################################################

    #### Initializers #########################################################

    # fixme: Quick hack to associate the new-style text editor with this
    # resource type.
    def _editor_default(self):
        """ Trait initializer. """

        from enthought.plugins.text_editor.editor.text_editor import TextEditor

        return TextEditor
    
    def _node_type_default(self):
        """ Trait initializer. """

        return FileNodeType(resource_type=self)

    #### Methods ##############################################################
    
    def is_type_for(self, obj):
        """ Returns True if the resource type recognizes an object. """

        if len(self.ext) == 0:
            result = isinstance(obj, File)

        else:
            result = isinstance(obj, File) and obj.ext.lower() == self.ext

        return result

    def clone(self, obj):
        """ Returns a clone of an object. """

        return File(obj.path)

    def get_name(self, obj):
        """ Returns the name of an object. """

        basename, filename = split(obj.path)

        return filename
    
    def set_name(self, obj, name):
        """ Sets the name of an object. """

        basename, filename = split(obj.path)

        obj.path = join(basename, name)

        return

    def copy(self, context, name, obj):
        """ Copies a resource into the specified context. """

        # Create an appropriate file object for the context that we are
        # copying to.
        f = File(join(context.path, name))

        # Copy the actual file...
        obj.copy(f)

        # ... and bind the name.
        context.bind(name, f)

        return

    def get_id(self, obj):
        """ Returns a unique identifier for a resource.

        Unique here means, 'unique within the current process'.

        """

        return obj.path

    def get_reference(self, obj):
        """ Returns a pickleable reference to a resource.

        By default this returns the object itself. This is only acceptable
        for SMALL resources!

        """

        if len(obj.path) == 0:
            reference = None

        else:
            reference = super(FileResourceType, self).get_reference(obj)

        return reference
        
##### EOF #####################################################################
