""" The node type for folder resources. """


# Enthought library imports.
from enthought.naming.api import Binding, Context
from enthought.naming.ui.api import ContextMonitor
from enthought.pyface.tree.api import NodeType

# Local imports.
from resource_node_type import ResourceNodeType


class FolderNodeType(ResourceNodeType):
    """ The node type for folder resources. """

    #### 'NodeType' interface #################################################

    # The image used to represent nodes that allow children and are NOT
    # expanded.
    closed_image = NodeType.CLOSED_FOLDER

    # The image used to represent nodes that allow children and ARE expanded.
    open_image = NodeType.OPEN_FOLDER

    ###########################################################################
    # 'NodeType' interface.
    ###########################################################################

    #### Content ####
    
    def allows_children(self, node):
        """ Does the node allow children (ie. a folder vs a file). """

        return True

    def has_children(self, node):
        """ Returns True if a node has children, otherwise False. """

        return len(node.obj.list_names('')) > 0

    def get_children(self, node):
        """ Returns the children of a node. """

        return node.obj.list_bindings('')

    def get_monitor(self, node):
        """ Returns a monitor that detects changes to a node. """

        return ContextMonitor(node=node)

    #### Copy/Cut/Delete/Paste ####

    def can_copy(self, node):
        """ Can a node be copied? """

        return True

    def can_cut(self, node):
        """ Can a node be cut? """

        return False

    def can_delete(self, node):
        """ Can a node be deleted? """

        return True
    
    def can_paste(self, node, data):
        """ Returns True if a node allows an object to be dropped onto it. """

        # fixme: Folder resource types probably belong in the project package!
        from enthought.envisage.project import Project
        if isinstance(data, Binding) and isinstance(data.obj, Project):
            can_paste = False

        else:
            can_paste = True
            
        return can_paste

    #### Drag and drop ####
    
    def can_drop(self, node, data):
        """ Returns True if a node allows an object to be dropped onto it. """

        return False

    def drop(self, node, data):
        """ Drops an object onto a node. """

        node.obj.bind(data.name, data.obj)

        return
    
#### EOF ######################################################################
