#------------------------------------------------------------------------------
# Copyright (c) 2005, Enthought, Inc.
# All rights reserved.
# 
# This software is provided without warranty under the terms of the BSD
# license included in enthought/LICENSE.txt and may be redistributed only
# under the conditions described in the aforementioned license.  The license
# is also available online at http://www.enthought.com/licenses/BSD.txt
# Thanks for using Enthought open source!
# 
# Author: Enthought, Inc.
# Description: <Enthought naming package component>
#------------------------------------------------------------------------------
""" A Python object explorer tree. """


# Enthought library imports.
from enthought.envisage.resource import ArrayResourceType
from enthought.envisage.resource import BoolResourceType
from enthought.envisage.resource import ClassResourceType
from enthought.envisage.resource import DictResourceType
from enthought.envisage.resource import FloatResourceType
from enthought.envisage.resource import InstanceResourceType
from enthought.envisage.resource import IntResourceType
from enthought.envisage.resource import ListResourceType
from enthought.envisage.resource import MethodResourceType
from enthought.envisage.resource import NoneResourceType
from enthought.envisage.resource import ResourceManager
from enthought.envisage.resource import StringResourceType
from enthought.envisage.resource import TupleResourceType
from enthought.naming.api import Binding, Context
from enthought.traits.api import Any, Str

# Local imports.
from resource_tree import ResourceTree


class ExplorerTree(ResourceTree):
    """ The main application window. """

    #### 'Tree' interface #####################################################

    # Should the root of the tree be shown?
    show_root = False
    
    #### 'ExplorerTree' interface #############################################

    # The object that we are exploring.
    obj = Any

    # The name to display in the tree.
    name = Str('Object')
    
    ###########################################################################
    # 'object' interface.
    ###########################################################################

    def __init__(self, parent, obj, **traits):
        """ Creates a new explorer tree. """

        # The object that we are exploring.
        self.obj  = obj

        # Create a resource manager populated with all of the Python resource
        # types.
        self.resource_manager = self._create_resource_manager()

        # Create the root node of the tree.
        root = self._create_root(
            self.resource_manager.type_manager, self.name, self.obj
        )
        
        # Base class constructor.
        super(ExplorerTree, self).__init__(parent, root=root, **traits)

        return
    
    ###########################################################################
    # Private interface.
    ###########################################################################

    def _create_resource_manager(self):
        """ Creates a resource manager. """

        resource_manager = ResourceManager(
            resource_types = [
                NoneResourceType(),
                BoolResourceType(),
                IntResourceType(),
                FloatResourceType(),
                StringResourceType(),
                ArrayResourceType(),
                DictResourceType(),
                ListResourceType(),
                TupleResourceType(),
                MethodResourceType(),
                ClassResourceType(),
                InstanceResourceType()
            ]
        )
        
        return resource_manager

    def _create_root(self, type_manager, name, obj):
        """ Creates the root node of the tree. """

        # Create the root context.
        context = Context()
        context.environment[Context.TYPE_MANAGER] = type_manager
        context.bind(name, obj)

        return Binding(name='Root', obj=context)

##### EOF #####################################################################
