// lpcfilter.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include "application.h"
#include "lpcdata.h"
#include "lpcfilter.h"

void (*LPC_Filter::interpFunction)(float*,float*,float*,float,int) = LPC_Filter::linearInterp;

// we have to cast to LPCData* below because we do not do the cast-and-assign
// until the call to setLPC()

LPC_Filter::LPC_Filter(Data* lpd, int nsamps, int normalize, Data* warp)
		: lpcdata(nil), amplitudes(nil), warpData(warp),
		  doNormalize(normalize), doWarping(warp != nil),
		  totalSamps(nsamps), warpFact(0.0), ampAdjust(1.0), filtAdjust(1.0),
		  filterFunction(&LPC_Filter::standardFilter), oldSig(0.0),
		  pastValues(nil), coeffArray(nil), currentArray(nil), nextArray(nil) {
	if(!lpd->isA(LPC_Data))
		Application::alert("Selected source for filter is not an LPC data file.");
	else if(lpd->channels() < ((LPCData *)lpd)->nPoles() + 4)
		Application::alert("Selected source for filter must include",
			"all channels of the LPC data.  Please reselect.");
	else {
		setWarp(warp);
		setLPC(lpd);
	}
}

LPC_Filter::~LPC_Filter() {
	delete [] pastValues;
	delete [] nextArray;
	delete [] currentArray;
	delete [] coeffArray;
	Resource::unref(lpcdata);
	Resource::unref(amplitudes);
	Resource::unref(warpData);
}

void
LPC_Filter::setLPC(Data* lpd) {
	if (lpd != lpcdata) {
		Resource::unref(lpcdata);
		lpcdata = (LPCData *) lpd;
		if (lpcdata != nil) {
			lpcdata->ref();
			nPoles = lpcdata->nPoles();		// cache these for speed
			sampleRate = lpcdata->sRate();
			lastFrame = lpcdata->length() - 1;
		}
	}
}

void
LPC_Filter::setWarp(Data* warp) {
	if (warp != warpData) {
		Resource::unref(warpData);
		warpData = warp;
		if (warpData != nil) {
			warpData->ref();
			doWarping = true;
			filterFunction = &LPC_Filter::warpFilter;
		}
		else {
			doWarping = false;
			filterFunction = &LPC_Filter::standardFilter;
		}
	}
}

void
LPC_Filter::initialize() {
	amplitudes = new Data(FloatData, lpcdata->length() + 1);
	amplitudes->ref();
	lpcdata->getEnvelope(amplitudes, 0, AbsoluteMagnitude);
	if(doNormalize)
		amplitudes->normalize();
	pastValues = new float[nPoles * 2];
	coeffArray = new float[nPoles + 4];
	currentArray = new float[nPoles + 4];
	nextArray = new float[nPoles + 4];
	reset();
}

void
LPC_Filter::reset() {
	counter = frameSamps = 0;
	oldFrame = -1;
	sampsToGo = totalSamps;
	framesPerSamp = double(lpcdata->length() - 1) / totalSamps;
	oldSig = warpFact = 0;
	ampAdjust = filtAdjust = 1; 
	int end = nPoles * 2;
	for(int i=0; i < end; i++)
		pastValues[i] = 0;
}

float *
LPC_Filter::getFrame(float* voicedAmp, float* frameError, float* framePitch) {
	static const int lpcOffset = 4;			// first 4 locs are not coeffs
	float frameNumber = (totalSamps - sampsToGo) * framesPerSamp;
	float* array = coeffArray;
	if(needNewFrame()) {
		int currentFrame = int(frameNumber);
 		if(currentFrame != oldFrame) {
			cacheFrames(currentFrame);
 			oldFrame = currentFrame;
 		}
		// get interpolated coeff values from cached coeffs
		array = interpolate(frameNumber - currentFrame);
		if(doWarping) ampAdjust = warpFrame(array+lpcOffset, frameNumber);
		*frameError = *(array+2);
		float pitch = *(array+3);
		*framePitch = pitch > 0.0 ? pitch : 256.0;	// can't be zero!
		frameSamps = int(min(sampleRate / *framePitch, float(sampsToGo)));
	}
	else {							// these must be computed every time
		*frameError = *(array+2);
		float pitch = *(array+3);
		*framePitch = pitch > 0.0 ? pitch : 256.0;	// can't be zero!
	}
	*voicedAmp = amplitudes->get(frameNumber) * ampAdjust;	// may be normalized
	advance();
	return array + lpcOffset;
}

// frame arrays are cached to avoid doing interp frame lookup every samp

void
LPC_Filter::cacheFrames(int currentFrame) {
	int current = min(lastFrame, currentFrame);
	int next = min(lastFrame, currentFrame + 1);
	int framesize = nPoles + 4;
	lpcdata->getFrame(currentArray, framesize, current);
	lpcdata->getFrame(nextArray, framesize, next);
}

void
LPC_Filter::setInterpMode(InterpMode mode) {
	interpFunction = (mode == Linear) ?
		LPC_Filter::linearInterp : LPC_Filter::computedInterp;
}

float *
LPC_Filter::interpolate(double fraction) {
	int framesize = nPoles + 4;
	(*interpFunction)(currentArray, nextArray, coeffArray, fraction, framesize);
	return coeffArray;
}

// this was warpset() in original Cmix lpcplay code

float
LPC_Filter::warpFrame(float *frame, float frameNumber) {
	warpFact = warpData->get(frameNumber);
	for(int i = 1; i < nPoles; i++)
		*(frame+i) += *(frame+i-1) * warpFact;
	// amplitude adjustment is returned to the caller
	float ampfact = 1.0/(1.0 - warpFact * *(frame+nPoles-1));
	// this is a member variable and is stored till use in warpFilter
	filtAdjust = ampfact * (1.0 - warpFact * warpFact);
	return ampfact;
}

void
LPC_Filter::advance() {
	sampsToGo--;
	frameSamps--;
}

double
LPC_Filter::filter(float* coeffs, double sig) {
	return (this->*filterFunction)(coeffs, sig);
}

double
LPC_Filter::standardFilter(float* coeffs, double sig) {
	register float* past = pastValues;
	for(int j = counter, nfint=0;  nfint < nPoles;  nfint++, j++)
		sig += (*(coeffs+nfint) * *(past+j));
	*(past + counter) = *(past + counter + nPoles) = sig;
	counter = (counter + 1) % nPoles;
	return sig;	
}

double
LPC_Filter::warpFilter(float* coeffs, double sig) {
	register float* past = pastValues;
	float temp1 = *(past+nPoles-1);		// start with last past value
	*(past+nPoles-1) = (filtAdjust * oldSig) - warpFact * temp1;
	register int n;
	for(n = nPoles - 2; n >= 0; n--) {
		float temp2 = *(past+n);
		*(past+n) = warpFact * (*(past+n+1) - *(past+n)) + temp1;
		temp1 = temp2;
	}
	for(n = 0; n < nPoles; n++)
		sig += *(coeffs+n) * *(past+n);
	oldSig = sig;
	return sig;
}

void
LPC_Filter::linearInterp(float* current, float* next, float* coeffs,
		float fraction, int framesize) {
	for(int n = 0; n < framesize; n++)
		*(coeffs+n) = *(current+n) + fraction * (*(next+n) - *(current+n));
}

void
LPC_Filter::computedInterp(float *bold, float *bnew, float *bout,
		float fraction, int framesize) {
    /* System generated locals */
	long npols = framesize - 4;
	static const int pMax = 64;
	static const int pMaxP = 65;
    /* Local variables */
    static double a[pMax*pMax]    /* was [24][24] */;
    static double aold[pMax*pMax] /* was [24][24] */,
	              anew[pMax*pMax]   /* was [24][24] */,
				  rold[pMaxP],
				  rnew[pMaxP];
    register int i, j;
    static double r[pMaxP];
    static float sum1, sum2;

    /* Parameter adjustments */
    --bold;
    --bnew;
    --bout;

	// Special adjustments to skip the first 4 elements in each array
	bold += 4;
	bnew += 4;
	bout += 4;
	
    /* Function Body */
    int npp1 = npols + 1;
    for (i = 1; i <= npols; ++i) {
        int ncom2 = npp1 - i;
        aold[npols + i * pMax - pMaxP] = bold[ncom2];
        anew[npols + i * pMax - pMaxP] = bnew[ncom2];
    }
    int npm1 = npols - 1;
    for (int k = 1; k <= npm1; ++k) {
        i = npols - k;
        for (j = 1; j <= i; ++j) {
/* Computing 2nd power */
            double d_1 = aold[i + 1 + (i+1) * pMax - pMaxP];
            aold[i + j * pMax - pMaxP] = (aold[i + 1 + j * pMax - pMaxP]
				+ aold[i + 1 + (i+1) * pMax - pMaxP]
				* aold[i + 1 + (i + 1 - j) * pMax - pMaxP]) / (1. - d_1 * d_1);
/* Computing 2nd power */
            d_1 = anew[i + 1 + (i+1) * pMax - pMaxP];
            anew[i + j * pMax - pMaxP] = (anew[i + 1 + j * pMax - pMaxP]
				+ anew[i + 1 + (i+1) * pMax - pMaxP]
				* anew[i + 1 + (i + 1 - j) * pMax - pMaxP]) / (1. - d_1 * d_1);
        }
    }
/* ......COMPUTE AUTOCORRELATION COEFFICIENTS RTEMP(I) WITH R(0) = 1 */
    rold[0] = 1.;
    rnew[0] = 1.;
    for (i = 1; i <= npols; ++i) {
        sum1 = 0.;
        sum2 = 0.;
        for (j = 1; j <= i; ++j) {
            sum1 += aold[i + j * pMax - pMaxP] * rold[i + 1 - j - 1];
            sum2 += anew[i + j * pMax - pMaxP] * rnew[i + 1 - j - 1];
        }
        rold[i] = sum1;
        rnew[i] = sum2;
    }
/* ......COMPUTE R(0) FROM RTEMP(I) */
    for (i = 1; i <= npols; ++i) {
        rold[0] -= aold[npols + i * pMax - pMaxP] * rold[i];
        rnew[0] -= anew[npols + i * pMax - pMaxP] * rnew[i];
    }
    rold[0] = 1. / rold[0];
    rnew[0] = 1. / rnew[0];
/* ......COMPUTE AUTOCORRELATION COEFFICIENTS R(I) WITH CORRECT R(0) */
    for (i = 2; i <= npp1; ++i) {
        rold[i - 1] = rold[0] * rold[i - 1];
        rnew[i - 1] = rnew[0] * rnew[i - 1];
    }
/* ......INTERPOLATE AUTOCORRELATION COEFFICIENTS */
    for (i = 1; i <= npp1; ++i) {
        r[i - 1] = fraction * (rnew[i - 1] - rold[i - 1]) + rold[i - 1];
    }
/* ......INSERT TEST FOR SINGULARITY OF NEW AUTOCORRELATION MATRIX */
/* ......COMPUTE NEW PREDICTOR COEFFICIENTS */
    a[0] = r[1] / r[0];
    for (i = 2; i <= npols; ++i) {
        sum1 = r[0];
        sum2 = r[i];
        int iminus = i - 1;
        for (j = 1; j <= iminus; ++j) {
            sum1 -= r[j] * a[i - 1 + j * pMax - pMaxP];
            sum2 -= r[i - j] * a[i - 1 + j * pMax - pMaxP];
        }
        a[i + i * pMax - pMaxP] = sum2 / sum1;
        for (j = 1; j <= iminus; ++j) {
            a[i + j * pMax - pMaxP] = a[i - 1 + j * pMax - pMaxP]
				- a[i + i * pMax - pMaxP] * a[i - 1 + (i - j) * pMax - pMaxP];
        }
    }
	// do linear interp of first 4 array elements (index adjusted back)
	for(int n = -3; n < 1; n++)
		*(bout + n) = *(bold + n) + fraction * (*(bnew + n) - *(bold + n));
    for (i = 1; i <= npols; ++i) {
        int ncom2 = npp1 - i;
        *(bout + i) = a[npols + ncom2 * pMax - pMaxP];
    }
}
