// optionsetter.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/

#ifdef __GNUG__
#pragma implementation
#endif

#include <String.h>
#undef String
#include "mxv_types.h"
#include "application.h"
#include "dialogbox.h"
#include "fileselector.h"
#include "formatrequester.h"
#include "query_templates.h"
#include "request.h"
#include "requester.h"
#include "lpcdata.h"
#include "smartmem.h"
#include "sound.h"
#include "soundheader.h"
#include "optionsetter.h"

class GlobalOptionRequester : public TitledRequester {
	friend GlobalOptionSetter;
protected:
	GlobalOptionRequester(GlobalOptionSetter *);
	redefined void configureRequest(Request *);
private:
	GlobalOptionSetter* client;
};

GlobalOptionRequester::GlobalOptionRequester(GlobalOptionSetter* gos)
	: TitledRequester("Set Global Program Options:"), client(gos) {}

void
GlobalOptionRequester::configureRequest(Request* request) {
	request->appendValue(
#ifndef __GNUG__
		createQuery(
			"Alert beep volume:",
			&DialogBox::setBeepLevel,
			&DialogBox::beepLevel,
			Range(0, 100),
			true
		)
#else
		new Query<int>(
			"Alert beep volume:",
			new FunctionSetterGetter<boolean, int>(
				&DialogBox::setBeepLevel,
				&DialogBox::beepLevel
			),
			Range(0, 100),
			true
		)
#endif
	);
	request->appendChoice("Dialog Panels Ignore Window Manager:",
	                      "|No|Yes|", &client->panelOverride, true);
	request->appendChoice("Auto-Place Windows on Screen:",
	                     "|No|Yes|", &client->autoPlace, true);
}

//********

GlobalOptionSetter::GlobalOptionSetter()
	  : panelOverride(DialogBox::overrideWindowManager() ?
		Requester::True : Requester::False),
	    autoPlace(
		Application::globalResourceIsTrue("AutoPlaceWindows") ?
			Requester::True : Requester::False) {}

Requester *
GlobalOptionSetter::createRequester() {
	return new GlobalOptionRequester(this);
}

int
GlobalOptionSetter::apply() {
	DialogBox::overrideWindowManager(panelOverride == Requester::True);
	Application::setGlobalResource(
		"AutoPlaceWindows",
		(autoPlace == Requester::True) ? "true" : "false"
	);
	return true;
}

//********

class MemoryOptionRequester : public TitledRequester {
	friend MemoryOptionSetter;
protected:
	MemoryOptionRequester(MemoryOptionSetter *);
	redefined void configureRequest(Request *);
	redefined boolean confirmValues();
private:
	MemoryOptionSetter* client;
};

MemoryOptionRequester::MemoryOptionRequester(MemoryOptionSetter* mos)
	: TitledRequester("Set Data Memory Options:"), client(mos) {}

void
MemoryOptionRequester::configureRequest(Request* request) {
	request->appendLabel("(Use to adjust memory limits for data only.)");
	request->appendLabel("----------------");
	static char label[120];
	sprintf(label, "Total memory currently in use for data:  %.2f mB",
		SmartMemory::totalBytesAllocated()/1.0e+06);
	request->appendLabel(label);
	client->maxSingle = int(SmartMemory::maxSingleAllocation()/1.0e+06);
	client->maxTotal = int(SmartMemory::maxTotalAllocation()/1.0e+06);
	request->appendValue("Maximum Single Memory Allocation (mB.):",
	                    &client->maxSingle,  Range(1.0, 1000.0), true);
	request->appendValue("Maximum Total Memory Allocation (mB.):",
	                    &client->maxTotal, Range(10.0, 5000.0), true);
}

boolean
MemoryOptionRequester::confirmValues() {
	if(client->maxTotal <= client->maxSingle) {
		Application::alert("Total Allocation limit must be greater than",
			"the Single Allocation limit.  Please recheck settings.");
		return false;
	}
	return true;
}

//********

int
MemoryOptionSetter::apply() {
	SmartMemory::setMaxSingleAllocation(unsigned(maxSingle * 1.0e+06));
	SmartMemory::setMaxTotalAllocation(unsigned(maxTotal * 1.0e+06));
	return true;
}

Requester *
MemoryOptionSetter::createRequester() {
	return new MemoryOptionRequester(this);
}

//********

class FileOptionRequester : public TitledRequester {
	friend FileOptionSetter;
protected:
	FileOptionRequester(FileOptionSetter* fos)
		: TitledRequester("Set File Options:"), client(fos) {}
	redefined void configureRequest(Request *);
private:
	FileOptionSetter* client;
};

void
FileOptionRequester::configureRequest(Request* request) {
	request->appendChoice("Read Raw (Headerless) Files:",
	                     "|No|Yes|", &client->readRaw, true);
	request->appendChoice("Store/Recall Browser Path:",
	                     "|No|Yes|", &client->storePath, true);
	request->appendChoice("Browser Shows Invisible Files:",
	                     "|No|Yes|", &client->showInvisible, true);
}

//********

FileOptionSetter::FileOptionSetter()
	: readRaw(Application::globalResourceIsTrue("ReadRawFiles") ?
		Requester::True : Requester::False),
	  storePath(
		Application::globalResourceIsTrue("BrowserUseLastPath") ?
			Requester::True : Requester::False),
	  showInvisible(FileSelector::showDotFiles() ?
		Requester::True : Requester::False) {}

Requester *
FileOptionSetter::createRequester() {
	return new FileOptionRequester(this);
}

int
FileOptionSetter::apply() {
	Application::setGlobalResource(
		"ReadRawFiles", (readRaw == Requester::True) ? "true" : "false"
	);
	Application::setGlobalResource(
		"BrowserUseLastPath",
		(storePath == Requester::True) ? "true" : "false"
	);
	FileSelector::showDotFiles(showInvisible == Requester::True);
	return true;
}

//********

class SoundOptionRequester : public FormatRequester {
	friend SoundOptionSetter;
protected:
	SoundOptionRequester(SoundOptionSetter* s)
		: FormatRequester(nil, "Default Sound Sample Format:",
		                  s->sampleFormat),
		  TitledRequester("Set Sound Options:"),
		  client(s) {}
	redefined void configureRequest(Request *);
private:
	SoundOptionSetter* client;
};

void
SoundOptionRequester::configureRequest(Request* request) {
	request->appendValue(
#ifndef __GNUG__
		createQuery(
			"Default Sampling Rate:",
			&Sound::setDefaultSampleRate,
			&Sound::defaultSampleRate,
			NonNegativeNumbers,
			false
		)
#else
		new Query<int>(
			"Default Sampling Rate:",
			new FunctionSetterGetter<boolean, int>(
				&Sound::setDefaultSampleRate,
				&Sound::defaultSampleRate
			),
			NonNegativeNumbers,
			false
		)
#endif
	);
	FormatRequester::configureRequest(request);
	request->appendChoice("Default Sound Header Format:",
		"|Raw (No Header)|Snd/au|Hybrid|BSD/IRCAM|AIF-C|WAVE|",
		&client->headerFormat
	);
}

//********

SoundOptionSetter::SoundOptionSetter()
	: sampleFormat(Sound::defaultDataType()),
	  headerFormat(SoundHeader::defaultHeaderType()) {}

Requester *
SoundOptionSetter::createRequester() {
	return new SoundOptionRequester(this);
}

int
SoundOptionSetter::apply() {
	Sound::setDefaultDataType(DataType(sampleFormat));
	SoundHeader::setDefaultHeaderType(SoundHeader::Type(headerFormat));
	return true;
}

//********

class LPCOptionRequester : public TitledRequester {
	friend LPCOptionSetter;
protected:
	LPCOptionRequester(LPCOptionSetter* l)
		: TitledRequester("Set LPC Data Options:"), client(l) {}
	redefined void configureRequest(Request *);
private:
	LPCOptionSetter* client;
};

void
LPCOptionRequester::configureRequest(Request* request) {
	request->appendLabel("(Default samp rate = default sound samp rate.)");
	request->appendValue(
#ifndef __GNUG__
		createQuery(
			"Default Frame Rate:",
			&LPCData::setDefaultFrameRate,
			&LPCData::defaultFrameRate,
			NonNegativeNumbers,
			false
		)
#else
		new Query<double>(
			(const char *) "Default Frame Rate:",
			(SetterGetter<double> *)
			new FunctionSetterGetter<boolean, double>(
				&LPCData::setDefaultFrameRate,
				&LPCData::defaultFrameRate
			),
			NonNegativeNumbers,
			false
		)
#endif
	);
	request->appendValue(
#ifndef __GNUG__
		createQuery(
			"Default Number of Filter Poles:",
			&LPCData::setDefaultNumberOfPoles,
			&LPCData::defaultNumberOfPoles,
			Range(12, 64),
			true
		)
#else
		new Query<int>(
			(const char *) "Default Number of Filter Poles:",
			(SetterGetter<int> *)
			new FunctionSetterGetter<boolean, int>(
				&LPCData::setDefaultNumberOfPoles,
				&LPCData::defaultNumberOfPoles
			),
			Range(12, 64),
			true
		)
#endif
	);
	request->appendChoice("Default LPC Header Format: ",
		"|Raw (No Header)|Csound-format Header|",
		&client->headerFormat
	);
}

//********

LPCOptionSetter::LPCOptionSetter()
	: headerFormat(LPCHeader::defaultHeaderType()) {}

Requester *
LPCOptionSetter::createRequester() {
	return new LPCOptionRequester(this);
}

int
LPCOptionSetter::apply() {
	LPCHeader::setDefaultHeaderType(LPCHeader::Type(headerFormat));
	return true;
}

//********

enum OptionType { StringType, IntType, DoubleType, BooleanType };

struct OptionEntry {
	const char* optionName;
	OptionType optionType;
};

static OptionEntry optionEntries[] = {
	// Global Options
	{ "AlertBeepVolume", IntType },
	{ "DialogIgnoreWindowManager", BooleanType },
	{ "AutoPlaceWindows", BooleanType },
	// Memory Options
	{ "MaximumSingleAllocation", DoubleType },
	{ "MaximumTotalAllocation", DoubleType },
	// File Options
	{ "ReadRawFiles", BooleanType },
	{ "BrowserUseLastPath", BooleanType },
	{ "BrowserShowDotFiles", BooleanType },
	{ "DefaultSoundFileDir", StringType },
	{ "DefaultLPCFileDir", StringType },
	{ "DefaultPitchFileDir", StringType },
	{ "DefaultFFTFileDir", StringType },
	{ "DefaultEnvelopeFileDir", StringType },
	{ "DefaultPvocFileDir", StringType },
	// Data Options
	{ "DefaultSoundSampleRate", IntType },
	{ "DefaultSoundSampleFormat", StringType },
	{ "DefaultSoundHeaderType", StringType },
	{ "DefaultLPCFrameRate", DoubleType },
	{ "DefaultLPCNumberOfPoles", IntType },
	{ "DefaultLPCHeaderType", StringType },
	{ nil }
};

#ifndef __GNUG__
extern "C" char* getenv(const char*);
#endif

void read_mxvrc() {
	char mxvrc_path[1024];
	const char* homeDir = getenv("HOME");
	if(!homeDir) homeDir = ".";
	sprintf(mxvrc_path, "%s/.mxvrc", homeDir);
	FILE* file = fopen(mxvrc_path, "r");
	if(file == nil) return;
	char oname[80];
	char ovalue[1024];
	while(fscanf(file, "%s %s", oname, ovalue) != EOF) {
		String optionName(oname);
		for(OptionEntry* entry = optionEntries;
				entry->optionName != nil; entry++) {
			if(fcompare(optionName, String(entry->optionName)) == 0) {
				Application::setGlobalResource(entry->optionName, ovalue);
			}
		}
	}
	fclose(file);
}

void write_mxvrc() {
	char mxvrc_path[1024];
	const char* homeDir = getenv("HOME");
	if(!homeDir) homeDir = ".";
	sprintf(mxvrc_path, "%s/.mxvrc", homeDir);
	FILE* file = fopen(mxvrc_path, "w+");
	if(file != nil) {
		for(OptionEntry* entry = optionEntries;
				entry->optionName != nil; entry++) {
			const char* resource =
				Application::getGlobalResource(entry->optionName);
			if(resource && strlen(resource))
				fprintf(file, "%s\t\t%s\n", entry->optionName, resource);
		}
		fclose(file);
	}
	else {
		Application::error("Unable to open .mxvrc file for writing.",
			mxvrc_path);
	}
}

