// query.h

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


// This file declares a set of classes used to store and transport parameters
// that are set by the user in the dialog process.  These QueryItems and their
// subclasses carry string, int, double, or state flags, from the user
// interface objects to the Requester sublass that has "queried" for them.
// QueryItem instances are stored in Request objects, which are created by the
// Requester subclasses.  They are passed to the dialog objects via the 
// DialogConstructor class (see dialog_ctor.h).

#ifndef QUERY_H
#ifdef __GNUG__
#pragma interface
#endif
#define QUERY_H

#include <ctype.h>
#include <String.h>
#include <InterViews/resource.h>
#include "range.h"

// CharCheck class contains the char test functions for checking user input and
// for scanning default strings for value type (int, double, string)

class CharCheck {
public:
	// control chars and ints only
	static boolean posIntsOnly(const char c) { 
		return (iscntrl(c) || isdigit(c));
	}
	static boolean intsOnly(const char c) {
		return (c == '-' || posIntsOnly(c));
	}
	// control chars, '.',  and ints only
	static boolean posNumsOnly(const char c) {
		return (c == '.' || posIntsOnly(c));
	}
	static boolean numsOnly(const char c) { return (c == '.' || intsOnly(c)); }
	// for entering strings
	static boolean anyChar(const char) { return true; }
	static boolean isIntegerString(const char*);
	static boolean isDoubleString(const char*);
private:
	virtual void dontAllowInstantiation()=0;
};

class String;

// base class for all query items, and also used for plain labels

class QueryItem : public Resource {
public:
	enum QueryType { Label = 0, Browser, Choice,
		         Value, RangedValue, Button };
public:
	QueryItem(const char *);
	QueryItem(const char *, const char *);
	virtual ~QueryItem();
	virtual const char* label();
	virtual QueryType type() { return Label; }
protected:
	static const char* separator;	// token for separating labels
private:
	String *myLabel;
};

// class for file browser

class QueryValue;

class QueryFile : public QueryItem {
	typedef QueryItem Super;
public:
	QueryFile(String* path, const char* suffixes);
	virtual ~QueryFile();
	redefined QueryType type() { return Browser; }
	QueryValue* path() { return myPath; }
	const char* suffixes() { return *mySuffixes; }
private:
	QueryValue *myPath;
	String *mySuffixes;
};

// class for response buttons (confirm, cancel, etc.)

class QueryButton : public QueryItem {
	typedef QueryItem Super;
public:
	QueryButton(const char *, Response resp=Yes);
	virtual ~QueryButton();
	redefined QueryType type() { return Button; }
	Response response() { return myResponse; }
	boolean set(Response r) { myResponse = r; return true; }
private:
	Response myResponse;
};

class String;

// class for querying sets of boolean options

class QueryChoice : public QueryItem {
	typedef QueryItem Super;
public:
	QueryChoice(const char *, const char *list, ChoiceValue*, boolean ex);
	virtual ~QueryChoice();
	redefined QueryType type() { return Choice; }
	boolean areExclusive() { return exclusive; }
	ChoiceValue state() { return *statePtr; }
	void setState(ChoiceValue state) { *statePtr = state; }
	int numberOfValues() { return myList.freq(separator) - 1; }
	void start();
	const char *nextLabel();
	boolean nextValue(int *);
private:
	ChoiceValue* statePtr;	// hex representation of state
	String myList;		// encoded list of button labels
	String iterator;	// for extracting individual labels
	String currentLabel;
	ChoiceValue stateIterator;
	ChoiceValue currentValue;
	boolean exclusive;	// only one button selection at a time?
};


struct QueryLabelInfo {
	char *label;
};

struct QueryButtonInfo {
	char *label;		// label for button
	Response response;	// behavior for this button (yes, no, cancel)
};

struct QueryInfo {
	QueryLabelInfo *labelInfo;	// labels
	QueryButtonInfo *buttonInfo;	// response buttons
	Response defaultResponse;
};

#endif
